"""Build python projects from arcadia and upload them to pypi."""
import logging
import os

import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.juggler import virtualenv
from sandbox.projects.walle.memoize import MemoizeStage
from sandbox.projects.walle.pypi import upload_to_pypi
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process


class ProjectsParameter(parameters.SandboxStringParameter):
    name = 'projects'
    description = 'Project paths, relative to arcadia root, \';\'-separated'
    required = True
    multiline = True


class TestParameter(parameters.SandboxBoolParameter):
    name = 'test'
    description = 'Build and test with arcadia tools before releasing.'
    default_value = True


class ReleaseToPypi(ArcadiaTask):
    """Release package from arcadia to pypi."""

    type = 'RELEASE_PACKAGE_TO_PYPI'
    input_parameters = [ArcadiaUrl, ProjectsParameter, TestParameter]

    VAULT_OWNER = 'WALLE'
    VAULT_USERNAME_KEY = 'pypi-username'
    VAULT_PASSWORD_KEY = 'pypi-password'

    BUILD_SYSTEM = consts.YMAKE_BUILD_SYSTEM

    def do_execute(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        projects = self._get_projects()

        if self.should_test():
            # build all projects first. Fail task if any projects fails to build.
            @MemoizeStage.commit_on_success(self, "build_projects")
            def build_projects():
                arcadia_sdk.do_build(self.BUILD_SYSTEM, arcadia_src_dir, projects, test=True, clear_build=False)

        # release projects on by one. allow retries on exceptions.
        for project in projects:
            stage_name = "release_project_{}".format(project.replace("/", "_"))

            @MemoizeStage.commit_on_success(self, stage_name)
            def release_project():
                self._release_project(os.path.join(arcadia_src_dir, project))

    def should_test(self):
        return self.ctx[TestParameter.name]

    def _get_projects(self):
        return list(self._parse_packages_list(self.ctx[ProjectsParameter.name]))

    @staticmethod
    def _parse_packages_list(packages_str):
        for packages_line in packages_str.splitlines(False):
            for package in packages_line.split(";"):
                yield package.strip()

    def _release_project(self, source_dir):
        build_requirements = [os.path.join(source_dir, "requirements-dev.txt")]

        with virtualenv.venv_context(self, self.path("build"), requirements_list=build_requirements) as venv:
            self._create_dist(source_dir, venv)

            pypi_username = self.get_vault_data(self.VAULT_OWNER, self.VAULT_USERNAME_KEY)
            pypi_password = self.get_vault_data(self.VAULT_OWNER, self.VAULT_PASSWORD_KEY)

            artifacts = os.listdir(os.path.join(source_dir, "dist"))
            self._upload_to_pypi(pypi_username, pypi_password, source_dir, artifacts)

    @staticmethod
    def _create_dist(source_dir, venv):
        """Build distribution with standard python tools."""

        run_process(
            [venv.executable, "setup.py", "sdist", "--formats=gztar"],
            log_prefix="setup.py",
            work_dir=source_dir
        )

    @staticmethod
    def _upload_to_pypi(username, password, source_dir, artifacts):
        """Upload distributions to Yandex PyPI."""

        for dist in artifacts:
            logging.info('Uploading dist/{}...'.format(dist))
            upload_to_pypi(os.path.join(source_dir, "dist", dist), username, password)


__Task__ = ReleaseToPypi
