# coding: utf-8

import os
import tarfile

import sandbox.common.types.client as ctc
from sandbox.common.errors import SandboxException
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxSelectParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.walle import resource_types

MODE_REPORT_HOST_ERRORS = "report-host-and-switch-errors"
MODE_REPORT_SWITCH_ERRORS = "report-switch-only-errors"
MODE_REPORT_DISABLED_AUTOMATION = "report-disabled-automation"
MODE_SYNC_TOPOLOGY = "sync-topology"
MODE_CHECK_EXTRA_VLANS = "check-search-extra-vlans"


class WalleScriptsResource(LastReleasedResource):
    name = 'walle_scripts_resource'
    description = 'wall-e.scripts resource'
    resource_type = resource_types.WALLE_SCRIPTS
    required = False


class RunModeSelector(SandboxSelectParameter):
    name = 'run_mode'
    description = 'mode to run'
    choices = [
        ("report errors on hosts", MODE_REPORT_HOST_ERRORS),
        ("report errors on switches", MODE_REPORT_SWITCH_ERRORS),
        ("report disabled automation", MODE_REPORT_DISABLED_AUTOMATION),
        ("sync topology", MODE_SYNC_TOPOLOGY),
        ("check search extra VLANs", MODE_CHECK_EXTRA_VLANS)
    ]
    default_value = MODE_SYNC_TOPOLOGY
    required = True


class RunWalleScripts(SandboxTask):
    """
    Run specified wall-e script.
    """

    type = 'RUN_WALLE_SCRIPTS'

    execution_space = 100
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [
        WalleScriptsResource,
        RunModeSelector,
    ]

    def on_execute(self):
        walle_scripts_resource = self.ctx.get(WalleScriptsResource.name)
        if not walle_scripts_resource or walle_scripts_resource == '0':
            walle_scripts_resource = channel.sandbox.list_releases(
                resource_type=resource_types.WALLE_SCRIPTS, limit=1
            )[0].resources[0].id
            self.ctx[WalleScriptsResource.name] = walle_scripts_resource

        walle_config = self.get_vault_data('WALLE', 'wall-e.scripts.config')
        with open("config.yaml", "w") as fd:
            fd.write(walle_config)

        resource_path = self.sync_resource(walle_scripts_resource)
        venv_path = self.path('walle_scripts')
        tarfile.open(resource_path).extractall(venv_path)

        python_path = os.path.join(venv_path, 'bin', 'python')

        run_mode = self.ctx.get(RunModeSelector.name)
        if run_mode == MODE_REPORT_HOST_ERRORS:
            script_argv = [
                os.path.join(venv_path, 'bin', 'report-error-hosts'),
                '--with-top-problematic-hosts', '--with-rack-and-switch-problems'
            ]
        elif run_mode == MODE_REPORT_SWITCH_ERRORS:
            script_argv = [
                os.path.join(venv_path, 'bin', 'report-error-hosts'),
                '--with-rack-and-switch-problems'
            ]
        elif run_mode == MODE_REPORT_DISABLED_AUTOMATION:
            script_argv = [
                os.path.join(venv_path, 'bin', 'report-disabled-automation')
            ]
        elif run_mode == MODE_SYNC_TOPOLOGY:
            script_argv = [
                os.path.join(venv_path, 'bin', 'sync-topology'),
                '--cron'
            ]
        elif run_mode == MODE_CHECK_EXTRA_VLANS:
            script_argv = [
                os.path.join(venv_path, 'bin', 'check-extra-vlans.py'),
                '--vlan', 'all',
                '--project', 'all',
                '--host', 'all',
            ]
        else:
            raise SandboxException('unknown run mode given')

        environment = os.environ.copy()
        environment.pop("PYTHONPATH", None)
        environment["LANG"] = "en_US.UTF-8"
        run_process([python_path] + script_argv, log_prefix="script", environment=environment)


__Task__ = RunWalleScripts
