"""Calculate Wall-E statistics."""

import os
from datetime import datetime

from sandbox.projects.walle import resource_types
from sandbox.sandboxsdk import ssh
from sandbox.sdk2.vcs.git import Git

from sandbox.sandboxsdk.parameters import (
    SandboxStringParameter, SandboxBoolParameter, ListRepeater, SandboxIntegerParameter, Container
)
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


def check_date(date_str):
    return datetime.strptime(date_str, '%Y-%m-%d').strftime('%Y-%m-%d')


class VaultKeyParameter(SandboxStringParameter):
    name = 'vault_key'
    description = 'Sandbox Vault key'
    default_value = ''


class StartDateParameter(SandboxStringParameter):
    name = 'start_date'
    description = 'Start date (YYYY-MM-DD or empty for the beginning of the previous period)'
    default_value = ''
    required = False


class NPeriodsParameter(SandboxIntegerParameter):
    name = 'periods'
    description = 'Number of periods to calculate stats for.'
    required = False


class ReportsToRunParameter(ListRepeater, SandboxStringParameter):
    name = 'reports_to_run'
    description = 'Reports to run'
    choices = [
        ('Adjacents', 'adjacents'),
        ('Downtime', 'downtime'),
        ('HTTP Statuses', 'http_statuses'),
        ('Server Errors', 'server_errors')
    ]
    default_value = dict(choices).values()
    required = True


class ReportsScaleParameter(SandboxStringParameter):
    name = 'scale'
    description = 'Report scale for stat reports'
    choices = [
        ('Daily', 'daily'),
        ('Weekly', 'weekly'),
        ('Monthly', 'monthly')
    ]
    default_value = 'daily'
    required = True


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'master'
    required = False


class IsPeriodicTaskParameter(SandboxBoolParameter):
    name = "_is_periodic"
    description = "Is this task run from the scheduler?"
    default_value = False
    do_not_copy = True


class LxcContainerParameter(Container):
    required = True
    description = "LXC container for task"
    alias = "linux_ubuntu_12.04_precise"
    resource_type = "LXC_CONTAINER"
    default_value = "900620903"


class CalculateWalleStatistics(SandboxTask):
    """Calculate Wall-E statistics."""
    type = 'WALLE_STATS'

    GIT_URL = 'ssh://git@bb.yandex-team.ru/nanny/wall-e.stats.git'
    VAULT_OWNER = 'WALLE'
    SSH_KEY_VAULT_NAME = 'ssh-key'

    input_parameters = [GitTagParameter, VaultKeyParameter, IsPeriodicTaskParameter,
                        StartDateParameter, NPeriodsParameter,
                        ReportsScaleParameter, ReportsToRunParameter, LxcContainerParameter]

    def arcadia_info(self):
        return None, 'wall-e stats {tag}'.format(tag=self.ctx.get(GitTagParameter.name)), None

    def on_execute(self):
        scale = self.ctx[ReportsScaleParameter.name]
        if self.ctx[IsPeriodicTaskParameter.name] and scale == 'monthly' and datetime.today().day > 9:
            print("Won't rebuild automatic monthly reports after the first week of the month.")
            return

        sources_folder = self._checkout_sources(self.ctx.get(GitTagParameter.name, 'master'))
        vault_key = self.ctx[VaultKeyParameter.name]
        yt_token = self.get_vault_data(vault_key, 'yt-token')
        yt_username = self.get_vault_data(vault_key, 'yt-username')
        stat_username = self.get_vault_data(vault_key, 'stat-username')
        stat_password = self.get_vault_data(vault_key, 'stat-password')

        self._make_statistics(sources_folder, scale, self.ctx[ReportsToRunParameter.name],
                              self.ctx.get(NPeriodsParameter.name), self.ctx.get(StartDateParameter.name),
                              yt_token, yt_username, stat_username, stat_password)

    def _checkout_sources(self, tag):
        src_dir = self.path('wall-e.stats.src')
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            Git(self.GIT_URL).clone(src_dir, branch=tag)

        return src_dir

    def _make_statistics(self, src_dir, scale, reports_to_run, periods, start_date, yt_token, yt_username,
                         stat_username, stat_password):
        bin_path = os.path.join(src_dir, 'bin/walle_stats')
        cfg_path = os.path.join(src_dir, 'walle_stats/walle_stats.conf.yaml')

        for report_name in reports_to_run:
            run_process(
                [bin_path, scale, report_name] +
                (['--start-date', check_date(start_date)] if start_date else []) +
                (['--periods', str(periods)] if periods else []) +
                [
                    '--build-wheel',
                    '-c', cfg_path,
                    '-V', 'yt_token=' + yt_token,
                    '-V', 'yt_username=' + yt_username,
                    '-V', 'stat_username=' + stat_username,
                    '-V', 'stat_password=' + stat_password,
                ],
                log_prefix='walle_stats',
                work_dir=src_dir,
                environment={'PYTHONPATH': src_dir},
            )

        if 'downtime' in reports_to_run:
            self.create_resource(description='Wall-E statistics: {}/{} {}s'.format(start_date, periods, scale[:-2]),
                                 resource_path=os.path.join(src_dir, 'reports'),
                                 resource_type=resource_types.WALLE_STATS,
                                 arch='any')


__Task__ = CalculateWalleStatistics
