# -*- coding: utf-8 -*-

import logging
import os
import re
import time

import requests

from sandbox import sdk2

import sandbox.common.errors
import sandbox.common.types.task as ctt
import sandbox.common.utils

from sandbox.projects.sandbox_ci.utils import github
from sandbox.projects.sandbox_ci import managers, parameters

from .. import hermione_common_task


class WEATHER_FRONTEND_HERMIONE_CI_REPORT(hermione_common_task.WEATHER_FRONTEND_HERMIONE_REPORT):
    ttl = 90


class WeatherFrontendHermioneCI(hermione_common_task.Task):
    """
    Task to run frontend hermione tests inside sandbox
    Based on sandbox/projects/Ufo/Hermione/__init__.py
    """

    report_resource_class = WEATHER_FRONTEND_HERMIONE_CI_REPORT

    @sandbox.common.utils.singleton_property
    def github_statuses(self):
        return managers.GitHubStatusesManager(self)

    @property
    def github_context(self):
        return 'Hermione'

    @property
    def project_conf(self):
        return {
            'github': {
                'report_statuses': self.Parameters.report_github_statuses
            }
        }

    class Parameters(hermione_common_task.Parameters):
        # these parameters are for status reports
        with sdk2.parameters.Group('Github statuses parameters.') as github_repo_block:
            project_github_owner = parameters.project_github_owner()
            project_github_repo = parameters.project_github_repo()
            project_github_commit = parameters.project_github_commit()
            report_github_statuses = sdk2.parameters.Bool('Should report to github', default_value=True)

        pull_request_num = sdk2.parameters.Integer('Pull request number', required=True)

        install_commands = sdk2.parameters.String(
            'Commands to clone git repo and install dependencies',
            multiline=True
        )

    class Context(hermione_common_task.Context):
        ticket_ids = []

    def on_enqueue(self):
        super(WeatherFrontendHermioneCI, self).on_enqueue()

        logging.getLogger().setLevel(logging.INFO)

        tasks = sdk2.Task.find(
            task_type=self.type,
            status=(
                ctt.Status.Group.QUEUE,
                ctt.Status.Group.EXECUTE,
                ctt.Status.Group.WAIT,
            ),
            input_parameters=dict(
                pull_request_num=self.Parameters.pull_request_num
            )
        ).limit(100)

        logging.info('Found %ld tasks with same pull request', tasks.count)

        for task in tasks:
            logging.info('Stopping task {}'.format(task.id))
            task.stop()

    def on_execute(self):
        super(WeatherFrontendHermioneCI, self).on_execute()

        if not os.environ.get('SANDBOX_AUTH_TOKEN'):
            os.environ['SANDBOX_AUTH_TOKEN'] = self.vault.read('SANDBOX_AUTH_TOKEN')

        with self.memoize_stage.install:
            with sdk2.helpers.ProgressMeter('Checkout and install npm packages'):
                self.exec_multiple_commands(self.Parameters.install_commands.split('\n'), 'install')

        with sdk2.helpers.ProgressMeter('Running tests'):
            self.run_tests()

        if not self.Context.tests_status_success:
            raise sandbox.common.errors.TaskFailure('Tests failed')

    def report_self_status_to_github(self, state=None, description='', url=None):
        if not os.environ.get('GITHUB_API_TOKEN'):
            os.environ['GITHUB_API_TOKEN'] = self.vault.read('WEATHER_GITHUB_TOKEN')

        self.github_statuses.report_status_to_current_sha(
            context=self.github_context,
            state=state if state else github.convert_sandbox_status_to_github_state(self.status),
            url=url if url else sandbox.common.utils.get_task_link(self.id),
            description=description
        )

    def on_prepare(self):
        super(WeatherFrontendHermioneCI, self).on_prepare()
        self.report_self_status_to_github()

    def on_success(self, prev_status):
        super(WeatherFrontendHermioneCI, self).on_success(prev_status)
        self.report_self_status_to_github(
            url=self.Context.report_url,
            state=github.GitHubStatus.SUCCESS
        )

    def on_failure(self, prev_status):
        super(WeatherFrontendHermioneCI, self).on_failure(prev_status)
        self.report_self_status_to_github(
            url=self.Context.report_url,
            state=github.GitHubStatus.FAILURE
        )

    def on_break(self, prev_status, status):
        super(WeatherFrontendHermioneCI, self).on_break(prev_status, status)
        self.report_self_status_to_github(
            description='The task was stopped' if status == 'STOPPED' else ''
        )
