# -*- coding: utf-8 -*-

import datetime
import re

import pytz

from sandbox import sdk2

from sandbox.common import telegram
import sandbox.common.errors
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.utils

from sandbox.projects.sandbox_ci import managers, parameters
from sandbox.common.types.client import Tag
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT


__all__ = (
    'WEATHER_FRONTEND_HERMIONE_REPORT',
    'Requirements',
    'Parameters',
    'Context',
    'Task',

    'TelegramNotifyParameters',
    'TelegramNotifyTaskMixin',
)


class WEATHER_FRONTEND_HERMIONE_LXC_IMAGE(sdk2.service_resources.LxcContainer):
    __default_attribute__ = sdk2.parameters.String

    releasable = True
    any_arch = False
    executable = False
    auto_backup = True
    releasers = ['WEATHER']
    ttl = 365

class WEATHER_FRONTEND_HERMIONE_REPORT(sdk2.Resource):
    __default_attribute__ = sdk2.parameters.String

    auto_backup = True
    any_arch = True

    tool = sdk2.parameters.String
    project = sdk2.parameters.String

class WeatherFrontendHermioneContainer(sdk2.parameters.Container):
    @sandbox.common.utils.classproperty
    def default_value(cls):
        return sdk2.task.Task.server.resource.read(
            type=WEATHER_FRONTEND_HERMIONE_LXC_IMAGE,
            state=ctr.State.READY,
            limit=1,
        )['items'][0]['id']

class Requirements(sdk2.Requirements):
    disk_space = 5 * 1024
    dns = ctm.DnsType.DNS64
    ram = 2 * 1024
    client_tags = Tag.SSD
    #ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 15 * 1024, None)

class Parameters(parameters.CommonParameters):
    _container = WeatherFrontendHermioneContainer(
        'Environment container resource',
        resource_type=WEATHER_FRONTEND_HERMIONE_LXC_IMAGE,
        required=True
    )
    install_commands = sdk2.parameters.String(
        'Commands to clone git repo and install dependencies',
        multiline=True
    )
    test_commands = sdk2.parameters.String(
        'Commands to run tests',
        multiline=True
    )
    cop_name = sdk2.parameters.String(
        'Testcop project name',
        default_value='weatherfront',
        required=True
    )
    html_report_path = sdk2.parameters.String(
        'Html report path',
        default_value='hermione/reports',
        required=True
    )
    json_report_path = sdk2.parameters.String(
        'JSON report path',
        default_value='hermione/reports-json/report.json',
        required=True
    )

class Context(sdk2.Context):
    tests_status_success = None
    report_url = ''

class Task(sdk2.Task):
    """
    Task to run frontend hermione tests inside sandbox
    Based on sandbox/projects/Ufo/Hermione/__init__.py
    """

    vault_owner = 'WEATHER'
    report_resource_class = WEATHER_FRONTEND_HERMIONE_REPORT

    @sandbox.common.utils.singleton_property
    def vault(self):
        return managers.VaultManager(self)

    class Requirements(Requirements):
        pass

    class Parameters(Parameters):
        pass

    class Context(Context):
        pass

    def run_tests(self):
        set_json_reporter_env = 'export json_reporter_path="' + self.Parameters.json_report_path + '"'
        self.exec_command(command=set_json_reporter_env, logname='Set json-reporter path env')

        tool = 'hermione'
        project = self.Parameters.cop_name

        status = self.exec_multiple_commands(
            self.Parameters.test_commands.split('\n'),
            'test',
            break_on_failure=False,
            set_info=True
        )

        html_report_resource = SANDBOX_CI_ARTIFACT(
            self,
            'hermione html report',
            type='hermione-report',
            project=project,
            tool=tool,
            path=self.Parameters.html_report_path,
            multiple=True,
            register_dependency=False
        )
        sdk2.ResourceData(html_report_resource).ready()

        json_report_resource = SANDBOX_CI_ARTIFACT(
            self,
            'hermione json report',
            type='json-reporter',
            project=project,
            tool=tool,
            path=self.Parameters.json_report_path,
            multiple=True,
            register_dependency=False
        )
        sdk2.ResourceData(json_report_resource).ready()

        self.Context.tests_status_success = not bool(status)
        self.Context.report_url = '{}/index.html'.format(html_report_resource.http_proxy)
        self.Context.profile_url = '{}/profile/index.html'.format(html_report_resource.http_proxy)
        self.Context.testcop_url = 'https://testcop.si.yandex-team.ru/task/{taskId}?project={project}'.format(
            taskId=self.id,
            project=project
        )
        self.set_info(
            "Report <a href='{url}'>{url}</a><br />TestCop <a href='{testcop}'>{testcop}</a><br />Profile <a href='{purl}'>{purl}</a>".format(
                url=self.Context.report_url,
                purl=self.Context.profile_url,
                testcop=self.Context.testcop_url
            ),
            do_escape=False
        )

    def exec_multiple_commands(self, commands, logname, break_on_failure=True, set_info=False):
        status_summary = 0
        for command in commands:
            command = command.strip()
            if not command:
                return

            status = self.exec_command(command=command, logname=logname)
            status_summary = status_summary or status

            if break_on_failure and status != 0:
                raise sandbox.common.errors.TaskFailure(
                    'Failed to execute {}'.format(command)
                )

            if set_info:
                self.set_info(
                    "[{}] {}".format(
                        'Success' if status == 0 else 'Failure',
                        command
                    )
                )

        return status_summary

    def exec_command(self, command, logname, cwd='.'):
        with sdk2.helpers.ProcessLog(self, logger=logname) as process_log:
            return sdk2.helpers.subprocess.Popen(
                ('bash', '-c', command),
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()


class TelegramNotifyHoursParameter(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value:
            if not re.match('^[0-9]{2}:[0-9]{2}-[0-9]{2}:[0-9]{2}$', value):
                raise ValueError('Wrong value')
        return value


class TelegramNotifyParameters(Parameters):
    with sdk2.parameters.Group('Telegram notification parameters.'):
        telegram_chat_id = sdk2.parameters.String('Telegram chat id')
        telegram_notify_only_if_tests_failed = sdk2.parameters.Bool(
            'Notify to telegram only if tests failed',
            default=True
        )
        telegram_notify_hours = TelegramNotifyHoursParameter(
            'Telegram notify hours',
            description='Europe/Moscow, by dash, example 08:00-24:00'
        )


class TelegramNotifyTaskMixin(Task):
    def telegram_notify_should_skip(self):
        if self.Parameters.telegram_notify_only_if_tests_failed and self.Context.tests_status_success:
            return True

        if self.Parameters.telegram_notify_hours:
            now = datetime.datetime.now(
                pytz.timezone('Europe/Moscow')
            )
            notify_hours = self.Parameters.telegram_notify_hours.split('-')

            start = map(int, notify_hours[0].split(':'))
            start_hours = start[0]
            start_minutes = start[1]

            end = map(int, notify_hours[1].split(':'))
            end_hours = end[0]
            end_minutes = end[1]

            if now.hour < start_hours \
                or (now.hour == start_hours and now.minute < start_minutes) \
                or now.hour > end_hours \
                or (now.hour == end_hours and now.minute > end_minutes):
                return True

    def telegram_notify_chat(self, chat_id):
        bot = telegram.TelegramBot(
            self.vault.read('WEATHER_TELEGRAM_BOT_TOKEN')
        )
        bot.send_message(
            chat_id,
            '{} [{}] {}'.format(
                self.Parameters.description,
                'Success' if self.Context.tests_status_success else 'Failure',
                self.Context.report_url
            )
        )
