#!/usr/bin/python
# -*- coding: utf-8 -*-

import collections
import logging
import re
import datetime as dt
import textwrap
import time

from pogoda_analytics_common import clickhouse as pclck


def days(num):
    return dt.timedelta(num)


OS_TO_PLATFORM = {
    "Windows Phone OS": "WindowsPhone",
    "Google Android": "android",
    "iOS": 'iOS'
}


def os_to_platform(line):
    return re.sub(r'^[^\t]+', lambda x: OS_TO_PLATFORM.get(x.group(0), 'unk'), line)

AttendRecord = collections.namedtuple(
    "AttendRecord", "StartDate  visits  users   new_users platform")
AttendRecordApp = collections.namedtuple(
    "AttendRecordApp", "platform StartDate  visits  users   new_users")
AttendRecordAppRegional = collections.namedtuple(
    "AttendRecordAppRegional", "platform StartDate district visits  users   new_users")
AttendRecordRegional = collections.namedtuple("AttendRecordRegional",
                                              "StartDate district visits users new_users platform")
AttendRecordGroupRegional = collections.namedtuple("AttendRecordGroupRegional",
                                                   "StartDate district visits users new_users region_visits visits_frac platform")


def parse_atten_line(line, platform):
    return AttendRecord._make(line.split() + [platform])


def parse_atten_line_reg(line, platform, record_type):
    if record_type == "region_group":
        return AttendRecordGroupRegional._make(line.split() + [platform])
    else:
        return AttendRecordRegional._make(line.split() + [platform])


def parse_atten(lines, platform):
    return [parse_atten_line(line, platform) for line in re.split(r"\n", lines) if re.search(r'\S', line)]


def statface_mapper(entry):
    return dict(
        fielddate=entry.StartDate,
        website=entry.platform,
        visits=int(entry.visits),
        users=int(entry.users),
        new_users=int(entry.new_users)
    )


UploadQuery = collections.namedtuple("UploadQuery", "query_id scale")
UploadURL = collections.namedtuple("UploadURL", "url scale")
ToCount = collections.namedtuple("ToCount", "query_id CounterID filters")
ToUpload = collections.namedtuple(
    "ToUpload", "query_id platform report_url record_type")
TO_UPLOAD = map(ToUpload._make, [
    ["desk_att", "desktop", 'pogoda.yandex/vitrina.new/totals-metrika-precise', "atten"],
    ["touch_att", "touch", 'pogoda.yandex/vitrina.new/totals-metrika-precise', "atten"],
    ["gismeteo_ru_desktop", "Gismeteo.ru desktop",
        'pogoda.yandex/vitrina.new/gismeteo-metrics', "atten"],
    ["gismeteo_by_desktop", "Gismeteo.by desktop",
        'pogoda.yandex/vitrina.new/gismeteo-metrics', "atten"],
    ["gismeteo_ru_mobile", "Gismeteo.ru mobile",
        'pogoda.yandex/vitrina.new/gismeteo-metrics', "atten"],
    ["gismeteo_by_mobile", "Gismeteo.by mobile",
        'pogoda.yandex/vitrina.new/gismeteo-metrics', "atten"],
    # ["m_gismeteo", "m.gismeteo.ru RSYA", 'pogoda.yandex/vitrina.new/gismeteo-metrics', "atten"],
    ["pp_webview_new", "PP", 'pogoda.yandex/vitrina.new/totals-metrika-precise', "atten"],
    ["nowcast_apps", None, 'pogoda.yandex/vitrina.new/totals-nowcast-precise', "webapp_atten"],
    ["nowcast_desktop", 'desktop',
        'pogoda.yandex/vitrina.new/totals-nowcast-precise', "atten"],
    ["nowcast_touch", 'touch', 'pogoda.yandex/vitrina.new/totals-nowcast-precise', "atten"],
    ["nowcast_pp", 'PP', 'pogoda.yandex/vitrina.new/totals-nowcast-precise', "atten"],
    ["details_desktop", 'desktop',
        'pogoda.yandex/vitrina.new/totals-details-precise', "atten"],
    ["details_touch", 'touch', 'pogoda.yandex/vitrina.new/totals-details-precise', "atten"],
    ["details_pp", 'PP', 'pogoda.yandex/vitrina.new/totals-details-precise', "atten"],
    ["desk_att_regional", "desktop",
        'pogoda.yandex/vitrina.new/totals-precise-regional', "region"],
    ["touch_att_regional", "touch",
        'pogoda.yandex/vitrina.new/totals-precise-regional', "region"],
    ["pp_webview_regional", "PP",
        'pogoda.yandex/vitrina.new/totals-precise-regional', "region"],
    ["gis_ru_desktop_regional", "Gismeteo.ru desktop",
        'pogoda.yandex/vitrina.new/totals-gis-metrics-regional', "region"],
    ["gis_by_desktop_regional", "Gismeteo.by desktop",
        'pogoda.yandex/vitrina.new/totals-gis-metrics-regional', "region"],
    ["gis_ru_mobile_regional", "Gismeteo.ru mobile",
        'pogoda.yandex/vitrina.new/totals-gis-metrics-regional', "region"],
    ["gis_by_mobile_regional", "Gismeteo.by mobile",
        'pogoda.yandex/vitrina.new/totals-gis-metrics-regional', "region"],
    # ["m_gismeteo_regional", "m.gismeteo.ru RSYA", 'pogoda.yandex/vitrina.new/totals-gis-metrics-regional', "region"],
    ["details_desktop_regional", 'desktop',
        'pogoda.yandex/vitrina.new/totals-details-precise-regional', "region"],
    ["details_touch_regional", 'touch',
        'pogoda.yandex/vitrina.new/totals-details-precise-regional', "region"],
    ["details_pp_regional", 'PP',
        'pogoda.yandex/vitrina.new/totals-details-precise-regional', "region"],
    ["nowcast_desktop_regional", 'desktop',
        'pogoda.yandex/vitrina.new/totals-nowcast-precise-regional', "region"],
    ["nowcast_touch_regional", 'touch',
        'pogoda.yandex/vitrina.new/totals-nowcast-precise-regional', "region"],
    ["nowcast_pp_regional", 'PP',
        'pogoda.yandex/vitrina.new/totals-nowcast-precise-regional', "region"],
    ["nowcast_apps_regional", None,
        'pogoda.yandex/vitrina.new/totals-nowcast-precise-regional', "region_webapp"]
])


TOUCH_FILTERS = [
    "StartURL not like '%nowcast=1%'",  # nowcast from apps
    # r"StartURL not like '%appsearch_header=1%'"     #searchapp webview
]

REGIONAL_FILTERS = [
    "regionToCountry(RegionID) = 225"  # only Russia
]

GISMETEO_FILTER = [
    # "StartURL not like '%/news%'"
]

ROBOT_FILTERS = [
    "IsRobot = 0",
    "IsRobotInternal = 0",
    "IsRobotInternalDelayed = 0",
    "IsRobotInternalDelayedPreprod = 0",
]

INTERESTING = map(ToCount._make, [
    ["desk_att", "Pogoda desktop", ROBOT_FILTERS],
    ["touch_att", "Pogoda touch", TOUCH_FILTERS + ROBOT_FILTERS],
    ["gismeteo_ru_desktop", "Gismeteo new", ["IsMobile == 0"] + GISMETEO_FILTER + ROBOT_FILTERS],
    ["gismeteo_by_desktop", "Gismeteo.by", ["IsMobile == 0"] + GISMETEO_FILTER + ROBOT_FILTERS],
    ["gismeteo_ru_mobile", "Gismeteo new", ["IsMobile == 1"] + GISMETEO_FILTER + ROBOT_FILTERS],
    ["gismeteo_by_mobile", "Gismeteo.by", ["IsMobile == 1"] + GISMETEO_FILTER + ROBOT_FILTERS],
    ["pp_webview_new", "Pogoda PP webview", None]
    # ["m_gismeteo", 42798839, None]
])

REGIONAL = map(ToCount._make, [
    ["desk_att_regional", "Pogoda desktop", REGIONAL_FILTERS + ROBOT_FILTERS],
    ["touch_att_regional", "Pogoda touch", [
        "StartURL not like '%appsearch_header=1%'"] + REGIONAL_FILTERS + ROBOT_FILTERS],
    ["pp_webview_regional", "Pogoda PP webview", REGIONAL_FILTERS + ROBOT_FILTERS],
    ["gis_ru_desktop_regional", "Gismeteo new",
        ["IsMobile == 0"] + REGIONAL_FILTERS+GISMETEO_FILTER + ROBOT_FILTERS],
    ["gis_by_desktop_regional", "Gismeteo.by", ["IsMobile == 0"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS],
    ["gis_ru_mobile_regional", "Gismeteo new", ["IsMobile == 1"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS],
    ["gis_by_mobile_regional", "Gismeteo.by", ["IsMobile == 1"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS]
    # ["m_gismeteo_regional", "m.gismeteo.ru RSYA", REGIONAL_FILTERS]
])

GISMETEO_TOUCH = {"Gismeteo new", "Gismeteo.by"}
GISMETEO_DESK = {"Gismeteo new", "Gismeteo.by", 45167301}

ALL_TOUCH = GISMETEO_TOUCH | {
    "Pogoda Mail RSYA",
    "Pogoda touch",
    "Pogoda PP webview",
    "Rambler Weather",
    "Pogoda desktop"
}
ALL_DESK = GISMETEO_DESK | {
    "Pogoda Mail RSYA",
    "Pogoda desktop",
    "Rambler Weather"
}


ToCountGroup = collections.namedtuple("ToCountGroup", "name counters filters")


COUNT_AUDIENCE = map(ToCountGroup._make, [
    # ("Pogoda Mail.ru total", {"Pogoda Mail RSYA"}, None),
    ("Pogoda Mail.ru desktop", {"Pogoda Mail RSYA"}, ["IsMobile == 0"] + ROBOT_FILTERS),
    ("Pogoda Mail.ru touch", {"Pogoda Mail RSYA"}, ["IsMobile == 1"] + ROBOT_FILTERS),
    # ("Gismeteo total", {"Gismeteo new", "Gismeteo.by", "m.gismeteo.ru RSYA"}, None),
    ("Gismeteo desktop", GISMETEO_DESK, ["IsMobile == 0"] + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Gismeteo touch", GISMETEO_TOUCH, ["IsMobile == 1"] + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Pogoda desktop", {"Pogoda desktop"}, ["IsMobile == 0"] + ROBOT_FILTERS),
    ("Pogoda touch", {"Pogoda touch", "Pogoda PP webview", "Pogoda desktop"},
        ["IsMobile == 1"] + TOUCH_FILTERS + ROBOT_FILTERS),
    ("Rambler desktop", {"Rambler Weather"}, ["IsMobile == 0"] + ROBOT_FILTERS),
    ("Rambler touch", {"Rambler Weather"}, ["IsMobile == 1"] + ROBOT_FILTERS),
    ("Pogoda + Gismeteo touch", GISMETEO_TOUCH | {"Pogoda touch", "Pogoda PP webview", "Pogoda desktop"},
        ["IsMobile == 1"] + TOUCH_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Pogoda + Gismeteo desktop", GISMETEO_DESK |
     {"Pogoda desktop"}, ["IsMobile == 0"] + GISMETEO_FILTER + ROBOT_FILTERS),
])

COUNT_AUDIENCE_REGIONAL = map(ToCountGroup._make, [
    ("Gismeteo desktop regional", GISMETEO_DESK,
     ["IsMobile = 0"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Gismeteo touch regional", GISMETEO_TOUCH,
     ["IsMobile = 1"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Pogoda desktop regional", {"Pogoda desktop"}, [
     "IsMobile = 0"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Pogoda touch regional", {"Pogoda touch", "Pogoda PP webview", "Pogoda desktop"},
     ["IsMobile == 1"] + TOUCH_FILTERS + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Pogoda Mail.ru desktop regional", {"Pogoda Mail RSYA"}, [
     "IsMobile == 0"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Pogoda Mail.ru touch regional", {"Pogoda Mail RSYA"}, [
     "IsMobile == 1"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Rambler desktop regional", {"Rambler Weather"}, [
     "IsMobile == 0"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Rambler touch regional", {"Rambler Weather"},
     ["IsMobile == 1"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Primpogoda desktop regional", {"Primpogoda"}, [
     "IsMobile = 0"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Primpogoda touch regional", {"Primpogoda"}, [
     "IsMobile = 1"] + REGIONAL_FILTERS + ROBOT_FILTERS),
    ("Pogoda + Gismeteo touch regional", GISMETEO_TOUCH | {"Pogoda touch", "Pogoda PP webview", "Pogoda desktop"},
     ["IsMobile == 1"] + TOUCH_FILTERS + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS),
    ("Pogoda + Gismeteo desktop regional", GISMETEO_DESK |
     {"Pogoda desktop"}, ["IsMobile == 0"] + REGIONAL_FILTERS + GISMETEO_FILTER + ROBOT_FILTERS)
])

OVERALL_GROUPS = map(ToCountGroup._make, [
    ["Total touch", ALL_TOUCH, ["IsMobile == 1"] + TOUCH_FILTERS + ROBOT_FILTERS],
    ["Total desktop", ALL_DESK, ["IsMobile == 0"] + ROBOT_FILTERS],
])

OVERALL_GROUPS_REGIONAL = map(ToCountGroup._make, [
    ["Total touch Pogoda regional", {"Pogoda touch", "Pogoda PP webview", "Pogoda desktop"}, [
        "IsMobile == 1"] + REGIONAL_FILTERS + ROBOT_FILTERS],
    ["Total touch regional", ALL_TOUCH | {"Primpogoda"}, [
        "IsMobile == 1"] + TOUCH_FILTERS + REGIONAL_FILTERS + ROBOT_FILTERS],
    ["Total desktop regional", ALL_DESK | {"Primpogoda"}, [
        "IsMobile == 0"] + REGIONAL_FILTERS + ROBOT_FILTERS]
])

MORDAS = map(ToCountGroup._make, [
    ["Morda touch", {"Morda desktop", "Morda touch"}, ["IsMobile == 1"] + ROBOT_FILTERS],
    ["Morda desktop", {"Morda desktop"}, ["IsMobile == 0"] + ROBOT_FILTERS],
])

for countgr in COUNT_AUDIENCE + OVERALL_GROUPS + MORDAS:
    TO_UPLOAD.append(
        ToUpload(countgr.name, countgr.name,
                 'pogoda.yandex/vitrina.new/competitors-and-us', "atten")
    )

for countgr in COUNT_AUDIENCE + OVERALL_GROUPS + MORDAS:
    TO_UPLOAD.append(
        ToUpload(countgr.name, countgr.name,
                 'pogoda.yandex/vitrina.new/competitors-and-us', "atten")
    )

for countgr in COUNT_AUDIENCE_REGIONAL + OVERALL_GROUPS_REGIONAL:
    TO_UPLOAD.append(
        ToUpload(countgr.name, countgr.name,
                 'pogoda.yandex/vitrina.new/competitor-and-us-regional', "region_group")
    )


def statface_dedup(AoD, unique_key):
    seen = set()
    for entry in AoD:
        combined_key = tuple([entry[field] for field in unique_key])
        if combined_key not in seen:
            seen.add(combined_key)
            yield entry
    return


ToTransfer = collections.namedtuple(
    'ToTransfer', "yt_path report_url fieldmap min_date cluster scale")

TO_TRANSFER = map(ToTransfer._make, [
    [
        '//home/meteo/bushivan/app_stats_recent',
        'pogoda.yandex/vitrina.new/totals-metrika-precise',
        dict(StartDate='fielddate', sessions='visits'),
        "2017-05-20",
        'hahn',
        "daily",
    ],
    [
        '//home/meteo/bushivan/money_recent',
        'pogoda.yandex/vitrina.new/money-fromlogs',
        dict(oborot_no_NDS='oborot_no_nds'),
        "2017-05-20",
        'hahn',
        "daily",
    ],
    [
        '//home/meteo/bushivan/pp_weather_ctr_recent',
        'pogoda.yandex/vitrina.new/weather-pp-ctr',
        dict(),
        "2017-05-20",
        'hahn',
        "daily",
    ],
    [
        '//home/weather-analytics/pmv/temp-for-stat',
        'pogoda.yandex/vitrina.new/pmv-temp-geo',
        dict(),
        "2017-05-20",
        'banach',
        "hourly",
    ],
    [
        '//home/weather-analytics/pmv/to-statface-latest',
        'pogoda.yandex/vitrina.new/PMV-2',
        dict(),
        "2017-05-20",
        'banach',
        "hourly",
    ],
    [
        '//home/meteo/babukhina/money_recent_regional',
        'pogoda.yandex/vitrina.new/money-fromlogs-regional',
        dict(oborot_no_NDS='oborot_no_nds'),
        "2017-05-20",
        'hahn',
        "daily",
    ]
])


DO_CLICKHOUSE = True


def calc_report(user, password, stat_token, yql_token, yt_token, end_date, count_last, publish_to_statface_prod):
    import statface_client

    config = {
        'user': user,
        'password': password,
        'stat_token': stat_token,
        'yql_token': yql_token,
        'yt_token': yt_token,
    }

    if DO_CLICKHOUSE:
        qg = pclck.QueryGenerator(config['user'], config['password'], output_format='TSV',
                                  server='mtstat',
                                  retries=10
                                  )

        down_data = {}
        NowQ = collections.namedtuple("NowQ", "name query")

        NOW_QS = [
            NowQ("nowcast_desktop", lambda start_date, end_date: qg.attendance_nowcast_desktop_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24*30*3600
            )),
            NowQ("nowcast_touch", lambda start_date, end_date: qg.attendance_nowcast_touch_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24*30*3600
            )),
            NowQ("nowcast_apps", lambda start_date, end_date: qg.attendance_nowcast_apps_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24*30*3600,
                groupby=["startdate_day", "os_parent"]
            )),
            NowQ("nowcast_pp", lambda start_date, end_date: qg.attendance_urlsvisit_withnew_or(
                "Pogoda PP webview", ["/nowcast", "/app-nowcast"],
                start_date, end_date, sample=0.5, exact=True, expiration=24*30*3600
            )),
            NowQ("details_desktop", lambda start_date, end_date: qg.attendance_details_desktop_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600
            )),
            NowQ("details_touch", lambda start_date, end_date: qg.attendance_details_touch_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600
            )),
            NowQ("details_pp", lambda start_date, end_date: qg.attendance_urlvisit_withnew(
                "Pogoda PP webview", "details",
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600
            )),
            NowQ("details_desktop_regional", lambda start_date, end_date: qg.attendance_details_desktop_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("details_touch_regional", lambda start_date, end_date: qg.attendance_details_touch_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("details_pp_regional", lambda start_date, end_date: qg.attendance_urlvisit_withnew(
                "Pogoda PP webview", "details",
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("nowcast_desktop_regional", lambda start_date, end_date: qg.attendance_nowcast_desktop_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("nowcast_touch_regional", lambda start_date, end_date: qg.attendance_nowcast_touch_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("nowcast_pp_regional", lambda start_date, end_date: qg.attendance_urlsvisit_withnew_or(
                "Pogoda PP webview", ["/nowcast", "/app-nowcast"],
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day']
            )),
            NowQ("nowcast_apps_regional", lambda start_date, end_date: qg.attendance_nowcast_apps_withnew(
                start_date, end_date, sample=0.5, exact=True, expiration=24 * 30 * 3600, filters=REGIONAL_FILTERS,
                groupby=['federal_district', 'startdate_day', 'os_parent']
            ))
        ]

        for maps_url in ('temperature', 'wind', 'pressure'):
            for platform in ('desktop', 'touch'):
                now_q_name = "{}_{}".format(maps_url, platform)
                TO_UPLOAD.append(ToUpload(
                    now_q_name,
                    platform,
                    'pogoda.yandex/vitrina.new/totals-{}-precise'.format(maps_url),
                    "atten"
                ))
                NOW_QS.append(NowQ(
                    now_q_name,
                    lambda start_date, end_date, counter_platform=platform, url=maps_url:
                        qg.attendance_urlvisit_withnew(
                            "Pogoda {}".format(
                                counter_platform), "maps/{}".format(url),
                            start_date, end_date, sample=0.5, exact=True, expiration=24*30*3600
                        )
                )),

        for platform in ('desktop', 'touch'):
            now_q_name = "total_maps_{}".format(platform)
            TO_UPLOAD.append(ToUpload(
                now_q_name,
                platform,
                'pogoda.yandex/vitrina.new/totals-maps-precise',
                "atten"
            ))
            NOW_QS.append(NowQ(
                now_q_name,
                lambda start_date, end_date, counter_platform=platform:
                    qg.attendance_web_withnew(
                        "Pogoda {}".format(counter_platform),
                        start_date, end_date, sample=1, exact=True, expiration=24*30*3600,
                        filters=[
                            textwrap.dedent("""\
                            arrayExists(
                                x_0 -> x_0 IN
                                    (
                                        SELECT WatchID
                                        FROM
                                            merge.hits SAMPLE {sample}
                                        WHERE
                                                EventDate >= toDate('{start_date}')
                                            and
                                                EventDate <= toDate('{end_date}')
                                            and
                                                CounterID = {counter}
                                            and
                                                NOT DontCountHits
                                            and
                                                NOT Refresh
                                            and
                                                (positionCaseInsensitive(URL,'/maps') > 0
                                                    or
                                                positionCaseInsensitive(URL,'/nowcast') > 0)
                                    ),
                                WatchIDs
                            )
                            """.format(
                                sample=0.5, start_date=start_date, end_date=end_date,
                                counter=pclck.COUNTER_IDS.get(
                                    "Pogoda {}".format(counter_platform))
                            ))
                        ],
                    )
            ))

        for tocount in REGIONAL:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.query_id, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_withnew(tocount.CounterID,
                                                    max_date -
                                                    days(count_last), max_date,
                                                    sample=1,
                                                    expiration=1*24*3600,
                                                    filters=tocount.filters,
                                                    groupby=[
                                                        'federal_district', 'startdate_day']
                                                    )
                    counter += 1
                    logging.debug("`{}' query {}".format(tocount.query_id, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        for tocount in COUNT_AUDIENCE_REGIONAL + OVERALL_GROUPS_REGIONAL:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.name, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_withnew_regional(tocount.counters,
                                                             max_date - days(count_last),
                                                             max_date,
                                                             sample=1,
                                                             expiration=1 * 24 * 3600,
                                                             filters=tocount.filters,
                                                             groupby=[
                                                                 'federal_district', 'startdate_day']
                                                             )
                    counter += 1
                    logging.debug("`{}' query {}".format(tocount.name, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        for tocount in INTERESTING:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.query_id, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_withnew(tocount.CounterID,
                                                    max_date -
                                                    days(count_last), max_date,
                                                    sample=1,
                                                    expiration=1*24*3600,
                                                    filters=tocount.filters
                                                    )
                    counter += 1
                    logging.debug("`{}' query {}".format(tocount.query_id, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        for tocount in COUNT_AUDIENCE + OVERALL_GROUPS:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.name, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_countergroup_withnew(tocount.counters,
                                                                 max_date -
                                                                 days(count_last),
                                                                 max_date,
                                                                 sample=1,
                                                                 expiration=1*24*3600,
                                                                 filters=tocount.filters
                                                                 )
                    counter += 1
                    logging.debug("`{}' group query {}".format(tocount.name, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        COUNT_ONE_WEEK = 10
        for tocount in COUNT_AUDIENCE + OVERALL_GROUPS:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.name, 'weekly')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_countergroup_withnew(tocount.counters,
                                                                 max_date -
                                                                 days(
                                                                     COUNT_ONE_WEEK), max_date,
                                                                 sample=1,
                                                                 expiration=1*24*3600,
                                                                 filters=(tocount.filters or []) +
                                                                 ["StartWeek >= toDate('{}')".format(
                                                                     max_date - days(COUNT_ONE_WEEK))],
                                                                 groupby=[
                                                                     'startdate_week']
                                                                 )
                    counter += 1
                    logging.debug("`{}' week group query {}".format(tocount.name, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        for tocount in MORDAS:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.name, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_countergroup_withnew(tocount.counters,
                                                                 max_date -
                                                                 days(count_last),
                                                                 max_date,
                                                                 sample=0.1,
                                                                 exact=False,
                                                                 expiration=1*24*3600,
                                                                 filters=tocount.filters
                                                                 )
                    counter += 1
                    logging.debug("`{}' group query {}".format(tocount.name, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        COUNT_ONE_WEEK = 10
        for tocount in MORDAS:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(tocount.name, 'weekly')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    myq = qg.attendance_web_countergroup_withnew(tocount.counters,
                                                                 max_date -
                                                                 days(
                                                                     COUNT_ONE_WEEK), max_date,
                                                                 sample=0.1,
                                                                 exact=False,
                                                                 expiration=1*24*3600,
                                                                 filters=(tocount.filters or []) +
                                                                 ["StartWeek >= toDate('{}')".format(
                                                                     max_date - days(COUNT_ONE_WEEK))],
                                                                 groupby=[
                                                                     'startdate_week']
                                                                 )
                    counter += 1
                    logging.debug("`{}' week group query {}".format(tocount.name, counter))
                    down_data[upl_key] += myq.data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        for now_q in NOW_QS:
            counter = 0
            max_date = dt.date.today() - days(1) if end_date is None else end_date
            num_retries = 100
            upl_key = UploadQuery(now_q.name, 'daily')
            down_data[upl_key] = ''
            while num_retries > 0:
                try:
                    counter += 1
                    logging.debug("`{}' query {}".format(now_q.name, counter))
                    down_data[upl_key] += now_q.query(
                        max_date - days(count_last), max_date).data()
                except pclck.ClickhouseQueryError as e:
                    logging.debug("Error `{}', retrying".format(e))
                    time.sleep(600)
                    num_retries -= 1
                else:
                    break

        statface_style = {}
        for to_up in TO_UPLOAD:
            for downloaded_key in [x for x in down_data.keys() if x.query_id == to_up.query_id]:
                upload_key = UploadURL(to_up.report_url, downloaded_key.scale)
                if upload_key not in statface_style:
                    statface_style[upload_key] = []
                # if to_up.query_id not in [x.query_id for x in down_data]:
                #     logging.debug("{} does not exist!".format(to_up.query_id))
                #     continue
                for line in re.split(r"\n", down_data[downloaded_key]):
                    if not re.search(r'\S', line):
                        continue
                    if to_up.record_type == "atten":
                        try:
                            entry = parse_atten_line(line, to_up.platform)
                        except TypeError:
                            logging.debug("Bad line: {}".format(line))
                            continue
                    elif "region" in to_up.record_type:
                        if to_up.record_type == "region_webapp":
                            try:
                                entry = AttendRecordAppRegional._make(
                                    os_to_platform(line).split())
                                if entry.platform not in ('android', 'iOS', 'WindowsPhone'):
                                    continue
                            except TypeError:
                                logging.debug("Bad line: {}".format(line))
                                continue
                        else:
                            try:
                                entry = parse_atten_line_reg(
                                    line, to_up.platform, to_up.record_type)
                            except TypeError:
                                logging.debug("Bad line: {}".format(line))
                                continue
                    elif to_up.record_type == "app_atten":
                        try:
                            entry = AttendRecordApp._make(line.split())
                            if entry.platform not in ('android', 'iOS', 'WindowsPhone'):
                                continue
                        except TypeError:
                            logging.debug("Bad line: {}".format(line))
                            continue
                    elif to_up.record_type == "webapp_atten":
                        try:
                            entry = AttendRecordApp._make(
                                os_to_platform(line).split())
                            if entry.platform not in ('android', 'iOS', 'WindowsPhone'):
                                continue
                        except TypeError:
                            logging.debug("Bad line: {}".format(line))
                            continue
                    else:
                        raise NotImplementedError(to_up.record_type)
                    if re.search('gismeteo|competitors', to_up.report_url):
                        statface_style[upload_key].append(dict(
                            fielddate=entry.StartDate,
                            website=entry.platform,
                            visits=int(entry.visits),
                            users=int(entry.users),
                            new_users=int(entry.new_users)
                        ))
                    elif re.search('regional', to_up.report_url):
                        if to_up.record_type == "region" or to_up.record_type == "region_webapp":
                            statface_style[upload_key].append(dict(
                                fielddate=entry.StartDate,
                                platform=entry.platform,
                                district=int(entry.district),
                                visits=int(entry.visits),
                                users=int(entry.users),
                                new_users=int(entry.new_users)
                            ))
                        elif to_up.record_type == "region_group":
                            statface_style[upload_key].append(dict(
                                fielddate=entry.StartDate,
                                platform=entry.platform,
                                district=int(entry.district),
                                visits=int(entry.visits),
                                users=int(entry.users),
                                new_users=int(entry.new_users),
                                region_visits=int(entry.region_visits),
                                visits_frac=float(entry.visits_frac)
                            ))
                        else:
                            continue
                    else:
                        statface_style[upload_key].append(dict(
                            fielddate=entry.StartDate,
                            platform=entry.platform,
                            visits=int(entry.visits),
                            users=int(entry.users),
                            new_users=int(entry.new_users)
                        ))

        st_client = statface_client.StatfaceClient(
            host=statface_client.STATFACE_PRODUCTION if publish_to_statface_prod else None,
            oauth_token=config['stat_token'],
        )
        statupload_retries = 1
        while True:
            try:
                for upload_key in statface_style.keys():
                    logging.debug("Uploading to {}, {}".format(upload_key.url, upload_key.scale))
                    if re.search('gismeteo|competitors', upload_key.url):
                        deduplicated = statface_dedup(statface_style[upload_key], [
                                                      'fielddate', 'website'])
                    elif re.search('regional', upload_key.url):
                        deduplicated = statface_dedup(statface_style[upload_key], [
                                                      'fielddate', 'platform', 'district'])
                    else:
                        deduplicated = statface_dedup(statface_style[upload_key], [
                                                      'fielddate', 'platform'])
                    report = st_client.get_report(upload_key.url)
                    try:
                        report.upload_data(
                            scale=upload_key.scale, data=deduplicated)
                    except TypeError as e:
                        # logging.debug(statface_style[upload_url])
                        raise e
            except statface_client.errors.StatfaceHttpResponseError as stat_e:
                statupload_retries -= 1
                if statupload_retries > 0:
                    logging.debug("Statface error {}".format(stat_e))
                    logging.debug("Retrying...")
                    continue
                else:
                    raise stat_e
            else:
                break

        logging.debug("Clickhouse Success.")
