#!/usr/bin/python
# -*- coding: utf-8 -*
import requests
import requests.exceptions
import time
import datetime as dt
import re
import textwrap
import collections
import shelve

CLICKHOUSE_SERVERS = {
    'mtstat': 'http://mtstat.yandex.ru:8123/',
    'mtsmart': 'http://mtsmart.yandex.ru:8123/',
    'mtmega': 'http://mtmega.yandex.ru:8123/'
}

COUNTER_NAMES = {
    115080: "Pogoda desktop",
    345696: "Pogoda touch",
    44593393: "Pogoda PP webview",
    43422864: "Pogoda testing",
    5212426: "Pogoda mini",
    39370095: "Gismeteo new",
    35824995: "Gismeteo.by",
    1028356: "Music",
    26750127: "Meduza.io",
    29112495: "Radio",
    160656: "Market",
    29582730: "Rambler Weather",
    33270605: "Pogoda Mail RSYA",
    42798839: 'm.gismeteo.ru RSYA',
    38239695: "m.gismeteo.ru RSYA 2",
    722545: "Morda desktop",
    23474449: "Morda touch",
    25153646: "Primpogoda"

}

API_KEY_NAMES = {
    28210: "Pogoda",
    42989: "PP iOS",
    10321: "PP android"
}

cities_for_regional_query = (213, 2)

COUNTER_IDS = {v: k for k, v in COUNTER_NAMES.items()}

for k, v in COUNTER_NAMES.items():
    COUNTER_NAMES[str(k)] = v

API_KEYS = {v: k for k, v in API_KEY_NAMES.items()}

for k, v in API_KEY_NAMES.items():
    API_KEY_NAMES[str(k)] = v

# arrayExists(x_0,x_1 -> x_0 =  'test_id'  AND (x_1 IN('45260', '45261')),
#                             `ParsedParams.Key1`,`ParsedParams.Key2`)


def param_extract(param_expr_list, level=1):
    '''
    generate a parameter extraction string for prallel arrays used in Clickhouse to store visit parameters
    parameter at <level> is "brought to the front" and will be the result of arrayFilter
    '''
    field_names = []
    exprs = []

    for i in xrange(max(len(param_expr_list), level)):
        field_names.append(
            (i+1, "k_{}".format(i+1), '`ParsedParams.Key{}`'.format(i+1)))
    for i, expr in enumerate(param_expr_list):
        if expr is not None:
            if not re.search(r'\{arg\}', expr):
                exprs.append("({} {})".format(field_names[i][1], expr))
            else:
                exprs.append("({})".format(expr.format(arg=field_names[i][1])))
    # move desired level first
    field_names.sort(key=lambda x: x[0] == level, reverse=True)
    req_expression = "{args} -> {exprs}, {fields}".format(
        args=','.join([x[1] for x in field_names]),
        fields=','.join([x[2] for x in field_names]),
        exprs=' and '.join(exprs)
    )
    return req_expression


def param_exists(param_expr_list):
    return 'arrayExists({})'.format(param_extract(param_expr_list))


def param_filter(param_expr_list, level):
    '''returns param level filtered by equation'''
    return 'arrayFilter({})'.format(param_extract(param_expr_list, level))


def param_filter_member(param_expr_list, level=1, member=1):
    return '{}[{}]'.format(param_filter(param_expr_list, level), member)


def in_expr(vals):
    return "in({})".format(",".join(["'{}'".format(x) for x in vals]))


def testid_exists(testids):
    return param_exists(["== 'test_id'", in_expr(testids)])


def testid_filter(testids):
    in_expr = "in({})".format(",".join(["'{}'".format(x) for x in testids]))
    return param_filter(["== 'test_id'", in_expr], 2)


def testid_filter_first(testids):
    return "{}[1]".format(testid_filter(testids))


def visited_counters(counterids, start_date, end_date, sample=1, negative=False):
    counterstr = 'CounterID IN({})'.format(
        ",".join([str(COUNTER_IDS.get(ci, ci)) for ci in counterids]))
    return textwrap.dedent("""\
        UserID GLOBAL {maybenot} IN (
            SELECT DISTINCT
                UserID
            from
                visits_all sample {sample}
            where
                    {counterstr}
                and
                    StartDate >= toDate('{start}')
                and
                    StartDate <= toDate('{end}')
        )
        """.format(
        counterstr=counterstr,
        sample=sample,
        start=start_date,
        end=end_date,
        maybenot=("NOT" if negative else "")
    ))


def counter_filter(counterid):
    if counterid is None:
        return []
    counter_num = str(COUNTER_IDS.get(counterid, counterid))
    if not re.search(r'^\d+$', counter_num):
        raise ValueError("Unknown counter name `{}'".format(counterid))
    counter_expr = ['CounterID = {}'.format(
        COUNTER_IDS.get(counterid, counterid))]
    return counter_expr


def district(city):
    return textwrap.dedent("""\
        CASE
            WHEN (regionToCity(RegionID) IN {city})
            THEN regionToCity(RegionID)
            ELSE regionToDistrict(RegionID)
            END""").format(city=city)


GroupBy = collections.namedtuple('GroupBy', 'alias expr')

GROUPBY_COMMON = {              # tuples
    'startdate_day': GroupBy('StartDate', 'StartDate'),
    'startdate_week':  GroupBy('StartWeek', 'toMonday(StartDate)'),
    'startdate_month': GroupBy('StartMonth', 'toStartOfMonth(StartDate)'),
    'eventdate_day': GroupBy('EventDate', 'EventDate'),
    'eventdate_week':  GroupBy('EventWeek', 'toMonday(EventDate)'),
    'eventdate_month': GroupBy('EventMonth', 'toStartOfMonth(EventDate)'),
    'clienttime_hour': GroupBy('ClientHour', 'toStartOfHour(toDateTime(StartTimestamp + StartTimeZone))'),
    'app_platform': GroupBy('AppPlatform', 'AppPlatform'),
    'os_parent': GroupBy("os_parent", "dictGetString('OS', 'value', dictGetUInt64('OS', 'ParentId', toUInt64(OS)))"),
    'useragent_name': GroupBy('UserAgentName', "dictGetString('UserAgent', 'value', toUInt64(UserAgent))"),
    'geo_city': GroupBy('geo_city', 'regionToName(regionToCity(RegionID))'),
    'geo_city_id': GroupBy('geo_city', 'regionToCity(RegionID)'),
    'geo_area': GroupBy('geo_area', 'regionToName(regionToArea(RegionID))'),
    'geo_oblast': GroupBy('geo_area', 'regionToName(regionToArea(RegionID))'),
    'federal_district': GroupBy('district', district(cities_for_regional_query))
}


RequestResult = collections.namedtuple('RequestResult', 'timestamp result')


# Grouped = collections.namedtuple('GroupBy', 'alias expr having')

class ClickhouseQueryError(Exception):
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)


class Query(object):
    """
    Implements extermely basic object interface for Clickhouse queries used in Weather analytics.
    Specif
    """

    TIMEOUT = 1500

    def __init__(self, sql, user, password, server='mtstat', expiration=600, retries=2,
                 persistence=None, storage_path=None
                 ):
        super(Query, self).__init__()
        self.sql = sql
        self.user = user
        self.password = password
        self.server = CLICKHOUSE_SERVERS.get(server, server)
        self.result = None
        self.valid_till = 0
        self.expiration = expiration  # delay in seconds
        self.retries = retries
        self.persistence = persistence
        self.storage_path = storage_path
        if (self.storage_path is None) and self.persistence:
            self.storage_path = 'results_cache.' + self.persistence

    def data(self, ignore_cache=False):
        """
        Returns the data corresponding to a query (cached data for successful requests)
        """
        if self.persistence:
            if self.persistence == 'shelve':
                cache = shelve.open(self.storage_path)
                if self.sql in cache:
                    self.result = cache[self.sql].result
                    self.valid_till = cache[self.sql].timestamp + \
                        self.expiration
                cache.close()
            else:
                raise NotImplementedError(
                    'Storage engine {} unknown'.format(self.persistence))

        n_tried = 0
        if ignore_cache or self.valid_till < time.time():
            while True:
                try:
                    # print self.sql
                    r = requests.post(
                        self.server,
                        auth=(self.user, self.password),
                        params={'query': self.sql},
                        timeout=Query.TIMEOUT
                    )
                except requests.exceptions.RequestException as err:
                    if n_tried < self.retries:
                        n_tried += 1
                        continue
                    else:
                        raise err
                else:
                    break

            self.result = r.text
            # if re.search(r'DB::\w*Exception', self.result):
            if r.status_code != requests.codes.ok:
                raise ClickhouseQueryError(
                    "Request error code {}! Also, DB returned: {}".format(
                        r.status_code, self.result)
                )
            if re.search(r'DB::\w*Exception', self.result):
                raise ClickhouseQueryError(self.result)
            if self.persistence:  # if cache is ignored, we still write an update
                if self.persistence == 'shelve':
                    cache = shelve.open(self.storage_path)
                    cache[self.sql] = RequestResult(time.time(), self.result)
                    cache.close()
                else:
                    raise NotImplementedError(
                        'Storage engine {} unknown'.format(self.persistence))
            self.valid_till = time.time() + self.expiration
        return self.result

    def tofile(self, filename, flags="w"):
        with open(filename, flags) as outfile:
            outfile.write(self.data().encode("utf8"))


class QueryGenerator(object):
    """\
    QueryGenerator creates Clickhouse queries for analytics applications using templates.
    Each template is a method.
    """

    def __init__(self, user, password, server='mtstat', output_format="TSVWithNames",
                 retries=2, persistence=None, storage_path=None
                 ):
        super(QueryGenerator, self).__init__()
        self.user = user
        self.password = password
        self.server = CLICKHOUSE_SERVERS.get(server, server)
        self.output_format = output_format
        self.retries = retries
        self.persistence = persistence
        self.storage_path = storage_path
        if (self.storage_path is None) and self.persistence:
            self.storage_path = 'results_cache.' + self.persistence

    def change_merge_table(self, sql):
        if 'mtstat' in self.server:
            return re.sub(r'\b merge\.hits \b', 'merge.hits_v2', sql, flags=re.VERBOSE)
        else:
            return sql

    def gen(self, sql, expiration=600):
        return Query(self.change_merge_table(sql), self.user, self.password, self.server, expiration,
                     retries=self.retries,
                     persistence=self.persistence,
                     storage_path=self.storage_path
                     )

    def _from_skeleton(self, skeleton, what, where, groupby=None, sample=1, having=None, testids=None, testids_app=None,
                       outer_what=None, outer_where=None, outer_groupby=None, inner_any=False, offset=0):
        # outer groupby is useful only for joins, not for subqueries

        if groupby is None:
            groupby = []
        if outer_what is None:
            outer_what = []
        if outer_where is None:
            outer_where = []
        if outer_groupby is None:
            outer_groupby = []
        if having is None:
            having = []
        groupby = [GROUPBY_COMMON.get(x, x) for x in groupby]
        outer_groupby = [GROUPBY_COMMON.get(x, x) for x in outer_groupby]
        if testids_app is not None:
            where += ["EventName == 'TestIds'",
                      param_exists([in_expr(testids_app)])]
            groupby += [GroupBy('gr',
                                param_filter_member([in_expr(testids_app)]))]
        if testids is not None:
            groupby.append(GroupBy('gr', testid_filter_first(testids)))
            where += [testid_exists(testids)]
        groupby_aliases = []
        for group_field in groupby:
            # print group_field
            try:
                if inner_any:
                    what = ['any({}) as {}'.format(
                        group_field.expr, group_field.alias)] + what
                else:
                    what = ['{} as {}'.format(
                        group_field.expr, group_field.alias)] + what
            except AttributeError:
                raise NotImplementedError(
                    "Cannot interpret {} as a group by expression".format(group_field))
            groupby_aliases.append(group_field.alias)
            outer_what = [group_field.alias] + outer_what
        for group_field in outer_groupby:
            groupby_aliases.append(group_field.alias)
            outer_what = ['{} as {}'.format(
                group_field.expr, group_field.alias)] + outer_what
        sql = skeleton.format(
            what=", ".join(what),
            sample=sample,
            where=" and ".join(['({})'.format(x) for x in where]),
            outer_where=" and ".join(['({})'.format(x) for x in outer_where]),
            groupby_aliases=", ".join(groupby_aliases),
            output_format=self.output_format,
            having=(" and ".join(['({})'.format(x) for x in having]) or 1),
            outer_what=", ".join(outer_what),
            offset=offset
        )
        return sql

    def attendance_web(
        self,
        counterid, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []

        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def distinct_day_users(
        self,
        counterid, start_date, end_date=None, filters=None, sample=0.01,
        expiration=23*3600, seen_at_least=2
    ):
        query_skeleton = textwrap.dedent("""\
            SELECT {what}
            FROM
            (
                SELECT
                    UserID
                    FROM visits_all
                    SAMPLE {sample}
                    WHERE {where}
                GROUP BY UserID
                having
                    {having}
            )
            format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []

        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        having = [
            "uniqUpTo({numseen})(StartDate) >= {numseen}".format(
                numseen=seen_at_least)
        ]
        what = [
            'count(*)*{} as users'.format(1./sample)
        ]
        sql = self._from_skeleton(
            query_skeleton, what=what, where=where, sample=sample, having=having)

        return self.gen(sql, expiration)

    def revisitors_by_uids(
        self,
        counterid, start_date, end_date=None, testids=None, filters=None, sample=0.1,
        expiration=23*3600, max_days_since=29
    ):
        # select StartDate, countIf(runningDifference(StartDate) < 30 AND runningDifference(UserID) == 0)*100 as revisits
        # from (
        #         Select DISTINCT StartDate, UserID
        #         from visits_all sample 0.01
        #         where CounterID == 115080 and StartDate > toDate('2017-01-30') and IsMobile ==0
        #         ORDER BY
        #         UserID, StartDate
        #     )
        # group by StartDate
        # having StartDate > toDate('2017-01-30') + 30
        # max_days_since - delta vith prev visit, e.g. for yesterday it is 1
        query_skeleton = textwrap.dedent("""\
            SELECT {outer_what}
            FROM
            (
                SELECT DISTINCT
                    UserID, StartDate
                    FROM visits_all
                    SAMPLE {sample}
                    WHERE {where}
                ORDER BY
                    UserID, StartDate
            )
            group by {groupby_aliases}
            having {having}
            order by {groupby_aliases}
            format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []

        # if groupby is None:
        groupby = ['startdate_day']
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}') - {}".format(start_date,
                                                    max_days_since),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        having = [
            "StartDate >= toDate('{}')".format(start_date)
        ]
        outer_what = [
            'countIf(runningDifference(StartDate) <= {} AND runningDifference(UserID) == 0)*{} as users'.format(
                max_days_since,
                1./sample,
            )
        ]
        what = [
            'UserID',
            'StartDate',
        ]
        sql = self._from_skeleton(
            query_skeleton, what=what, where=where, sample=sample, having=having,
            outer_what=outer_what,
            groupby=groupby,
        )

        return self.gen(sql, expiration)

    def dump_userids(
        self,
        counterid, start_date, end_date=None, filters=None, sample=0.01,
        expiration=23*3600
    ):
        query_skeleton = textwrap.dedent("""\
            SELECT {what}
            FROM visits_all SAMPLE {sample}
            WHERE {where}
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        what = [
            'UserID',
            'StartDate'
        ]
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        sql = self._from_skeleton(
            query_skeleton, what=what, where=where, sample=sample)
        return self.gen(sql, expiration)

    def attendance_web_countergroup_withnew(
        self,
        counterids, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = [
            'CounterID IN({})'.format(
                ",".join([str(COUNTER_IDS.get(ci, ci)) for ci in counterids])),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample),
            'uniq{}If(UserID, FirstVisit == StartTime)*{} as new_users'.format(
                'Exact' if exact else 'Combined', 1./sample
            )
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def attendance_web_withnew_regional(
        self,
        counterids, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23 * 3600
    ):

        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []

        query_skeleton = textwrap.dedent("""\
            select
                {outer_what}
            from
                (select
                    {what}
                from
                    visits_all sample {sample}
                where
                    {where}
                group by
                    {groupby_aliases})
            """) \
            + textwrap.dedent("""\
            all inner join
                (select
                    StartDate,
                    sum(Sign)*10 as region_visits,
                    {district} as district
                from
                    visits_all sample 1/10
                where
                    StartDate >= toDate('{start_date}') and
                    StartDate <= toDate('{end_date}')
                    and (regionToCountry(RegionID) = 225)
                group by
                    StartDate, district
                )
                using (StartDate, district)
            """).format(district=district(cities_for_regional_query), start_date=start_date, end_date=end_date) \
            + textwrap.dedent("""\
            order by
                {groupby_aliases}
                format {output_format};
            """)

        where = [
            'CounterID IN({})'.format(
                ",".join([str(COUNTER_IDS.get(ci, ci)) for ci in counterids])),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1. / sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1. / sample),
            'uniq{}If(UserID, FirstVisit == StartTime)*{} as new_users'.format(
                'Exact' if exact else 'Combined', 1. / sample
            )
        ]
        outer_what = [
            'visits',
            'users',
            'new_users',
            'region_visits',
            '(visits/region_visits)*100 as visits_frac'
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, outer_what=outer_what, testids=testids)

        return self.gen(sql, expiration)

    def attendance_web_withnew(
        self,
        counterid, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample),
            'uniq{}If(UserID, FirstVisit == StartTime)*{} as new_users'.format(
                'Exact' if exact else 'Combined', 1./sample
            )
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def hits_web(
        self,
        counterid, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600, min_hits=10
    ):
        """
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            hits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        having
            {having}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "EventDate >= toDate('{}')".format(start_date),
            "EventDate <= toDate('{}')".format(end_date),
            "NOT DontCountHits",
            "NOT Refresh",
        ] + filters
        what = [
            'count()*{} as hits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample),
        ]
        if groupby is None:
            groupby = ['eventdate_day']
        having = [
            'hits >= {}'.format(min_hits)
        ]
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids, having=having)

        return self.gen(sql, expiration)

    def attendance_weather_touch(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL not like '%nowcast=1%'",  # nowcast from apps
            r"StartURL not like '%appsearch_header=1%'"  # searchapp webview
        ]
        return self.attendance_web('Pogoda touch', *args, **kwargs)

    def attendance_weather_desktop(self, *args, **kwargs):
        return self.attendance_web('Pogoda desktop', *args, **kwargs)

    def attendance_app(
        self,
        apikey, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.001, exact=False,
        expiration=23*3600
    ):
        """\
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            mobile.events_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = [
            'APIKey == {}'.format(API_KEYS.get(apikey, apikey)),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'uniq{}If(DeviceIDSessionIDHash,SessionType = 0 and EventType = 2)*{} as sessions'.format(
                'Exact' if exact else 'Combined', 1./sample
            ),
            'uniq{}If(DeviceIDHash, SessionType = 0 and EventType = 2)*{} as devices'.format(
                'Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day', 'app_platform']

        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids_app=testids)
        return self.gen(sql, expiration)

    def interactive_sessions_app(
        self,
        apikey, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.001, exact=False,
        expiration=23*3600
    ):
        """\
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            mobile.events_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = [
            'APIKey == {}'.format(API_KEYS.get(apikey, apikey)),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'uniq{}If(DeviceIDSessionIDHash,SessionType = 0)*{} as sessions'.format(
                'Exact' if exact else 'Combined', 1./sample
            ),
            'uniq{}If(DeviceIDHash, SessionType = 0)*{} as devices'.format(
                'Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day', 'app_platform']

        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids_app=testids)
        return self.gen(sql, expiration)

    def attendance_app_withnew(
        self,
        apikey, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.001, exact=False,
        expiration=23*3600
    ):
        """\
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            mobile.events_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = [
            'APIKey == {}'.format(API_KEYS.get(apikey, apikey)),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        what = [
            'uniq{}If(DeviceIDSessionIDHash, SessionType = 0 and EventType = 2)*{} as sessions'.format(
                'Exact' if exact else 'Combined', 1./sample
            ),
            'uniq{}If(DeviceIDHash, SessionType = 0 and EventType = 2)*{} as devices'.format(
                'Exact' if exact else 'Combined', 1./sample),
            'uniq{}If(DeviceIDHash, EventType == 1)*{} as new_devices'.format(
                'Exact' if exact else 'Combined', 1./sample
            )
        ]
        if groupby is None:
            groupby = ['startdate_day', 'app_platform']

        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids_app=testids)
        return self.gen(sql, expiration)

    def attendance_weather_app(self, *args, **kwargs):
        return self.attendance_app('Pogoda', *args, **kwargs)

    def all_retentions_web(
        self,
        counterid, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        query_skeleton = textwrap.dedent("""\
        select
            {outer_what}
        from
            (select
                {what}
            from
                visits_all sample {sample}
            where
                 {where}
            group by
                UserID,
                StartDate
            ) global all inner join (
                select
                    UserID,
                    StartDate as compar_date,
                    sum(Sign) as visits
                from visits_all sample {sample}
                where
                    {where}
                group by
                    UserID,
                    StartDate
                having
                    visits > 0
            ) using UserID
        where
            {outer_where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date)
        ] + filters
        outer_where = ['days_since1 >= 0']
        outer_what = [
            'sum(visits)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        what = [
            'UserID',
            'StartDate'
        ]
        if groupby is None:
            groupby = []
        groupby = [GroupBy('calc_date', 'StartDate')] + groupby
        outer_groupby = [GroupBy('days_since1', '(compar_date - calc_date)')]
        sql = self._from_skeleton(query_skeleton, what, where, groupby, sample,
                                  testids=testids, outer_groupby=outer_groupby, outer_what=outer_what, outer_where=outer_where,
                                  inner_any=True
                                  )
        return self.gen(sql, expiration)

    def all_retentions_web_manycount(
        self,
        counterids, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600, offset=0
    ):
        query_skeleton = textwrap.dedent("""\
        select
            {outer_what}
        from
            (select
                {what}
            from
                visits_all sample {sample} offset {offset}
            where
                 {where}
            group by
                UserID,
                StartDate
            ) global all inner join (
                select
                    UserID,
                    StartDate as compar_date,
                    sum(Sign) as visits
                from visits_all sample {sample} offset {offset}
                where
                    {where}
                group by
                    UserID,
                    StartDate
                having
                    visits > 0
            ) using UserID
        where
            {outer_where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        counter_expr = ["(" + " OR ".join([counter_filter(cid)[0]
                                           for cid in counterids]) + ")"]
        where = counter_expr + \
            [
                "StartDate >= toDate('{}')".format(start_date),
                "StartDate <= toDate('{}')".format(end_date)
            ] + filters
        outer_where = ['days_since1 >= 0']
        outer_what = [
            'sum(visits)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        what = [
            'UserID',
            'StartDate'
        ]
        if groupby is None:
            groupby = []
        groupby = [GroupBy('calc_date', 'StartDate')] + groupby
        outer_groupby = [GroupBy('days_since1', '(compar_date - calc_date)')]
        sql = self._from_skeleton(query_skeleton, what, where, groupby, sample,
                                  testids=testids, outer_groupby=outer_groupby, outer_what=outer_what, outer_where=outer_where,
                                  inner_any=True, offset=offset
                                  )
        return self.gen(sql, expiration)

    def all_retentions_app(
        self,
        apikey, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        query_skeleton = textwrap.dedent("""\
        select
            {outer_what}
        from
            (select
                {what}
            from
                mobile.events_all sample {sample}
            where
                 {where}
            group by
                DeviceIDHash,
                StartDate
            ) global all inner join (
                select
                    DeviceIDHash,
                    StartDate as compar_date,
                    uniqCombined(DeviceIDSessionIDHash) as visits
                from mobile.events_all sample {sample}
                where
                    {where}
                group by
                    DeviceIDHash,
                    StartDate
                having
                    visits > 0
            ) using DeviceIDHash
        where
            {outer_where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = [
            "APIKey == {}".format(API_KEYS.get(apikey, apikey)),
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            "SessionType = 0"
        ] + filters
        outer_where = ['days_since1 >= 0']
        outer_what = [
            'sum(visits) as sessions',
            'uniq{}(DeviceIDHash)*{} as devices'.format(
                'Exact' if exact else 'Combined', 1./sample),
        ]
        what = [
            'DeviceIDHash',
            'StartDate'
        ]
        if groupby is None:
            groupby = []
        if groupby is None:
            groupby = []
        groupby = [GroupBy('calc_date', 'StartDate')] + groupby
        outer_groupby = [GroupBy('days_since1', '(compar_date - calc_date)')]
        sql = self._from_skeleton(query_skeleton, what, where, groupby, sample,
                                  testids_app=testids, outer_groupby=outer_groupby, outer_what=outer_what, outer_where=outer_where,
                                  inner_any=True
                                  )
        return self.gen(sql, expiration)

    def timings_touch(
        self,
        counterid, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, quantile=0.75,
        min_hits=100,
        expiration=3600
    ):
        '''\
        These timings were found useful for profiling touch websites via Metrika
        '''
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            hits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        having
            {having}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        what = [
            'uniqCombined(WatchID) as matching_hits',
            '{} as quantile_level'.format(quantile),
            '''quantileTimingIf(quantile_level)(DOMCompleteTiming + LoadEventEndTiming,
                            DOMCompleteTiming NOT IN (-1, 0) AND LoadEventEndTiming NOT IN (-1, 0)
                        ) as fulload''',
            '''quantileTimingIf(quantile_level)(ResponseEndTiming + ResponseStartTiming,
                            ResponseEndTiming NOT IN (-1, 0) AND ResponseStartTiming NOT IN (-1, 0)
                        ) as server_response''',
            'quantileTimingIf(quantile_level)(FirstPaintTiming, FirstPaintTiming != -1) as Otrisovka',
            'quantileTimingIf(quantile_level)(DOMInteractiveTiming, DOMInteractiveTiming != -1) as DOMLoad'
        ]
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "EventDate >= toDate('{}')".format(start_date),
            "EventDate <= toDate('{}')".format(end_date)
        ] + filters
        if groupby is None:
            groupby = ['os_parent', 'useragent_name']
        having = [
            'matching_hits >= {}'.format(min_hits)
        ]

        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids, having=having)
        return self.gen(sql, expiration)

    def attendance_urlvisit(
        self,
        counterid, url, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        if re.search('mtstat', self.server):
            raise NotImplementedError("This query won't work on mtstat")
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            textwrap.dedent("""\
            arrayExists(
                x_0 -> x_0 IN
                    (
                        SELECT WatchID
                        FROM
                            merge.hits SAMPLE {sample}
                        WHERE
                                EventDate >= toDate('{start_date}')
                            and
                                EventDate <= toDate('{end_date}')
                            and
                                CounterID = {counter}
                            and
                                NOT DontCountHits
                            and
                                NOT Refresh
                            and
                                positionCaseInsensitive(URL,'{url}') > 0
                    ),
                `Event.ID`
            )
            """.format(
                sample=sample, start_date=start_date, end_date=end_date,
                counter=COUNTER_IDS.get(counterid, counterid),
                url=url
            ))
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def attendance_goalvisit(
        self,
        counterid, url, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        if re.search('mtstat', self.server):
            raise NotImplementedError("This query won't work on mtstat")
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            textwrap.dedent("""\
            arrayExists(
                x_0 -> x_0 IN
                    (
                        SELECT WatchID
                        FROM
                            merge.hits SAMPLE {sample}
                        WHERE
                                EventDate >= toDate('{start_date}')
                            and
                                EventDate <= toDate('{end_date}')
                            and
                                CounterID = {counter}
                            and
                                URL like 'goal:%'
                            and
                                positionCaseInsensitive(URL,'{url}') > 0
                    ),
                `Event.ID`
            )
            """.format(
                sample=sample, start_date=start_date, end_date=end_date,
                counter=COUNTER_IDS.get(counterid, counterid),
                url=url
            ))
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def attendance_urlvisit_watchids(
        self,
        counterid, url, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        like URLvisit, but for mtstat
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            textwrap.dedent("""\
            arrayExists(
                x_0 -> x_0 IN
                    (
                        SELECT WatchID
                        FROM
                            merge.hits SAMPLE {sample}
                        WHERE
                                EventDate >= toDate('{start_date}')
                            and
                                EventDate <= toDate('{end_date}')
                            and
                                CounterID = {counter}
                            and
                                NOT DontCountHits
                            and
                                NOT Refresh
                            and
                                positionCaseInsensitive(URL,'{url}') > 0
                    ),
                WatchIDs
            )
            """.format(
                sample=sample, start_date=start_date, end_date=end_date,
                counter=COUNTER_IDS.get(counterid, counterid),
                url=url
            ))
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample)
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def attendance_urlvisit_withnew(
        self,
        counterid, url, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        # if re.search('mtstat', self.server):
        #     raise NotImplementedError("This query won't work on mtstat")
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            textwrap.dedent("""\
            arrayExists(
                x_0 -> x_0 IN
                    (
                        SELECT WatchID
                        FROM
                            merge.hits SAMPLE {sample}
                        WHERE
                                EventDate >= toDate('{start_date}')
                            and
                                EventDate <= toDate('{end_date}')
                            and
                                CounterID = {counter}
                            and
                                NOT DontCountHits
                            and
                                NOT Refresh
                            and
                                positionCaseInsensitive(URL,'{url}') > 0
                    ),
                WatchIDs
            )
            """.format(
                sample=sample, start_date=start_date, end_date=end_date,
                counter=COUNTER_IDS.get(counterid, counterid),
                url=url
            ))
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample),
            'uniq{}If(UserID, FirstVisit == StartTime)*{} as new_users'.format(
                'Exact' if exact else 'Combined', 1./sample
            )
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    def attendance_urlsvisit_withnew_or(
        self,
        counterid, url, start_date, end_date=None, testids=None, groupby=None, filters=None, sample=0.1, exact=True,
        expiration=23*3600
    ):
        """
        calculates visitors and unique users according to filters
        """
        query_skeleton = textwrap.dedent("""\
        select
            {what}
        from
            visits_all sample {sample}
        where
            {where}
        group by
            {groupby_aliases}
        order by
            {groupby_aliases}
        format {output_format};
        """)
        if end_date is None:
            end_date = dt.date.today()  # start and end dates can be both strings and dt.date
        if filters is None:
            filters = []
        url_parameters = '('+' or '.join(
            (("(positionCaseInsensitive(URL,'{url}') > 0)").format(url=el)) for el in url)+')'

        where = counter_filter(counterid) + \
            [
            "StartDate >= toDate('{}')".format(start_date),
            "StartDate <= toDate('{}')".format(end_date),
            textwrap.dedent("""\
            arrayExists(
                x_0 -> x_0 IN
                    (
                        SELECT WatchID
                        FROM
                            merge.hits SAMPLE {sample}
                        WHERE
                                EventDate >= toDate('{start_date}')
                            and
                                EventDate <= toDate('{end_date}')
                            and
                                CounterID = {counter}
                            and
                                NOT DontCountHits
                            and
                                NOT Refresh
                            and
                                {url_parameters}
                    ),
                WatchIDs
            )
            """.format(
                sample=sample, start_date=start_date, end_date=end_date,
                counter=COUNTER_IDS.get(counterid, counterid),
                url_parameters=url_parameters
            ))
        ] + filters
        what = [
            'sum(Sign)*{} as visits'.format(1./sample),
            'uniq{}(UserID)*{} as users'.format('Exact' if exact else 'Combined', 1./sample),
            'uniq{}If(UserID, FirstVisit == StartTime)*{} as new_users'.format(
                'Exact' if exact else 'Combined', 1./sample
            )
        ]
        if groupby is None:
            groupby = ['startdate_day']
        sql = self._from_skeleton(
            query_skeleton, what, where, groupby, sample, testids=testids)

        return self.gen(sql, expiration)

    # def attendance_nowcast_desktop(self, *args, **kwargs):
    #     kwargs['filters'] = kwargs.get('filters', []) + [
    #         "arrayExists(x_0 -> x_0 = 27302314,`Goals.ID`)",   #desktop nowcast goal
    #     ]
    #     return self.attendance_weather_desktop(*args, **kwargs)

    # def attendance_nowcast_desktop_withnew(self, *args, **kwargs):
    #     kwargs['filters'] = kwargs.get('filters', []) + [
    #         "arrayExists(x_0 -> x_0 = 27302314,`Goals.ID`)",   #desktop_withnew nowcast goal
    #     ]
    #     return self.attendance_web_withnew("Pogoda desktop", *args, **kwargs)

    def attendance_nowcast_desktop(self, start_date, *args, **kwargs):
        return self.attendance_urlvisit('Pogoda desktop', '/nowcast', start_date, *args, **kwargs)

    def attendance_nowcast_desktop_withnew(self, start_date, *args, **kwargs):
        return self.attendance_urlvisit_withnew('Pogoda desktop', '/nowcast', start_date, *args, **kwargs)

    def attendance_details_desktop_withnew(self, start_date, *args, **kwargs):
        return self.attendance_urlvisit_withnew('Pogoda desktop', 'details', start_date, *args, **kwargs)

    def attendance_nowcastusers_desktop(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            textwrap.dedent("""\
            UserID GLOBAL IN
            (
            SELECT UserID
            from visits_all sample {sample}
            where
                    arrayExists(x_0 -> x_0 = 27302314,`Goals.ID`)
                and
                    StartDate >= toDate('{start_date}')
                and
                    StartDate <= toDate('{end_date}')
            )
            """).format(
                sample=kwargs.get('sample', 0.1),
                start_date=args[0] if len(args) > 0 else kwargs['start_date'],
                end_date=args[1] if len(args) > 1 else kwargs.get(
                    'end_date', dt.date.today())
            ),  # desktop nowcast goal
        ]
        return self.attendance_weather_desktop(*args, **kwargs)

    # def attendance_nowcast_touch(self, *args, **kwargs):
    #     kwargs['filters'] = kwargs.get('filters', []) + [
    #         "arrayExists(x_0 -> x_0 = 27302329,`Goals.ID`)",   #touch nowcast goal
    #         "StartURL not like '%nowcast=1%'",              #nowcast from apps
    #         r"StartURL not like '%appsearch_header=1%'"     #searchapp webview
    #     ]
    #     return self.attendance_web_withnew("Pogoda touch", *args, **kwargs)

    # def attendance_nowcast_touch_withnew(self, *args, **kwargs):
    #     kwargs['filters'] = kwargs.get('filters', []) + [
    #         "arrayExists(x_0 -> x_0 = 27302329,`Goals.ID`)",   #touch_withnew nowcast goal
    #         "StartURL not like '%nowcast=1%'",              #nowcast from apps
    #         r"StartURL not like '%appsearch_header=1%'"     #searchapp webview
    #     ]
    #     return self.attendance_web_withnew("Pogoda touch", *args, **kwargs)

    def attendance_nowcast_touch(self, start_date, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL not like '%nowcast=1%'",  # nowcast from apps
            r"StartURL not like '%appsearch_header=1%'"  # searchapp webview
        ]
        return self.attendance_urlvisit('Pogoda touch', '/nowcast', start_date, *args, **kwargs)

    def attendance_nowcast_touch_withnew(self, start_date, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL not like '%nowcast=1%'",  # nowcast from apps
            r"StartURL not like '%appsearch_header=1%'"  # searchapp webview
        ]
        return self.attendance_urlvisit_withnew('Pogoda touch', '/nowcast', start_date, *args, **kwargs)

    def attendance_details_touch_withnew(self, start_date, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL not like '%appsearch_header=1%'"  # searchapp webview
        ]
        return self.attendance_urlvisit_withnew('Pogoda touch', 'details', start_date, *args, **kwargs)

    def attendance_nowcast_apps(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL like '%nowcast=1%'",
        ]
        return self.attendance_web("Pogoda touch", *args, **kwargs)

    def attendance_nowcast_apps_withnew(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            "StartURL like '%nowcast=1%'",
        ]
        return self.attendance_web_withnew("Pogoda touch", *args, **kwargs)

    def attendance_detailsusers_desktop(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            textwrap.dedent("""\
            UserID GLOBAL IN
            (
            SELECT UserID
            from hits_all sample {sample}
            WHERE
                    EventDate >= toDate('{start_date}')
                and
                    EventDate <= toDate('{end_date}')
                and
                    CounterID = {counter}
                and
                    NOT DontCountHits
                and
                    NOT Refresh
                and
                    positionCaseInsensitive(URL,'details') > 0
            )
            """).format(
                sample=kwargs.get('sample', 0.1),
                start_date=args[0] if len(args) > 0 else kwargs['start_date'],
                end_date=args[1] if len(args) > 1 else kwargs.get(
                    'end_date', dt.date.today()),
                counter=COUNTER_IDS['Pogoda desktop']
            ),  # desktop nowcast goal
        ]
        return self.attendance_weather_desktop(*args, **kwargs)

    def attendance_nowcastusers_touch(self, *args, **kwargs):
        kwargs['filters'] = kwargs.get('filters', []) + [
            textwrap.dedent("""\
            UserID GLOBAL IN
            (
            SELECT UserID
            from visits_all sample {sample}
            where
                    arrayExists(x_0 -> x_0 = 27302329,`Goals.ID`)
                and
                    StartDate >= toDate('{start_date}')
                and
                    StartDate <= toDate('{end_date}')
            )
            """).format(
                sample=kwargs.get('sample', 0.1),
                start_date=args[0] if len(args) > 0 else kwargs['start_date'],
                end_date=args[1] if len(args) > 1 else kwargs.get(
                    'end_date', dt.date.today())
            ),  # desktop nowcast goal
        ]
        return self.attendance_weather_touch(*args, **kwargs)
