import httplib
import inspect
import sys

import requests

from sandbox import common, sdk2
from sandbox.projects import weather as weather_resources


def is_translations_resource(obj):
    is_translations_subclass = inspect.isclass(obj) and issubclass(obj, weather_resources.WEATHER_TRANSLATIONS)
    is_translations_baseclass = obj is weather_resources.WEATHER_TRANSLATIONS

    return is_translations_subclass and not is_translations_baseclass


def get_translations_resources():
    return inspect.getmembers(sys.modules[weather_resources.__name__], is_translations_resource)


def get_translations_resource_types():
    return {resource.__name__: resource for _, resource in get_translations_resources()}


RESOURCE_TYPES = get_translations_resource_types()


class WeatherCollectTranslations(sdk2.Task):
    """Weather collect translations from Tanker and upload to resource"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String('Type of resource', required=True, multiline=True) as resource_type:
            for type_name in RESOURCE_TYPES:
                resource_type.values[type_name] = resource_type.Value(type_name)

        redefine_parameters = sdk2.parameters.Bool('Redefine task parameters')
        with redefine_parameters.value[True]:
            tanker_base_url = sdk2.parameters.String('Tanker API Base URL', default='https://tanker-api.yandex-team.ru', required=True)
            tanker_project = sdk2.parameters.String('Tanker project', default='pogoda', required=True)
            tanker_branch = sdk2.parameters.String('Tanker branch', default='master', required=True)
            tanker_keysets = sdk2.parameters.List('Tanker keysets', default=[], required=True)
            tanker_format = sdk2.parameters.String('Tanker format', default='json', required=True)
            tanker_params = sdk2.parameters.List('Tanker parameters', default=['all-forms=1', 'safe=1'], required=True)

    def on_execute(self):
        Resource = RESOURCE_TYPES[self.Parameters.resource_type]

        options = self.Parameters if self.Parameters.redefine_parameters else Resource

        tanker_base_url = getattr(options, 'tanker_base_url').strip('/')
        tanker_project = getattr(options, 'tanker_project')
        tanker_branch = getattr(options, 'tanker_branch')
        tanker_keysets = getattr(options, 'tanker_keysets')
        tanker_format = getattr(options, 'tanker_format')
        tanker_params = dict([item.split('=') for item in getattr(options, 'tanker_params')])

        if not tanker_keysets:
            raise common.errors.TaskError('Nothing to do, empty keysets list')

        params = {'project-id': tanker_project, 'branch-id': tanker_branch, 'keyset-id': ','.join(tanker_keysets)}
        params = dict(list(params.items()) + list(tanker_params.items()))
        response = requests.get('{}/projects/export/{}/'.format(tanker_base_url, tanker_format), params=params)

        if response.status_code != httplib.OK:
            raise common.errors.TaskFailure('Tanker unsuccessful response: {}'.format(response.text))

        resource = sdk2.ResourceData(Resource(self, 'Translations JSON', 'translations.json', ttl='inf'))
        resource.path.write_bytes(response.content)
