import json
import logging
import requests
import socket
import time
import urlparse
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.weather import WEATHER_LOAD_AMMO
import sandbox.common.errors as ce


AMMO_PARAMS = {
    'alert': {
        'handle_name': 'alert',
        'handle_params': ['lat', 'lon'],
        'url_template': '/api/v3/nowcast/alert?from_client=load'
    },
    'forecast': {
        'handle_name': 'forecast',
        'handle_params': ['geoid', 'lat', 'lon'],
        'url_template': '/api/v3/forecast?from_client=load'
    },
    'new_tiles': {
        'handle_name': 'nowcast/new_encoded_tile',
        'handle_params': ['x', 'y', 'w', 'h'],
        'url_template': '/frontend/nowcast/new_encoded_tile?from_client=load'
    },
    'tiles': {
        'handle_name': 'nowcast/tile',
        'handle_params': ['x', 'y', 'z', 'encoded', 'scale'],
        'url_template': '/api/v3/nowcast/tile?from_client=load'
    },
    'warnings': {
        'handle_name': 'forecast',
        'handle_params': ['geoid', 'lat', 'lon'],
        'url_template': '/frontend/warnings?from_client=load'
    },
    'home_warnings': {
        'handle_name': 'forecast',
        'handle_params': ['geoid', 'lat', 'lon'],
        'url_template': '/home/warnings?from_client=load'
    },
}
DECIMALS_COUNT = {
    'lat': 2,
    'lon': 2,
    'x': 0,
    'y': 0,
    'w': 0,
    'h': 0,
    'z': 0,
    'geoid': 0,
    'encoded': 0,
    'scale': 0
}
FILE_NAME = 'ammo.txt'
ITS_SECTIONS = ['ah.weather.yandex.net', 'cloudapi.weather.yandex.net']
ISO_DATETIME_FORMAT_STRING = "%Y-%m-%dT%H:%M:%S"
LOAD_BALANCERS_ES_NAME = 'awacs-rtc_balancer_ah_weather-load_yandex_net_sas'
LOAD_DC = 'sas'
LOGS_BASE_PATH = '//logs/balancer-weather-prod-api-access-log/1d'
LOGS_CLUSTER = 'hahn'
PARENT_TASK = 'WEATHER-17681'
TANKS_ES_NAME = 'weather-tanks-sas'
TEN_MINUTES = 10 * 60
ABSENT_VALUE = -9999
LUNAPARK_URL = 'https://lunapark.yandex-team.ru/{}'


def is_dc_closed(token):
    its_url = 'https://its.yandex-team.ru/v2/l7/heavy/{}/weights/values/'
    auth_header = {'Authorization': 'OAuth {}'.format(token)}

    for its_section in ITS_SECTIONS:
        its_ah_response = requests.get(its_url.format(its_section), headers=auth_header).json()
        for location_name, location in its_ah_response['sections'].iteritems():
            if LOAD_DC.upper() in location['locations']:
                if location['locations'][LOAD_DC.upper()]['weight'] > 0:
                    return False

    return True


def resolve_endpoint_set(es):
    from infra.yp_service_discovery.python.resolver.resolver import Resolver
    from infra.yp_service_discovery.api import api_pb2

    resolver = Resolver(client_name='weather-load-testing:{}'.format(socket.gethostname()), timeout=5)
    resolve_request = api_pb2.TReqResolveEndpoints()
    resolve_request.cluster_name = LOAD_DC
    resolve_request.endpoint_set_id = es
    result = resolver.resolve_endpoints(resolve_request)
    return ['{}:{}'.format(endpoint.fqdn, endpoint.port) for endpoint in result.endpoint_set.endpoints]


def create_startrek_issue(parent_issue, issue_from_params, token):
    from startrek_client import Startrek
    from startrek_client import exceptions as st_exceptions

    startrek_client = Startrek(useragent='Weather load testing', token=token)
    if not issue_from_params:
        startrek_task = startrek_client.issues.create(
            queue='WEATHER',
            summary='Load testing from {}'.format(time.strftime("%Y-%m-%d %H:%M")),
            followers=['ftlka']
        )
        startrek_task_name = startrek_task.key
    else:
        startrek_task = startrek_client.issues[issue_from_params]
        startrek_task_name = issue_from_params
    try:
        startrek_task.links.create(issue=parent_issue, relationship='is subtask for')
    except st_exceptions.StartrekError as error:
        logging.error(error)

    return startrek_task_name


def get_handle_params_from_yt(yt_client, table_path, handle_params):
    params_list = list()
    for row in yt_client.read_table(yt_client.TablePath(
        table_path,
        columns=['request']
    )):
        requested_url = row['request']
        if any(param in requested_url for param in handle_params):
            parsed = urlparse.urlparse(requested_url)
            query_params = urlparse.parse_qs(parsed.query)
            rounded_query_params_list = list()
            for param in handle_params:
                if param not in query_params or query_params[param][0] == 'undefined':
                    rounded_query_params_list.append(ABSENT_VALUE)
                    continue
                val = query_params[param][0]
                if DECIMALS_COUNT.get(param, 0) != 0:
                    rounded_query_params_list.append(round(float(val), DECIMALS_COUNT[param]))
                else:
                    rounded_query_params_list.append(val)
            params_list.append(tuple(rounded_query_params_list))
    return params_list


def create_yql_request(cluster, table_name, key_word, limit):
    return """SELECT `request` FROM {cluster}.`{table_name}` WHERE String::Contains(`request`, "{key_word}") LIMIT {limit};""".format(
        cluster=cluster, table_name=table_name, key_word=key_word, limit=limit
    )


def get_logs_table_name(yt_client):
    return '/'.join([LOGS_BASE_PATH, sorted(yt_client.get(LOGS_BASE_PATH).keys())[-1]])


def get_for_dates():
    current_ts = int(time.time())
    approx_now = current_ts - (current_ts % TEN_MINUTES)
    return [approx_now + i * TEN_MINUTES for i in xrange(-10, 11)]


def get_gentimes(yt_client, table_path):
    gentimes = list()
    epoch = datetime.utcfromtimestamp(0)

    for archive_name in yt_client.get(table_path + "/@files").keys():
        gentime_str = archive_name.split('/')[-1]
        gentimes.append(int((datetime.strptime(gentime_str, ISO_DATETIME_FORMAT_STRING) - epoch).total_seconds()))
    return gentimes


class WeatherLoad(sdk2.Task):
    """Weather load testing"""

    class Requirements(sdk2.Task.Requirements):
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=3101565265)

    class Parameters(sdk2.Task.Parameters):
        ammo_url = sdk2.parameters.String('Url for file with ammo', default_value='https://proxy.sandbox.yandex-team.ru/2268538986')
        check_weights = sdk2.parameters.Bool('Check location weights', default=True)
        dsync_cluster = sdk2.parameters.String('Dsync cluster to get nowcast gentimes from', default='locke')
        dsync_table_path = sdk2.parameters.String('Path to dsync table', default='//home/meteo/production/dsync/nowcasting')
        fordates_amount = sdk2.parameters.Integer('Fordates amount', default=3)
        gentimes = sdk2.parameters.List('Gentimes')
        get_gentimes = sdk2.parameters.Bool('Parse locke table and get list of current gentimes', default=False)
        schedule = sdk2.parameters.String('Load function', default_value='line(1,3,1m)')
        startrek_task = sdk2.parameters.String('Startrek task', default_value='')

    def on_execute(self):
        robot_meteum_secrets = sdk2.yav.Secret('sec-01cqc28nkbabqnxf71me1p6xt1').data()
        api_key = sdk2.yav.Secret('sec-01f1m8ht7ev01p7t6g8tec3g92').data()['WEATHER_API_KEY']

        if self.Parameters.check_weights and not is_dc_closed(robot_meteum_secrets['NANNY_API_TOKEN']):
            raise ce.TaskError('Please set weights to zero in its for {}'.format(LOAD_DC))

        tanks = resolve_endpoint_set(TANKS_ES_NAME)
        for tank in tanks:
            url_with_status = 'http://{}/api/v1/tank/status.json'.format(tank)
            tank_status_response = requests.get(url_with_status).json()
            if tank_status_response['is_testing'] or tank_status_response['is_preparing']:
                raise ce.TaskError('Tank {} is already in progress'.format(tank))

        balancers = resolve_endpoint_set(LOAD_BALANCERS_ES_NAME)

        if not len(balancers):
            raise ce.TaskError('Did not resolve any balancers')

        startrek_task = create_startrek_issue(PARENT_TASK, self.Parameters.startrek_task, robot_meteum_secrets['STARTREK_API_TOKEN'])

        self.Context.running_loads = list()
        with open('ammo', 'w+') as f_ammo:
            for_dates = get_for_dates()[:self.Parameters.fordates_amount]
            raw_ammo = requests.get(self.Parameters.ammo_url)
            gentimes = list()
            if not self.Parameters.gentimes and self.Parameters.get_gentimes:
                from yt.wrapper import YtClient
                yt_token = robot_meteum_secrets['YQL_YT_TOKEN']
                yt_client = YtClient(proxy=self.Parameters.dsync_cluster, token=yt_token)
                gentimes = get_gentimes(yt_client, self.Parameters.dsync_table_path)

            if self.Parameters.gentimes:
                gentimes = self.Parameters.gentimes

            logging.info('gentimes: {}'.format(gentimes))
            logging.info('fordates: {}'.format(for_dates))

            for line in raw_ammo.iter_lines():
                for gentime in gentimes:
                    for for_date in for_dates:
                        f_ammo.write('{url}&nowcast_gen_time={gentime}&for_date={for_date}\n'.format(url=line, gentime=gentime, for_date=for_date))
                if not gentimes:
                    f_ammo.write('{url}\n'.format(url=line))
            f_ammo.seek(0)

            for tank, balancer in zip(tanks, balancers):
                with open('conf', 'w+') as f_conf:
                    conf = {
                        'phantom': {
                            'address': balancer,
                            'load_profile': {'load_type': 'rps', 'schedule': self.Parameters.schedule},
                            'ammofile': '',
                            'headers': ['X-Yandex-API-Key: {}'.format(api_key)],
                            'uris': []
                        },
                        'uploader': {
                            'enabled': True,
                            'operator': 'robot-meteum',
                            'job_dsc': '',
                            'job_name': '',
                            'task': startrek_task,
                            'package': 'yandextank.plugins.DataUploader',
                            'ver': '',
                        }
                    }
                    f_conf.write(json.dumps(conf))
                    f_conf.seek(0)
                    starting_tank_url = 'http://{}/api/v1/tests/start.json'.format(tank)
                    starting_tank_response = requests.post(starting_tank_url, files={'load.conf': f_conf, 'ammo': f_ammo})
                    shooting_id = json.loads(starting_tank_response.content)['id']
                    self.Context.running_loads.append({
                        'tank': tank,
                        'shooting_id': shooting_id
                    })
                    f_ammo.seek(0)

        grouped_lunapark_url = LUNAPARK_URL.format(startrek_task)
        self.set_info('See all lunapark jobs here <a href="{url}">{url}</a>'.format(url=grouped_lunapark_url), do_escape=False)

        while len(self.Context.running_loads):
            load = self.Context.running_loads.pop(0)
            shooting_status_response = requests.get('http://{}/api/v1/tests/{}/status.json'.format(load['tank'], load['shooting_id'])).text
            tank_info = json.loads(shooting_status_response)
            lunapark_id = tank_info['lunapark_id']
            if lunapark_id and 'lunapark_id' not in load:
                load['lunapark_id'] = lunapark_id
                lunapark_url = LUNAPARK_URL.format(load['lunapark_id'])
                self.set_info(
                    'Started: <a href="{url}">{url}</a>'.format(url=lunapark_url),
                    do_escape=False
                )
            if tank_info['status_code'] != 'FINISHED':
                self.Context.running_loads.append(load)
            else:
                if 'lunapark_id' not in load:
                    raise ce.TaskError(tank_info['tank_msg'])
                lunapark_url = LUNAPARK_URL.format(load['lunapark_id'])
                self.set_info(
                    'Finished: <a href="{url}">{url}</a>'.format(url=lunapark_url),
                    do_escape=False
                )
            time.sleep(1)

    def on_break(self, prev_status, status):
        self.set_info('Received on break signal')
        while self.Context.running_loads and len(self.Context.running_loads):
            load = self.Context.running_loads.pop(0)
            stop_res = requests.get('http://{}/api/v1/tests/stop.json'.format(load['tank'])).text
            stop_json = json.loads(stop_res)
            self.set_info(stop_json)
            if not stop_json['success']:
                self.set_info('Could not stop {}. Will try again'.format(load['tank']))
                self.Context.running_loads.append(load)


class WeatherLoadAmmoGenerator(sdk2.Task):
    """Generates ammo for load testing"""

    class Requirements(sdk2.Task.Requirements):
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=3101565265)

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup('Ammo type', required=True) as ammo_types:
            ammo_types.values.alert = ammo_types.Value('Nowcast alert', checked=True)
            ammo_types.values.forecast = ammo_types.Value('Forecast')
            ammo_types.values.home_warnings = ammo_types.Value('Home runtime warnings')
            ammo_types.values.new_tiles = ammo_types.Value('Nowcast new tiles')
            ammo_types.values.tiles = ammo_types.Value('Nowcast tiles')
            ammo_types.values.warnings = ammo_types.Value('Runtime warnings')

        urls_amount = sdk2.parameters.Integer('Urls amount', default_value=10000)
        table_name = sdk2.parameters.String('Hahn table path', default_value='')

    def on_execute(self):
        import pandas as pd
        from weather.libs.utils.yql import process_yql
        from yt.wrapper import YtClient

        robot_meteum_secrets = sdk2.yav.Secret('sec-01cqc28nkbabqnxf71me1p6xt1').data()
        yt_token = robot_meteum_secrets['YQL_YT_TOKEN']
        yt_client = YtClient(proxy=LOGS_CLUSTER, token=yt_token)
        table_name = self.Parameters.table_name
        if not table_name:
            table_name = get_logs_table_name(yt_client)

        self.set_info('Using table {}'.format(table_name))

        resulting_urls = list()
        for ammo_type in self.Parameters.ammo_types:
            ammo_params = AMMO_PARAMS[ammo_type]
            yql_request = create_yql_request(LOGS_CLUSTER, table_name, ammo_params['handle_name'], 1000000)
            _, results = process_yql(yql_request, cluster=LOGS_CLUSTER, token=yt_token)
            tmp_table_name = "//" + results.json['data'][0]['Write'][0]['Ref'][0]['Reference'][-1]  # if it fails, increase limit in yql

            raw_params = get_handle_params_from_yt(yt_client, tmp_table_name, ammo_params['handle_params'])
            df = pd.DataFrame(raw_params, columns=ammo_params['handle_params'])
            sampled_df = df.sample(self.Parameters.urls_amount)

            for _, row in sampled_df.iterrows():
                url = ammo_params['url_template']
                for handle_param in ammo_params['handle_params']:
                    if row[handle_param] != ABSENT_VALUE:
                        url += '&{}={}'.format(handle_param, row[handle_param])
                resulting_urls.append(url)

        resource = sdk2.ResourceData(WEATHER_LOAD_AMMO(self, 'Ammo file', FILE_NAME))
        resource.path.write_bytes('\n'.join(resulting_urls).encode('utf-8'))
        resource.ready()
