# -*- coding: utf-8 -*-

import imaplib
import logging
import socket
import datetime
import email
import email.header
import re

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

MAIL_SERVER = 'imap.yandex-team.ru'
MAIL_LOGIN = 'robot-meteum'
MAIL_SECRET = 'WEATHER-ADMIN', 'weather_mistakes_pass'
MAIL_FOLDER = 'Yandex|weather-mistakes'
MAIL_SEARCH_CRITERIONS = '(ON {} TO weather-mistakes@yandex-team.ru HEADER X-Login "")'
MAIL_FETCH_MESSAGE_PARTS = '(BODY[HEADER.FIELDS (SUBJECT)])'

STAT_SECRET = 'WEATHER-ADMIN', 'weather_mistakes_token'
STAT_REPORT_URL = 'pogoda.yandex/weather-mistakes/weather-mistakes'

IMAP_DATE_FORMAT = '%d-%b-%Y'
STAT_DATE_FORMAT = '%Y-%m-%d'

STATES = {
    'Температура': ('temperature', True),
    'Осадки': ('precipitation', True),
    'Облачность': ('cloudiness', True),
    'Ветер': ('wind', True),
    'Давление': ('pressure', True),
    'Другое': ('other', False)
}

# Because imaplib doesn't have timeout setting
socket.setdefaulttimeout(60)


class WeatherParseWeatherMistakes(sdk2.Task):
    """Parse weather-mistakes@ mailing list for mistakes types"""
    class Requirements(sdk2.Task.Requirements):
        environments = (PipEnvironment('python-statface-client', use_wheel=True),)

    class Parameters(sdk2.Task.Parameters):
        """Define task parameters"""
        kill_timeout = 1800
        max_restarts = 3
        use_stat_prod = sdk2.parameters.Bool('Use production statface')
        date = sdk2.parameters.String('Date (yyyy-mm-dd)', default='')

    @staticmethod
    def _dict2stat(date, data):
        stat_data = data.items()
        stat_data.append(('fielddate', date))

        return dict(stat_data)

    @staticmethod
    def _init_mail():
        mail_pass = sdk2.Vault.data(*MAIL_SECRET)
        mail = imaplib.IMAP4_SSL(MAIL_SERVER)
        mail.login(MAIL_LOGIN, mail_pass)

        return mail

    def get_data(self, date):
        """Parse subject of weather-mistakes@ to get mistakes types"""
        date = date.strftime(IMAP_DATE_FORMAT)
        states_counter = {value[0]: 0 for value in STATES.values()}
        pattern = re.compile('|'.join((key for key, value in STATES.items() if value[1])))
        other_key, = (value[0] for value in STATES.values() if not value[1])

        mail = self._init_mail()
        mail.select(MAIL_FOLDER)
        status, msg_ids = mail.uid('search', None, MAIL_SEARCH_CRITERIONS.format(date))
        msg_ids = msg_ids[0].split()

        for msg_id in msg_ids:
            status, data = mail.uid('fetch', msg_id, MAIL_FETCH_MESSAGE_PARTS)
            msg = email.message_from_string(data[0][1])
            msg_subj = email.header.decode_header(msg['Subject'])[0][0]
            match = pattern.match(msg_subj)
            key = other_key if not match else STATES[match.group()][0]
            states_counter[key] += 1

        mail.close()
        mail.logout()
        logging.info('Email parsing is finished.')

        return states_counter

    def push_data(self, date, state_counter):
        """Push data on specific date to statface"""
        import statface_client
        stat_token = sdk2.Vault.data(*STAT_SECRET)
        date = date.strftime(STAT_DATE_FORMAT)
        stat_data = self._dict2stat(date, state_counter)
        if self.Parameters.use_stat_prod:
            host = statface_client.STATFACE_PRODUCTION
        else:
            host = statface_client.STATFACE_BETA

        client = statface_client.StatfaceClient(host=host, oauth_token=stat_token)
        report = client.get_report(STAT_REPORT_URL)
        report.upload_data(scale='daily', data=stat_data)
        logging.info('Data is uploaded to statface.')

    def on_execute(self):
        """This method is called on task execution"""
        if self.Parameters.date:
            try:
                date = datetime.datetime.strptime(self.Parameters.date, STAT_DATE_FORMAT)
            except:
                logging.exception('Something wrong with date.')
                raise
        else:
            date = datetime.date.today() - datetime.timedelta(1)
        state_counter = self.get_data(date)
        self.push_data(date, state_counter)
