import logging
import textwrap

import sandbox.sandboxsdk.environments as environments
from sandbox import sdk2


def get_update_request():
    return textwrap.dedent("""\
        use hahn;
        pragma SimpleColumns;
        pragma yt.Pool = 'weather-analytics';
        $nice_date = ($dtstr) -> {RETURN CAST(CAST(CAST($dtstr As Uint32) AS Datetime) As Date)};
        $today = CurrentUtcDate();
        $period_start = CAST(CAST($today - Interval("P3DT") AS Date) AS String);

        $campaign_name = ($url) -> {
            RETURN CASE
                WHEN $url like '%campaign=pollencampaign%' then 'pollencampaign'
                WHEN $url REGEXP @@(?i)from=(ya-teaser-morda-oblako-sharik-ru|Ya-teaser-morda-fluger-raduga-ru|ya-teaser-morda-ptichki-ru|ya-teaser-morda-posmotriti-pogodu-ru|ya-teaser-morda-snegovik-ru|ya-teaser-morda-shar-sneg-ru|ya-teaser-morda-gde-teplee-ru|ya-teaser-morda-gde-jdat-sneg-ru|Ya-teaser-morda-fluger-petux-ru|ya-teaser-morda-kak-na-ladoni-ru|ya-widget-link|smartbanner-teploe-mesto-ru|smartbanner-kak-na-ladoni-ru|fullscreen-naglyadni-prognoz-ru|ya-teaser-morda-teploe-mestechko-ru|ya-teaser-morda-kak-na-ladoni-ru|ya-teaser-morda-gde-teplee-ru|ya-teaser-morda-otkuda-veter-ru|ya-teaser-morda-sneg-idet-ru|ya-teaser-morda-gde-jdat-sneg-ru|ya-teaser-morda-posmotriti-pogodu-ru)@@
                THEN 'mapscampaign'
                WHEN $url like '%from=home%' then "type-in"
                ELSE "no_campaign"
            END
            };

        insert into [home/weather-analytics/kalugin-di/trafgen-update]
        with truncate
        select counterid, CAST(visit_date AS string) as fielddate,  campaign, count(*) as hits, CountDistinctEstimate(m.uniqid) as users
        from
            (select $nice_date(eventtime) as visit_date, counterid, uniqid from
            RANGE([//home/logfeller/logs/bs-watch-log/1d], $period_start)
            where (counterid == "115080" or counterid == "345696") and url not like "goal:%" and $nice_date(eventtime) >= CAST($period_start as  Date)
            ) as m inner join [home/meteo/bushivan/trafgen-recent] as traf using (counterid, uniqid)
        group by
            m.counterid as counterid, m.visit_date as visit_date, $campaign_name(traf.trafgen_url) as campaign
        """)


def get_addendum_request():
    return textwrap.dedent("""\
        use hahn;
        pragma SimpleColumns;
        pragma yt.Pool = 'weather-analytics';

        $today = CurrentUtcDate();
        $period_start = CAST(CAST($today - Interval("P3DT") AS Date) AS String);

        insert into [home/weather-analytics/kalugin-di/trafgen-addendum]
        with truncate
        select logs_table.counterid as counterid,
                logs_table.uniqid as uniqid,
                logs_table.oldest_hit_time as oldest_hit_time,
                logs_table.trafgen_url as trafgen_url
        from (select
            counterid, uniqid,
            CAST(CAST(MIN(eventtime) As Int64) AS Datetime) as oldest_hit_time,
            MIN_BY(url, eventtime) as trafgen_url from
            RANGE([//home/logfeller/logs/bs-watch-log/1d], $period_start)
            where (counterid == "115080" or counterid == "345696") and url not like "goal:%"
            group by counterid, uniqid
            ) as logs_table left only join [home/meteo/bushivan/trafgen-recent] as trafgen_recent using (counterid, uniqid)
        order by counterid, uniqid
        """)


class WeatherTrafficGeneration(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('python-statface-client'),
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        YT_YQL_TOKEN = sdk2.parameters.String('Vault secret name with YQL OAuth token', default='yt_yql_secret')
        STATFACE_TOKEN = sdk2.parameters.String('Vault secret name with Statface OAuth token', default='statface_token')

    def on_execute(self):
        import yt.wrapper
        from yql.api.v1.client import YqlClient
        import statface_client

        logging.info("packages imported")
        # configurations
        logging.info("start configurating...")
        yt_yql_token = sdk2.Vault.data(self.Parameters.YT_YQL_TOKEN)
        statface_token = sdk2.Vault.data(self.Parameters.STATFACE_TOKEN)

        update_request = get_update_request()
        addendum_request = get_addendum_request()

        yt_new_table = yt_old_table = "//home/meteo/bushivan/trafgen-recent"
        yt_update_table = "//home/weather-analytics/kalugin-di/trafgen-addendum"

        statface_report = "pogoda.yandex/vitrina.new/trafgen"
        statface_table = "//home/weather-analytics/kalugin-di/trafgen-update"

        yt.wrapper.config['proxy']['url'] = 'hahn'
        yt.wrapper.config['token'] = yt_yql_token
        yql_client = YqlClient(token=yt_yql_token)
        statface_client = statface_client.StatfaceClient(oauth_token=statface_token,
            host=statface_client.STATFACE_PRODUCTION_UPLOAD)

        logging.info("configurations finished")
        # updating tables of new users and new traffic for last 3 days
        for sql_request, request_name in zip([addendum_request, update_request],
                                             ['addentum', 'update']):
            logging.info("{} request started".format(request_name))
            request = yql_client.query(sql_request)
            request.run()
            if not request.get_results().is_success:
                logging.info("{} request failed".format(request_name))
                return
            else:
                logging.info("{} requset succeed".format(request_name))

        # merging new users with existing users
        logging.info("start merging yt tables...")
        yt.wrapper.run_merge(
                source_table=[yt_old_table, yt_update_table],
                destination_table=yt_new_table,
                mode='sorted', job_count=150
               )
        logging.info("yt tables merged")
        # adding new traffic to statface report
        logging.info("start updating statface report...")
        report = statface_client.get_report(statface_report)
        report.upload_yt_data(scale='d', cluster='hahn', table_path=statface_table)
        logging.info("statface report is up to date")
