# -*- coding: utf-8 -*-

import itertools
import json
import logging
import time

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search.eventlog.eventlog import get_evlogdump
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
import sandbox.projects.common.file_utils as fu
from . import evlogstat


class EVENTLOG_SUBSOURCE_REQUESTS(sdk2.Resource):
    """
        https://st.yandex-team.ru/MIDDLE-103
    """
    any_arch = True


evlog_params = cmu.create_eventlog_resource_params()
res_params = cmu.create_resource_params()
stat_params = cmu.create_eventlog_stat_params()


class ProcessCount(parameters.SandboxIntegerParameter):
    name = 'process_count'
    description = 'Process count (0 - all available)'
    default_value = 0


class CompareSubSourceRequests(parameters.SandboxBoolParameter):
    """MIDDLE-103"""
    name = 'compare_subsource_requests'
    description = 'Extract and compare SubSource requests'
    default_value = False


class CalcEventlogStats(bugbanner.BugBannerTask):
    type = 'CALC_EVENTLOG_STATS'

    cores = 6
    required_ram = 64 * 1024  # 64 Gb
    execution_space = 100 * 1024  # 100 Gb (SEARCH-8188)
    input_parameters = evlog_params.params + (res_params.Evlogdump,) + stat_params.params + (ProcessCount, CompareSubSourceRequests)

    parsed_eventlog = None

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx["kill_timeout"] = 10800  # 3 hours
        json_res = self._create_resource(self.descr, 'stats.json', resource_types.EVENTLOG_JSON_STATS)
        self.ctx['json_stats'] = str(json_res.id)
        full_res = self._create_resource(self.descr, 'stats.full', resource_types.EVENTLOG_FULL_STATS)
        self.ctx['full_stats'] = full_res.id
        stat_dimensions = self._create_resource(self.descr, 'stats.dimensions', resource_types.PLAIN_TEXT)
        self.ctx['stat_dimensions'] = stat_dimensions.id

        if self.ctx.get(CompareSubSourceRequests.name, False):
            self.ctx['subsource_requests'] = self._create_resource(self.descr, 'subsource_requests.json', EVENTLOG_SUBSOURCE_REQUESTS).id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)
        self.parse_eventlog()
        self.calc_stats()
        self.save_first_requests_max_response_time()
        if self.ctx.get(CompareSubSourceRequests.name, False):
            self.save_subsource_requests_for_first_requests()

    def parse_eventlog(self):
        logging.info('parsing binary eventlog')

        if self.ctx.get(res_params.Evlogdump.name):
            evlogdump_binary = self.sync_resource(self.ctx[res_params.Evlogdump.name])
        else:
            evlogdump_binary = get_evlogdump()
        evlog = self.sync_resource(self.ctx[evlog_params.Eventlog.name])
        self.parsed_eventlog = self.abs_path('eventlog.parsed')
        with open(evlog) as in_file, open(self.parsed_eventlog, "w") as out_file:
            try:
                process.run_process([evlogdump_binary], stdin=in_file, stdout=out_file)
            except Exception as e:
                eh.log_exception("Unable to dump eventlog. Wait coredump", e)
                time.sleep(60)
                raise

    def calc_stats(self):
        logging.info('start calculating stats')

        stat_items, stat_counters, hist_opts = [_split_items(s) for s in [
            self.ctx[param_name] for param_name in [
                stat_params.TableFields.name,
                stat_params.AdditionalStatCounters.name,
                stat_params.HistogrammOptions.name
            ]
        ]]

        options = evlogstat.Options()

        more_stat = set()
        more_stat.update(stat_counters)

        for item in stat_items:
            stat_name = item.split('.')[1]
            if stat_name not in options.stat_names:
                more_stat.add(stat_name)

        logging.info('calculating stats')

        options.process_count = self.ctx[ProcessCount.name] or None
        logging.debug("process count is %s", options.process_count)
        options.rearr = True
        options.rearr_adjusters_stat = utils.get_or_default(self.ctx, stat_params.RearrangeAdjustersStat)
        options.stat_names.extend(list(more_stat))
        options.mode = utils.get_or_default(self.ctx, stat_params.CalcMode)
        if options.mode == "blender":
            options.by_grouping = utils.get_or_default(self.ctx, stat_params.ByGrouping)
            options.old_stages_order = utils.get_or_default(self.ctx, evlog_params.OldRearrangeStagesOrder)

        for hist_opt in hist_opts:
            try:
                value_name, step, upper_bound = hist_opt.strip().split("_")
                options.hist[value_name] = (int(step), int(upper_bound))
            except Exception:
                logging.info("can't parse hist option %s", hist_opt)

        stat_calc = evlogstat.StatCalc(options)

        with open(self.parsed_eventlog, 'r') as eventlog:
            stat_calc.parse_eventlog(eventlog)
            logging.info('making json report')

        json_res = channel.sandbox.get_resource(int(self.ctx['json_stats']))
        json_path = json_res.path
        if stat_items:
            stat_items_values = dict([(item, stat_calc.get_stat_value(item)) for item in stat_items])
        else:
            stat_items_values = stat_calc.get_full_stat()

        if not stat_items_values:
            eh.check_failed("No stats dumped!")

        fu.json_dump(json_path, stat_items_values)

        self.mark_resource_ready(json_res)

        logging.info('making full report')

        full_res = channel.sandbox.get_resource(self.ctx['full_stats'])
        full_path = full_res.path
        with open(full_path, 'w') as f:
            stat_calc.dump(f, sort_func=evlogstat.sort_result_by(""))

        self.mark_resource_ready(full_res)

        if 'stat_dimensions' in self.ctx:
            stat_dimensions = channel.sandbox.get_resource(self.ctx['stat_dimensions'])
            with open(stat_dimensions.path, 'w') as f:
                print >>f, json.dumps(stat_calc.get_dimensions())
            self.mark_resource_ready(stat_dimensions)

    def save_first_requests_max_response_time(self, number_of_requests=100):  # SEARCH-6879
        with open(self.parsed_eventlog) as log:
            req_time_iter = evlogstat.iter_log(log, [evlogstat.RequestTime()])
            max_response_time = 0
            max_response_num = 0
            for i, frame in enumerate(itertools.islice(req_time_iter, number_of_requests)):
                req_time = evlogstat.calc_frame(frame).get(("RequestTime", "microseconds"), [0])[0]
                if req_time > max_response_time:
                    max_response_time = req_time
                    max_response_num = i
            self.ctx['first_requests_max_response_time'] = max_response_time
            self.ctx['first_requests_max_response_num'] = max_response_num

    def save_subsource_requests_for_first_requests(self, number_of_requests=200):  # MIDDLE-103
        with open(self.parsed_eventlog) as log:
            iterator = evlogstat.iter_log(log, [evlogstat.SubSourceRequests()])
            subsource_requests = []
            for frame in itertools.islice(iterator, number_of_requests):
                subsource_requests_one_frame = evlogstat.calc_frame(frame).get('SubSourceRequests')
                if subsource_requests_one_frame is not None:
                    subsource_requests.append(subsource_requests_one_frame[0])

        resource = channel.sandbox.get_resource(self.ctx['subsource_requests'])
        fu.json_dump(resource.path, subsource_requests)


def _split_items(items_str):
    return [item.strip() for item in items_str.split(',')] if items_str else []


__Task__ = CalcEventlogStats
