# -*- coding: utf-8 -*-

import logging
import time
import json

from sandbox.projects.EvLogStatCalculator import resource_types as elsc_resource_types
from sandbox.projects import resource_types
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import util as sdkutil
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import apihelpers


stat_params = cmu.create_eventlog_stat_params()
res_params = cmu.create_resource_params()
evlog_params = cmu.create_eventlog_resource_params()


class CalcEvLogStatBinary(parameters.ResourceSelector):
    name = 'calc_evlog_stat_binary_resource_id'
    description = 'CalcEvLogStat'
    resource_type = elsc_resource_types.EVLOG_STAT_CALCUALTOR_EXECUTABLE
    required = False


class CalcEventlogStats2(bugbanner.BugBannerTask):

    type = 'CALC_EVENTLOG_STATS_2'

    cores = 1
    required_ram = 64 * 1024  # 64 Gb
    execution_space = 100 * 1024  # 100 Gb (SEARCH-8188)
    input_parameters = [
        stat_params.AdditionalStatCounters,
        stat_params.CalcMode,
        CalcEvLogStatBinary,
        evlog_params.Eventlog,
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx["kill_timeout"] = 10800  # 3 hours
        json_res = self._create_resource(self.descr, 'stats.json', resource_types.EVENTLOG_JSON_STATS)
        self.ctx['json_stats'] = json_res.id
        full_res = self._create_resource(self.descr, 'stats.full', resource_types.EVENTLOG_FULL_STATS)
        self.ctx['full_stats'] = full_res.id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)
        self.calc_evlog_stats()
        self.save_first_requests_max_response_time()

    def calc_evlog_stats(self):
        logging.info('start calculating stats')
        stat_items = self.ctx[stat_params.AdditionalStatCounters.name]
        if self.ctx.get(CalcEvLogStatBinary.name):
            self.evlog_stat_binary = self.sync_resource(self.ctx[CalcEvLogStatBinary.name])
        else:
            self.evlog_stat_binary = self._get_evlogstat_binary()
        self.evlog = self.sync_resource(self.ctx[evlog_params.Eventlog.name])
        json_res = channel.sandbox.get_resource(self.ctx['json_stats'])
        json_path = json_res.path
        full_res = channel.sandbox.get_resource(self.ctx['full_stats'])
        full_path = full_res.path
        with open(self.evlog) as in_file, open(full_path, 'w') as out_file:
            try:
                process.run_process([
                    self.evlog_stat_binary,
                    "--json-stat", json_path,
                    "--stat", stat_items
                ], stdin=in_file, stdout=out_file)
            except Exception as e:
                eh.log_exception("Unable to calc eventlog statistics. Wait coredump", e)
                time.sleep(60)
                raise

    def save_first_requests_max_response_time(self, number_of_requests=100):
        first_req_stat_path = self.abs_path('first_reqs.stat.json')
        with open(self.evlog) as in_file:
            try:
                process.run_process([
                    self.evlog_stat_binary,
                    "--json-stat", first_req_stat_path,
                    "--stat", "Max",
                    "--mode", "request",
                    "--limit-frame-count", repr(number_of_requests)
                ], stdin=in_file)
            except Exception as e:
                eh.log_exception("Unable to calc eventlog statistics. Wait coredump", e)
                time.sleep(60)
                raise
        with open(first_req_stat_path) as firs_req_stats:
            stats = json.loads(firs_req_stats.read())
            self.ctx['first_requests_max_response_time'] = stats['RequestTime.Max']

    def _get_evlogstat_resource(self, resource_id=None):
        evlogstat_resource = None
        if resource_id:
            logging.info('Try to find evlogstat for resource %s', resource_id)
            source_resource_object = channel.sandbox.get_resource(resource_id)
            evlogstat_resources = apihelpers.list_task_resources(
                source_resource_object.task_id, 'EVLOG_STAT_CALCUALTOR_EXECUTABLE'
            )
            for resource in evlogstat_resources:
                if sdkutil.is_arch_compatible(resource.arch):
                    evlogstat_resource = resource
        if not evlogstat_resource:
            logging.info('Try to get evlogstat from the last released evlogstat task')
            released_resource = apihelpers.get_last_released_resource(
                elsc_resource_types.EVLOG_STAT_CALCUALTOR_EXECUTABLE
            )
            if released_resource:
                logging.info("Got evlogstat resource #%s", released_resource.id)
                evlogstat_resource = channel.sandbox.get_resource(released_resource.id)
        return evlogstat_resource

    def _get_evlogstat_binary(self, resource_id=None):
        evlogstat_resource = self._get_evlogstat_resource(resource_id)
        if evlogstat_resource:
            return channel.task.sync_resource(evlogstat_resource)
        else:
            eh.fail('Cannot find correct EVLOG_STAT_CALCUALTOR_EXECUTABLE in the system.')


__Task__ = CalcEventlogStats2
