# -*- coding: utf-8 -*-

import os
import time

from sandbox import sdk2
from sandbox.sandboxsdk import network
from sandbox.projects.common import requests_wrapper
from sandbox.projects import resource_types
from sandbox.projects.common.search.config import NoapacheupperConfig
from sandbox.projects.websearch.CheckPrechargeAfterMemoryMap.component import Component
from sandbox.projects.websearch.params import ResourceWithDefaultByAttr
from sandbox.projects.websearch.upper import resources as upper_resources


class Noapache(Component):
    class Parameters(Component.Parameters):
        binary = sdk2.parameters.Resource("Noapache executable", resource_type=upper_resources.NoapacheUpper, required=True)
        config = ResourceWithDefaultByAttr('Noapache config', resource_type=resource_types.NOAPACHEUPPER_CONFIG,
                                           default_attr_name='autoupdate_resources_noapacheupper_web_kubr_trunk_task_id', required=True)
        noapache_data = sdk2.parameters.Resource("Noapache data (Rearrange Data + Rearrange Dynamic + Rearrange Data Fast)", resource_type=resource_types.NOAPACHEUPPER_DATA)
        rearrange_data = sdk2.parameters.Resource("Alternate Rearrange Data", resource_type=resource_types.REARRANGE_DATA)
        rearrange_dynamic = sdk2.parameters.Resource("Alternate Rearrange Dynamic", resource_type=resource_types.REARRANGE_DYNAMIC_DATA)
        rearrange_data_fast = sdk2.parameters.Resource("Alternate Rearrange Data Fast", resource_type=upper_resources.RearrangeDataFast)

    def __init__(self, parameters, port):
        self._binary = self._sync_resource(parameters.binary)
        self._config = self._sync_resource(parameters.config)
        if parameters.noapache_data is not None:
            noapache_data = self._sync_resource(parameters.noapache_data)
            self._rearrange_data_path = os.path.join(str(noapache_data.path), 'rearrange')
            self._rearrange_dynamic_path = os.path.join(str(noapache_data.path), 'rearrange.dynamic')
            self._rearrange_data_fast_path = os.path.join(str(noapache_data.path), 'rearrange.fast')
        if parameters.rearrange_data is not None:
            self._rearrange_data_path = str(self._sync_resource(parameters.rearrange_data).path)
        if parameters.rearrange_dynamic is not None:
            self._rearrange_dynamic_path = str(self._sync_resource(parameters.rearrange_dynamic).path)
        if parameters.rearrange_data_fast is not None:
            self._rearrange_data_fast_path = str(self._sync_resource(parameters.rearrange_data_fast).path)
        self._port = port

        config = NoapacheupperConfig.get_config_from_file(str(self._config.path))
        config.apply_local_patch({
            'Server/Port': self._port,
            'Server/AppHostOptions': 'Port={}, Threads=24, GrpcPort=+2, GrpcThreads=24'.format(self._port + 1),
            'PushAgentOptions': None,
            'Server/ScarabReqAnsLog': None,
            'Server/DirectScarabReqAnsLog': None,
            'Server/SiteSearchScarabReqAnsLog': None,
            'Server/ReqAnsLog': None,
            'Server/LoadLog': '/dev/null',
            'Server/EventLog': '/dev/null',
            'Collection/RearrangeDataDir:': self._rearrange_data_path,
            'Collection/RearrangeDynamicDataDir:': self._rearrange_dynamic_path,
            'Collection/RearrangeDataFastDir:': self._rearrange_data_fast_path,
        })
        self._patched_config_path = 'noapache.cfg.patched'
        config.save_to_file(self._patched_config_path)

    @staticmethod
    def name():
        return "noapache"

    def binary_path(self):
        return str(self._binary.path)

    def data_paths(self):
        return [self._rearrange_data_path, self._rearrange_dynamic_path, self._rearrange_data_fast_path]

    def execution_arguments(self):
        return ['-d', self._patched_config_path]

    def ready_for_first_request(self):
        return not network.is_port_free(self._port + 1)

    def send_request(self):
        resp = requests_wrapper.get('http://localhost:{port}/yandsearch?text=query'.format(port=self._port))
        time.sleep(20)
        return resp

    def stop(self):
        return requests_wrapper.get('http://localhost:{port}/admin?action=shutdown'.format(port=self._port))
