# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
import json
import logging

from sandbox.projects.common import error_handlers as eh


class CheckScraperProfile(sdk2.Task):
    """Checking that all profiles in arcadia/search/metrics_templates exist in arcadia/search/scraper/profile"""

    def on_execute(self):
        metrics_templates_path, profiles_path = self._sync_files()
        profiles_to_check = self._get_profiles_to_check(metrics_templates_path)
        existing_profiles = self._get_existing_profiles(profiles_path)
        eh.ensure(
            profiles_to_check <= existing_profiles,
            "Following profiles don't exist:\n{}".format('\n'.join(profiles_to_check - existing_profiles))
        )

    @staticmethod
    def _sync_files():
        metrics_templates_path = os.path.join(os.getcwd(), 'metric_templates')
        profiles_path = os.path.join(os.getcwd(), 'profiles')
        sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url('search/metrics_templates'), metrics_templates_path)
        sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url('search/scraper/profile'), profiles_path)
        return metrics_templates_path, profiles_path

    @staticmethod
    def _get_profiles_to_check(metrics_templates_path):
        """ return set of profiles """

        all_jsons = []
        for path, _, files in os.walk(metrics_templates_path):
            for file_name in files:
                if file_name.endswith('.json'):
                    file_path = os.path.join(path, file_name)
                    with open(file_path) as f:
                        all_jsons.append(json.load(f))
        profiles_to_check = recursive_search(all_jsons, 'profileId')
        logging.info('Profiles to check: \n{}'.format('\n'.join(profiles_to_check)))
        return profiles_to_check

    @staticmethod
    def _get_existing_profiles(profiles_path):
        """ return set of profiles """

        existing_profiles = set()
        for file_name in os.listdir(profiles_path):
            if file_name.endswith('.json'):
                existing_profiles.add(file_name[:-5])  # cut off '.json'
        logging.info('Existing profiles: \n{}'.format('\n'.join(existing_profiles)))
        return existing_profiles


def recursive_search(obj, key_to_look_for):
    """
    :param obj: json object (list or dict)
    :param key_to_look_for: key to look for in dict
    :return: set of values of the key
    """

    result = set()
    if isinstance(obj, dict):
        for key, value in obj.items():
            if key == key_to_look_for:
                result.add(value)
            else:
                result.update(recursive_search(value, key_to_look_for))
    elif isinstance(obj, list):
        for value in obj:
            result.update(recursive_search(value, key_to_look_for))
    return result
