# -*- coding: utf-8 -*-
import logging

from sandbox.common.types import client as sb_client

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.middlesearch import single_host
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.common.search import functional_helpers as fh
from sandbox.projects.common.search.eventlog import eventlog as el
from sandbox.projects.common import network
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.websearch.middlesearch import resources as ms_resources


class Subtype(sp.SandboxStringParameter):
    name = "search_subtype"
    description = "Search subtype"
    choices = [
        (fh.WEB, fh.WEB),
        (fh.IMAGES, fh.IMAGES),
        (fh.VIDEO, fh.VIDEO),
        (fh.QUICK, fh.QUICK),  # now tested same as web
    ]
    default_value = fh.WEB


class SearchType(sp.SandboxStringParameter):
    name = "search_type"
    description = "Search type"
    choices = [
        (fh.BASE, fh.BASE),
        (fh.MIDDLE, fh.MIDDLE),
    ]
    sub_fields = {
        fh.BASE: [p.name for p in sc.DefaultBasesearchParams.params],
        fh.MIDDLE: [p.name for p in sc.DefaultMiddlesearchParams.params]
    }
    default_value = fh.BASE


class TestPorts(sp.SandboxBoolParameter):
    name = "test_ports"
    description = "Test ports"
    default_value = True


class FunctionalTestSearch(task.SandboxTask):
    type = "FUNCTIONAL_TEST_SEARCH"

    execution_space = 50 * 1024  # 50 Gb (coredump for middle, started without models)

    required_ram = single_host.META_MIN_PHYSMEM_GB << 10

    input_parameters = (
        (
            Subtype,
            SearchType,
            TestPorts,
        ) +
        (
            fh.QueriesParameter,
            fh.QueriesLimitParameter,
        ) +
        (
            search_requester.Params.WorkersCount,
            search_requester.Params.RequestTimeout,
        ) +
        sc.DefaultMiddlesearchParams.params +
        sc.DefaultBasesearchParams.params
    )

    client_tags = sb_client.Tag.GENERIC & sb_client.Tag.Group.LINUX

    def initCtx(self):
        self.ctx['kill_timeout'] = 1 * 60 * 60  # 1 hours

    def on_execute(self):
        search_type = utils.get_or_default(self.ctx, SearchType)
        if search_type == fh.BASE:
            self.process_basesearch()
        elif search_type == fh.MIDDLE:
            self.process_middlesearch()
        else:
            eh.check_failed("Unknown search type: {}".format(search_type))

    def process_basesearch(self):
        basesearch = sc.get_basesearch()
        # disabled after switching to clang5 (should be enabled when we switch to musl)
        # self._check_libm(basesearch)
        info_requests = fh.INFO_REQUESTS_BASE.get(fh.WEB)

        fh.run_component_requesters(
            self,
            basesearch,
            info_requests=info_requests,
        )

        # crashers may produce errors
        crash_requests = fh.CRASH_REQUESTS_BASE.get(fh.WEB)
        fh.run_component_requesters(
            self,
            basesearch,
            info_requests=crash_requests,
            ignore_error=True,
            retry_limit=0
        )

        # testcase for SEARCH-1358
        basesearch.replace_config_parameter("Collection/IndexArchiveMode", "IndexOnly")
        fh.run_component_requesters(self, basesearch, info_requests)

    def process_middlesearch(self):
        self.test_ports()
        subtype = self.ctx[Subtype.name]
        middlesearch = sc.get_middlesearch()
        info_requests = fh.INFO_REQUESTS_MIDDLE.get(subtype)
        fh.run_component_requesters(self, middlesearch, info_requests)
        self.test_mmeta_models_requirement()

    def test_ports(self):
        """
        SEARCH-5727: Check that MessengerService is up and running
        """
        logging.info("Testing ports")

        if not utils.get_or_default(self.ctx, TestPorts):
            self.set_info("Port test is disabled")
            return

        unusual_port = network.get_free_port()
        middlesearch = sc.get_middlesearch(port=unusual_port, patch_port_in_cfg=False)

        if not middlesearch.messenger_enabled:
            self.set_info("Messenger is disabled, will skip UDP port test (SEARCH-5727)")
            return

        with middlesearch:
            p = process.run_process("netstat -an | grep :{}".format(unusual_port), shell=True, outs_to_pipe=True)
            out, err = p.communicate()
            for protocol in ["tcp", "udp", "tcp6", "udp6"]:
                logging.info("Testing %s ports", protocol)
                if protocol not in out:
                    logging.error("Netstat output follows:\n%s\n", out)
                    eh.check_failed("Port {} with protocol {} must be busy. ".format(unusual_port, protocol))
        logging.info("OK! TCP and UDP ports are busy!")

    def test_mmeta_models_requirement(self):
        middlesearch_resource_id = self.ctx[sc.DefaultMiddlesearchParams.Binary.name]
        middlesearch_type = channel.sandbox.get_resource(middlesearch_resource_id).type
        if middlesearch_type == ms_resources.RankingMiddlesearchExecutable:
            logging.info("Remove models.archive from input parameters")
            self.ctx[sc.DefaultMiddlesearchParams.ArchiveModel.name] = 0
            middlesearch = sc.get_middlesearch()
            try:
                with middlesearch:
                    eh.check_failed("FAIL! Ranking middlesearch started without models.archive!")
            except Exception:
                logging.info("OK! Ranking middlesearch fails without models.archive!")
        else:
            logging.info("Middlesearch is not ranking (type = %s), nothing to check!", middlesearch_type)

        evlogdump_path = el.get_evlogdump(middlesearch_resource_id)
        evlogdump_svnrevision = subprocess.check_output([evlogdump_path, "--svnrevision"], shell=False)
        # light check, without version parsing
        eh.ensure(
            'arcadia.yandex.ru/arc' in evlogdump_svnrevision,
            'Tool `evlogdump` should contain svn revision info',
        )

    @staticmethod
    def _check_libm(component):
        cmd = 'ldd "{}" | grep "libm.so"'.format(component.binary)
        return_code = process.run_process(cmd, shell=True, check=False).returncode
        if return_code == 0:
            eh.check_failed("Basesearch depends on external libm.so (math library)")
        elif return_code >= 2:
            eh.check_failed("'{}' failed with exit code {}".format(cmd, return_code))


__Task__ = FunctionalTestSearch
