# -*- coding: utf-8 -*-

import logging
import os
import os.path
import shutil
import json
import tarfile
import requests

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects import resource_types
from sandbox.projects.report import common as rc
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.common.types.resource import State
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.report.common import Project


__all__ = (
    "PerlReportBase"
)


class PerlReportBase(sdk2.Task):
    """
    Базовый класс, запускающий перловый репорт
    ВНИМАНИЕ: Этот класс не запускается в виде отдельной таски!
    """

    kill_port = '18881'

    folders = {
        'templates': u'report-templates',
        'config': u'conf',
        'config/report': u'conf/report',
        'report': u'report',
        'apache': u'apache_bundle',
        'runtime': u'data.runtime',
        'log': u'logs_report',
        'run': u'run',
        'rtcc': u'upper_conf',
        'sdch': u'sdch-dictionaries',
        '.' : ''
    }

    #class Requirements(sdk2.Requirements):
        #cores = 4   # exactly 4 core
        #ram = 8192  # 8GiB or less

    class Context(sdk2.Task.Context):
        pr1 = {
            "root_folder": "arkanavt.1",
        }


    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "LCX Container",
            resource_type = sdk2.service_resources.LxcContainer,
            platform = "linux_ubuntu_16.04_xenial",
            attrs = { "report": 3 },
            required = True
        )

        pr1_label = "justapache #1"

        with sdk2.parameters.Group("PerlReport: {label}".format(label = pr1_label)) as pr1_second_pr_group:
            with sdk2.parameters.RadioGroup("Source:") as pr1_source:
                pr1_source.values['svn'] = pr1_source.Value(default=True)
                pr1_source.values['sb_resource'] = None
                with pr1_source.value['svn']:
                    pr1_svn_url = sdk2.parameters.ArcadiaUrl(
                        "Arcadia URL for perl report:",
                        default_value = Arcadia.trunk_url() + '/web/report/@HEAD',
                    )
                with pr1_source.value['sb_resource']:
                    pr1_sb_resource = sdk2.parameters.Resource(
                        "Sandbox resource with perl report:",
                        resource_type = resource_types.REPORT_CORE_PACKAGE,
                        required = True,
                    )

            pr1_apache_bundle = sdk2.parameters.Resource(
                "Apache bundle",
                resource_type = resource_types.APACHE_BUNDLE,
                required = True,
            )
            pr1_data_runtime = sdk2.parameters.Resource(
                "DataRuntime",
                resource_type = resource_types.WEB_REPORT_DATA_RUNTIME_BUNDLE,
                required = True,
            )
            pr1_sdch_dictionary = sdk2.parameters.Resource(
                "SDCH dictionary",
                resource_type = rc.SdchWebDictionaryPack,
                required = True,
            )
            pr1_rtcc_bundle = sdk2.parameters.Resource(
                "Config for upper",
                resource_type = resource_types.RTCC_BUNDLE,
                required = True,
            )
            pr1_report_templates = sdk2.parameters.Resource(
                "Report templates",
                resource_type = resource_types.REPORT_TEMPLATES_PACKAGE,
                required = True,
            )

            pr1_port = sdk2.parameters.Integer(
                "PerlReport port",
                required = True,
                default = 8081
            )
            pr1_env = sdk2.parameters.String(
                "ENV param(key1=val1 key2=val2 ...)",
                required = True,
                default = "IS_BETA=1 MAX_CLIENTS=4 REPORT_INVERTED=1"
            )
            pr1_addition_flags_json = sdk2.parameters.JSON(
                'Addition PerlReport flags, JSON: ARRAY of OBJECTS e.g. [{"taxi_form":0}]',
                required = False,
                default_value = [{}]
            )
            pr1_request_json_url = sdk2.parameters.Url(
                'URL of alternative request.json e.g. http://apphost.priemka.yandex.ru/viewconfig?name=request.json',
                required = False,
                default_value = None
            )
            pr1_flags_json_url = sdk2.parameters.Url(
                'URL of alternative Experiments 100% flags.json e.g. https://ab.yandex-team.ru/deploying/flags.json/2/content',
                required = False,
                default_value = None
            )

    def on_create(self):
        logging.debug("PerlReportBase.on_create")
        self.Parameters.pr1_sb_resource = sdk2.Resource.find(
            resource_type = resource_types.REPORT_CORE_PACKAGE,
            attrs = { "released": "stable" },
            state=(State.READY)
        ).first()

        self.Parameters.pr1_apache_bundle = sdk2.Resource.find(
            resource_type = resource_types.APACHE_BUNDLE,
            attrs = { "released": "stable" },
            state=(State.READY)
        ).first()

        self.Parameters.pr1_data_runtime = sdk2.Resource.find(
            resource_type = resource_types.WEB_REPORT_DATA_RUNTIME_BUNDLE,
            attrs = { "released": "stable" },
            state=(State.READY)
        ).first()

        self.Parameters.pr1_sdch_dictionary = sdk2.Resource.find(
            resource_type = rc.SdchWebDictionaryPack,
            attrs = { "released": "stable" },
            state = (State.READY)
        ).first()

        self.Parameters.pr1_rtcc_bundle = sdk2.Resource.find(
            resource_type = resource_types.RTCC_BUNDLE,
            attrs = { "released": "stable" },
            state = (State.READY)
        ).first()

        self.Parameters.pr1_report_templates = sdk2.Resource.find(
            resource_type = resource_types.REPORT_TEMPLATES_PACKAGE,
            attrs = { "released": "stable" },
            state = (State.READY)
        ).first()

    def ctx1(self) :
        logging.debug('ctx1')
        return {
            "root_folder": self.Context.pr1['root_folder'],
            "label": self.Parameters.pr1_label,
            "pr_source": self.Parameters.pr1_source,
            "svn_url": self.Parameters.pr1_svn_url,
            "sb_resource": self.Parameters.pr1_sb_resource,
            "apache_bundle": self.Parameters.pr1_apache_bundle,
            "data_runtime": self.Parameters.pr1_data_runtime,
            "sdch_dictionary": self.Parameters.pr1_sdch_dictionary,
            "rtcc_bundle": self.Parameters.pr1_rtcc_bundle,
            "report_templates": self.Parameters.pr1_report_templates,
            "port": self.Parameters.pr1_port,
            "env": self.Parameters.pr1_env,
            "addition_flags_json": self.Parameters.pr1_addition_flags_json,
            "request_json_url": self.Parameters.pr1_request_json_url,
            "flags_json_url": self.Parameters.pr1_flags_json_url
        }

    def folder(self, *args):
        return str(self.path(*args))

    def folder_join(self, *args):
        return str(os.path.join(*args))

    def mkfolder(self, folder):
        logging.debug('MkFolder: {}'.format(folder))
        if os.path.exists(folder):
            shutil.rmtree(folder)
        logging.debug('MkFolder[mkdir]: {}'.format(folder))
        os.makedirs(folder, mode=0o775)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("chmod")) as pl:
            sp.Popen(
                "chmod g+s {}".format(folder),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

    def expand_resource(self, resource, to_folder) :
        resource_data = sdk2.ResourceData(resource)
        logging.debug('expand_resource [resorce_data.path]: {}'.format(str(resource_data.path)))

        nm = 0
        dst_folder = self.folder_join(
            str(to_folder),
            str(os.path.basename(str(resource_data.path)) + '-' + str(resource.id) + '-' + str(nm))
        )
        while os.path.exists(dst_folder) :
            dst_folder = self.folder_join(
                str(to_folder),
                str(os.path.basename(str(resource_data.path)) + '-' + str(resource.id) + '-' + str(nm))
            )
            nm = nm + 1
        logging.debug('expand_resource [dst_folder]: {}'.format(dst_folder))
        logging.debug('BASENAME: {}, PATH: {}'.format(os.path.basename(str(resource_data.path)),str(resource_data.path)))

        if os.path.isdir(str(resource_data.path)) :
            shutil.copytree(str(resource_data.path), dst_folder)
        elif str(resource_data.path).endswith('.tar.gz') or str(resource_data.path).endswith('.tar') :
            with tarfile.open(str(resource_data.path), 'r') as tar :
                self.mkfolder(dst_folder)
                tar.extractall(dst_folder)
        else :
            self.mkfolder(dst_folder)
            dst_file = self.folder_join(dst_folder, os.path.basename(str(resource_data.path)))
            logging.debug('COPY TO: {}'.format(dst_file))
            shutil.copyfile(
                str(resource_data.path),
                dst_file
            )

        sub_folder = None
        for fname in os.listdir(dst_folder) :
            if fname != '.' and fname != '..' :
                if os.path.isdir(self.folder_join(dst_folder, fname)) :
                    if sub_folder != None :
                        return dst_folder
                    sub_folder = fname
                else :
                    return dst_folder
        if sub_folder != None :
            return self.folder_join(dst_folder, sub_folder)
        return dst_folder

    @staticmethod
    def download_file(path, url):
        logging.info("Downloading %s", url)
        r = requests.get(url, timeout=120)
        eh.ensure(r.content, "Downloaded zero-sized file, seems to be an error. ")
        with open(path, 'w') as f:
            f.write(r.content)
        logging.info("Downloaded %s of %s bytes size, written to %s", url, len(r.content), path)

    def setup_rtcc_bundle(self, ctx, dest='upper_conf', link_prefix='conf', project='WEB', is_beta=1):
        rtcc_bundle_folder = self.expand_resource(
            ctx['rtcc_bundle'],
            self.folder(ctx['root_folder'], 'resources')
        )
        logging.debug('setup_rtcc_bundle[rtcc_bundle_folder]: {}'.format(rtcc_bundle_folder))

        project_config = Project.upper_config[project]
        if is_beta :
            rtcc_bundle_folder = self.folder_join(rtcc_bundle_folder, 'hamster', project_config, 'vla')
        else :
            rtcc_bundle_folder = self.folder_join(rtcc_bundle_folder, 'production', project_config, 'vla')
        logging.debug('setup_rtcc_bundle[rtcc_bundle_folder]: {}'.format(rtcc_bundle_folder))
        os.symlink(
            rtcc_bundle_folder,
            self.folder(ctx['root_folder'], self.folders['rtcc'])
        )
        request_json_path = self.folder(ctx['root_folder'], self.folders['config'], "request.json")
        if len(ctx['request_json_url']) > 0 :
            self.download_file(request_json_path, ctx['request_json_url'])
            logging.debug('setup_rtcc_bundle[request.json]: download to {}'.format(request_json_path))
        else :
            os.symlink(
                self.folder_join(rtcc_bundle_folder, 'request.json'),
                request_json_path
            )
            logging.debug('setup_rtcc_bundle[request.json]: symlink {} -> {}'.format(self.folder(ctx['root_folder'], self.folders['rtcc'], 'request.json'), request_json_path))

    def setup_apache_bundle(self, ctx):
        apache_bundle_folder = self.expand_resource(
            ctx['apache_bundle'],
            self.folder(ctx['root_folder'], 'resources')
        )
        logging.debug('setup_apache_bundle[apache_bundle_folder]: {}'.format(apache_bundle_folder))
        os.symlink(
            apache_bundle_folder,
            self.folder(ctx['root_folder'], self.folders['apache'])
        )

    def setup_data_runtime(self, ctx):
        data_runtime_folder = self.expand_resource(
            ctx['data_runtime'],
            self.folder(ctx['root_folder'], 'resources')
        )
        logging.debug('setup_data_runtime[data_runtime_folder]: {}'.format(data_runtime_folder))
        os.symlink(
            data_runtime_folder,
            self.folder(ctx['root_folder'], self.folders['runtime'])
        )

    def setup_sdch(self, ctx):
        resource_folder = self.expand_resource(
            ctx['sdch_dictionary'],
            self.folder(ctx['root_folder'], 'resources')
        )
        logging.debug('setup_sdch[resource_folder]: {}'.format(resource_folder))
        os.symlink(
            self.folder_join(resource_folder, 'sdch-config.json'),
            self.folder(ctx['root_folder'], self.folders['config'], 'sdch-config.json')
        )
        os.symlink(
            self.folder_join(resource_folder, 'apache_include'),
            self.folder(ctx['root_folder'], self.folders['config'], 'apache_include')
        )
        os.symlink(
            resource_folder,
            self.folder(ctx['root_folder'], self.folders['sdch'])
        )
        # Fixed sdch_dict.m4
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("fix_sdch")) as pl:
            sp.Popen(
                "sed -r -i 's|(SdchDict)\\s+([^ ]+)\\s+\"([^\" ]+)\"\\s+([^ ]+)|\\1 \\2 \\3 \\4|g' {}".format(
                    self.folder(ctx['root_folder'], self.folders['config'], 'apache_include', 'sdch_dict.m4'),
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()


    def setup_templates(self, ctx):
        templates_path = self.expand_resource(
            ctx['report_templates'],
            self.folder(ctx['root_folder'], 'resources')
        )
        logging.debug('setup_templates[templates_path]: {}'.format(templates_path))
        os.symlink(
            templates_path,
            self.folder(ctx['root_folder'], self.folders['templates'])
        )

    def setup_perl_report(self, ctx):
        report_folder = None
        if ctx['pr_source'] == 'svn' and ctx['svn_url'] != None:
            logging.debug('Report from SVN: {}'.format(ctx['svn_url']))
            report_folder = self.folder(ctx['root_folder'], 'resources/report_svn')
            sdk2.svn.Arcadia.export(ctx['svn_url'], report_folder)
            os.symlink(
                report_folder,
                self.folder(ctx['root_folder'], self.folders['report'])
            )
        elif ctx['pr_source'] == 'sb_resource' and ctx['sb_resource'] != None :
            logging.debug('Report from SB: {}'.format(ctx['sb_resource']))
            report_folder = self.expand_resource(
                ctx['sb_resource'],
                self.folder(ctx['root_folder'], 'resources')
            )
            logging.debug('setup_perl_report[report_path]: {}'.format(report_folder))
            os.symlink(
                report_folder,
                self.folder(ctx['root_folder'], self.folders['report'])
            )
        else :
            return False;

        flags_json_path = self.folder(ctx['root_folder'], report_folder, "data/flags/experiments/flags.json")
        logging.debug('flags_json_path: {}'.format(flags_json_path))
        if ctx['flags_json_url'] :
            os.remove(flags_json_path)
            self.download_file(flags_json_path, ctx['flags_json_url'])
            logging.debug('setup_perl_report[flags.json]: download to {}'.format(flags_json_path))

        # Fixed skip_profile_log
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("fix_skip_profile_log")) as pl:
            sp.Popen(
                "sed -r -i -e 's|skip_profile_log|skip_profile_log_off|g' {}".format(
                    self.folder(ctx['root_folder'], self.folders['report'], 'lib', 'YxWeb', 'Handler.pm'),
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()


    def configure_addition_flags_json(self, ctx) :
        if ctx['addition_flags_json'] != None :
            dst_folder = self.folder(
                ctx['root_folder'],
                str(self.folders['config/report'])
            )
            logging.debug('configure_addition_flags_json[dst_folder]: {}'.format(dst_folder))
            with open(self.folder_join(dst_folder, 'flags.json'), 'w') as outfile:
                    json.dump(ctx['addition_flags_json'], outfile)

    def configure_apache_bundle(self, ctx) :
        perl_conf = {
            'PATH': self.folder(ctx['root_folder'], self.folders['apache'], 'bin'),
            'NOSUDO': '1',
        }

        with open(self.folder(ctx['root_folder'], self.folders['apache'], '.perl.conf.1'), 'r') as f:
            for line in f.readlines():
                for fld in ('report', 'apache'):
                    path = self.folder(ctx['root_folder'], self.folders[fld])
                    substr = self.folders[fld]
                    for prefix in (':', ' '):
                        line = line.replace(prefix + substr, prefix + path)
                line = line.replace(' .', ' ' + self.folder(ctx['root_folder']))
                kv = line.rstrip().split(' ', 2)
                logging.debug('configure_apache_bundle[kv] {} = {}'.format(kv[0], kv[1]))
                if perl_conf.get(kv[0]) and kv[0] == 'PATH' :
                    perl_conf[kv[0]] = "{}:{}".format(kv[1], perl_conf[kv[0]])
                else :
                    perl_conf[kv[0]] = kv[1]


        for ekey in ('PATH', 'TEMP', 'LANG') :
            if perl_conf.get(ekey) and ekey == 'PATH' :
                perl_conf[ekey] = "{}:{}".format(os.environ[ekey], perl_conf[ekey])
            else :
                perl_conf[ekey] = os.environ[ekey]

        perl_conf['YX_PID_FILE'] = self.folder(ctx['root_folder'], self.folders['run'], 'httpd.pid')
        perl_conf['PID_FILE'] = self.folder(ctx['root_folder'], self.folders['run'], 'httpd.pid')
        perl_conf['LOCK_FILE'] = self.folder(ctx['root_folder'], self.folders['run'], 'accept.lock')
        perl_conf['YX_LOG_DIR'] = self.folder(ctx['root_folder'], self.folders['log'])
        perl_conf['LOG_DIR'] = self.folder(ctx['root_folder'], self.folders['log'])

        if len(ctx['env']) :
            for env in ctx['env'].split(' ') :
                kv = env.split('=', 2)
                if perl_conf.get(kv[0]) :
                    perl_conf[kv[0]] = "{}:{}".format(kv[1], perl_conf[kv[0]])
                else :
                    perl_conf[kv[0]] = kv[1]

        perl_conf['PORT1'] = str(ctx['port'])

        with open(self.folder(ctx['root_folder'], '.perl.conf'), 'w') as f:
            for k in perl_conf :
                f.write(k + ' ' + perl_conf[k] + "\n")

    def start_cmd(self, cmd) :
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("shoot")) as pl:
            return sp.Popen(
                cmd,
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )

    def configure_report(self, ctx) :
        # пути
        working_dir = self.folder(ctx['root_folder'])
        report_path = self.folder(ctx['root_folder'], self.folders['report'])
        apache_path = self.folder(ctx['root_folder'], self.folders['apache'])
        y_local_env_path = self.folder_join(apache_path, 'bin/y-local-env')
        build_path = self.folder_join(report_path, 'scripts/l10n/build.pl')
        upperconf_path = self.folder_join(report_path, 'scripts/dev/upperconf.pl')
        upperconf_log_path = self.folder_join(report_path, 'upperconf.log')

        # Change kill-port
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("fix_kill_port")) as pl:
            self.start_cmd(''' \
                sed -r -i \
                -e 's|our +\$KILL_PORT *= *18880;|our $KILL_PORT = {kill_port};|g' \
                {mod_yx_report} \
                '''.format(
                    kill_port = self.kill_port,
                    mod_yx_report = self.folder_join(report_path, 'lib/mod_yx_report.pm')
                )
            ).wait()

        # локализация
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("l10n_build")) as pl:
            sp.Popen(
                "cd {} && {} perl {} -p serp".format(report_path, y_local_env_path, build_path),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

        # Генерация httpd конфига
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("config_httpd")) as pl:
            sp.Popen(
                "cd {} && {} {} -config_httpd {}".format(
                    working_dir,
                    y_local_env_path,
                    upperconf_path,
                    working_dir
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

    def start_report(self, ctx) :
        # пути
        working_dir = self.folder(ctx['root_folder'])
        report_path = self.folder(ctx['root_folder'], self.folders['report'])
        apache_path = self.folder(ctx['root_folder'], self.folders['apache'])
        y_local_env_path = self.folder_join(apache_path, 'bin/y-local-env')
        upperconf_path = self.folder_join(report_path, 'scripts/dev/upperconf.pl')
        upperconf_log_path = self.folder_join(report_path, 'upperconf.log')

        # Start
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("start_httpd")) as pl:
            sp.Popen(
                "cd {} && {} {} -start {}".format(
                    report_path,
                    y_local_env_path,
                    upperconf_path,
                    working_dir
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

        # Ping
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("ping_httpd")) as pl:
            sp.Popen(
                    "cd {} && curl 'http://localhost:{}/search/v?json=1' --max-time 30 -H 'Host: yandex.ru' -H 'X-Yandex-HTTPS: yes' --max-redir 5 -v 2>&1  > {}/version.json".format(
                    working_dir,
                    str(ctx['port']),
                    working_dir
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()


    def setup_perl_report_to(self, ctx):
        for f in ['.', 'config', 'config/report', 'log', 'run']:
            self.mkfolder(self.folder(ctx['root_folder'], self.folders[f]))

        self.setup_sdch(ctx)
        self.setup_templates(ctx)
        self.setup_data_runtime(ctx)
        self.setup_rtcc_bundle(ctx)
        self.setup_perl_report(ctx)
        self.setup_apache_bundle(ctx)

        self.configure_addition_flags_json(ctx)
        self.configure_apache_bundle(ctx)
        self.configure_report(ctx)

        self.start_report(ctx)

    def cleanup(self):
        return

    def on_execute(self):
        # получить sdch надо до генерации конфига апача
        self.setup_perl_report_to(self.ctx1())
