# -*- coding: utf-8 -*-

import re
import logging
import json
import sandbox.projects.common.build.parameters as build_parameters
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.websearch.SearchIntegrationTests2 import SearchIntegrationTests2
from sandbox.projects import resource_types
from sandbox.common import errors
import sandbox.projects.common.constants as consts
from sandbox.projects.websearch.SearchIntegrationTestsWithRetries.resources import FlakySearchIntegration


class SearchIntegrationTests3(SearchIntegrationTests2):
    """
        запускает SearchIntegrationTests2 N раз. Каждый последующий раз запускает только тесты упавшие на предыдущем запуске
    """

    class Context(sdk2.Task.Context):
        # task_id: {count: number, path: {"some/path": [testname]}}
        bad_tasks = {}
        children = []

    class Parameters(SearchIntegrationTests2.Parameters):
        retry_task = sdk2.parameters.Integer(
            'Retry sandbox task N times with failed tests',
            required=True,
            default_value=0
        )

        threshold_failed_tests = sdk2.parameters.Integer(
            'Do not retry if failed tests more than',
            required=True,
            default_value=100
        )

        retry_all_tests = sdk2.parameters.Bool(
            'Retry all tests if failed tests more than threshold_failed_tests',
            default_value=False
        )

    def _abort_soy_batch(self):
        pass

    def _create_child_task(self, retries, bad_tests):
        sub_params = dict(self.Parameters)
        if retries is not None:
            message = "Retry: {}.\nRerun {} tests.".format(retries, "all" if bad_tests is None else len(bad_tests))
            sub_params["description"] = self.Parameters.description + "\n{}".format(message)
        # не делаем уведомления для дочерних задач
        sub_params["notifications"] = []
        # do not restart sub task
        sub_params["max_restarts"] = 0

        sub_params[build_parameters.FailedTestsCauseError.name] = True
        if bad_tests is not None:
            # удалить test_filters
            if consts.TEST_FILTERS in sub_params:
                sub_params.pop(consts.TEST_FILTERS)
            # из extra_parameters удалить -F и test_filter
            if consts.YA_MAKE_EXTRA_PARAMETERS in sub_params:
                for i in range(len(sub_params[consts.YA_MAKE_EXTRA_PARAMETERS])-1, -1, -1):
                    if sub_params[consts.YA_MAKE_EXTRA_PARAMETERS][i].startswith("-F") or sub_params[consts.YA_MAKE_EXTRA_PARAMETERS][i].startswith("--test-filter"):
                        sub_params[consts.YA_MAKE_EXTRA_PARAMETERS].pop(i)
            # запустить только упавшие тесты
            for name in bad_tests:
                sub_params[consts.YA_MAKE_EXTRA_PARAMETERS].append("--test-filter={}".format(name))

        subtask = SearchIntegrationTests2(self, **sub_params)
        return subtask

    def run_child_and_wait(self, retries=None, bad_tests=None):
        subtask = self._create_child_task(retries, bad_tests)
        # Save ids to context to process them after waking up
        self.Context.children.append(subtask.id)
        subtask.enqueue()
        raise sdk2.WaitTask(subtask, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def get_test_stat(self, task_id):
        res = sdk2.Resource.find(
            type=resource_types.TEST_ENVIRONMENT_JSON_V2,
            task_id=task_id
        ).first()
        logging.info('In task: {} find resourse: {}'.format(task_id, res))
        if not res:
            raise errors.TaskFailure("Could not find resource in task: {}".format(task_id))

        data = sdk2.ResourceData(res)
        json_data = None
        with open(str(data.path)) as fh:
            json_data = json.load(fh)
            logging.info("Loaded JSON from %s", data.path)
        return json_data

    def find_bad_tests(self, task_id):
        json_data = self.get_test_stat(task_id)

        failed_tests = {
            "count": 0,
            "path": {}
        }
        for item in json_data["results"]:
            if item["type"] == "test" and "name" in item and "status" in item:
                if item["name"] == "py3test":
                    logging.info("Summary for path {}: {}".format(item["path"], item["rich-snippet"]))
                    logging.info("metrics for path {}: {}".format(item["path"], item["metrics"]))
                    # "failed_tests" "ok_tests" "xfailed_tests" " test_count" "xpassed_tests" skipped_tests"
                elif item["status"] == "FAILED":
                    failed_tests["count"] += 1
                    if failed_tests["count"] < self.Parameters.threshold_failed_tests:
                        if item["path"] not in failed_tests["path"]:
                            failed_tests["path"][item["path"]] = []
                        failed_tests["path"][item["path"]].append({
                            "name": item["name"],
                            "subtest_name": item["subtest_name"],
                            "log": item["links"]["log"][0]
                        })

        logging.info("Total failed tests: {}".format(failed_tests["count"]))

        return failed_tests

    def get_failed_tests_from_context(self, task_id):
        task_id = str(task_id)
        if task_id not in self.Context.bad_tasks:
            # после сохраниения в контексте, ключи становятся строкой, поэтому делаем все строками
            self.Context.bad_tasks[task_id] = self.find_bad_tests(task_id)

        return self.Context.bad_tasks[task_id]

    def on_execute(self):
        with self.memoize_stage.first_run:
            self.run_child_and_wait()

        with self.memoize_stage.create_children(self.Parameters.retry_task) as st:
            if len(self.Context.children) == 0:
                raise errors.TaskFailure("Can not find children")

            task_id = self.Context.children[len(self.Context.children)-1]
            failed_from_context = self.get_failed_tests_from_context(task_id)

            if self.Parameters.retry_task != 0:
                logging.info("Retry: {}".format(st.runs))
                if failed_from_context["count"] > self.Parameters.threshold_failed_tests:
                    logging.error("failed tests: {} more than {}.".format(failed_from_context["count"], self.Parameters.threshold_failed_tests))
                    if self.Parameters.retry_all_tests:
                        logging.info("Rerun all tests")
                        self.run_child_and_wait(st.runs)
                elif failed_from_context["count"] > 0:
                    # retry failed tests
                    failed = []
                    for (k, v) in failed_from_context["path"].iteritems():
                        for item in v:
                            failed.append("::".join((item["name"], item["subtest_name"])))
                    self.run_child_and_wait(st.runs, failed)

        with self.memoize_stage.final:
            last_task_id = self.Context.children[len(self.Context.children)-1]
            # нужно создать ресурс TEST_ENVIRONMENT_JSON_V2, чтобы testenv показал статистику
            # берем ресурс и оставляем в нем только те упавшие тесты, которые есть в последнем запуске
            # ресурс надо взять из первого таска у которого количество упавших меньше threshold_failed_tests
            # если таких нет, то берем последний
            first_task_id = None
            for task_id in self.Context.children:
                first_task_id = task_id
                if self.get_failed_tests_from_context(task_id)["count"] <= self.Parameters.threshold_failed_tests:
                    break
            flaky = {}
            json_data = self.get_test_stat(first_task_id)
            if len(self.Context.children) > 1 and first_task_id != last_task_id:
                failed_names = {}
                last_fail = self.get_failed_tests_from_context(last_task_id)
                for (path, v) in last_fail["path"].iteritems():
                    for item in v:
                        failed_names["::".join((path, item["name"], item["subtest_name"]))] = 1

                delta_good = 0
                # тут храним элементы со сводкой (name: py3test)
                total_stat = {}
                for item in json_data["results"]:
                    if item["type"] == "test" and "name" in item and "status" in item and item["status"] == "FAILED":
                        if item["name"] != "py3test":
                            name = "::".join((item["path"], item["name"], item["subtest_name"]))
                            if name not in failed_names:
                                delta_good += 1
                                item["status"] = "OK"
                                logging.info("update to OK path: {} name: {}::{}".format(item["path"], item["name"], item["subtest_name"]))
                                if item["path"] not in flaky:
                                    flaky[item["path"]] = []
                                flaky[item["path"]].append("::".join((item["name"], item["subtest_name"])))
                        else:
                            total_stat[item["path"]] = item

                # обновить количество упавших тестов
                for (path, item) in total_stat.iteritems():
                    m = re.search(r"\[\[bad\]\](\d+)", item["rich-snippet"])
                    first_bad = int(m.group(1))
                    m = re.search(r"\[\[good\]\](\d+)", item["rich-snippet"])
                    first_good = 0
                    if m:
                        first_good = int(m.group(1))
                        logging.info(
                            "Before update rich-snippet: {}. metrics.ok_tests:{}. metrics.failed_tests: {}".format(
                                item["rich-snippet"], item["metrics"]["ok_tests"], item["metrics"]["failed_tests"]
                        ))
                        item["rich-snippet"] = re.sub(r"\[\[good\]\]\d+", "[[good]]{}".format(first_good + delta_good), item["rich-snippet"])
                        item["rich-snippet"] = re.sub(r"\[\[bad\]\]\d+", "[[bad]]{}".format(first_bad - delta_good), item["rich-snippet"])
                        item["metrics"]["ok_tests"] += delta_good
                        item["metrics"]["failed_tests"] -= delta_good
                        if first_bad - delta_good == 0:
                            item["status"] = "OK"

                        logging.info("After update rich-snippet: {}. metrics.ok_tests:{}. metrics.failed_tests: {}".format(
                            item["rich-snippet"], item["metrics"]["ok_tests"], item["metrics"]["failed_tests"]
                        ))

            if flaky:
                resource = FlakySearchIntegration(self, self.Parameters.description, "flaky.json")
                if self.Parameters.tags:
                    if "TESTENV-AUXILIARY-CHECK" in self.Parameters.tags:
                        resource.manual_run = True

                        # "TESTENV-JOB-_META_TEST__BEGEMOT_REQUEST_INIT__SEARCH_INTEGRATION_TEST"
                        # "TESTENV-JOB-_META_TEST__BEGEMOT_REQUEST_INIT__SEARCH_INTEGRATION_TEST_SOY_HTTP"
                        # "TESTENV-JOB-_META_TEST__WEB_GRAPH_MAPPING__SEARCH_INTEGRATION_TEST"
                        # "TESTENV-JOB-_META_TEST__WEB_GRAPH_MAPPING__SEARCH_INTEGRATION_TEST"
                        # "TESTENV-JOB-_META_TEST__REPORT__SEARCH_INTEGRATION_TEST"
                        # "TESTENV-JOB-_META_TEST__REPORT__SEARCH_INTEGRATION_TEST_SOY_HTTP"
                        for item in self.Parameters.tags:
                            if item.startswith("TESTENV-JOB-_META_TEST__"):
                                start = len("TESTENV-JOB-_META_TEST__")
                                end = item.find("__", start)
                                if end != -1:
                                    resource.component = item[start:end]
                                break
                    else:
                        # "TESTENV-DATABASE-BEGEMOT_REQUEST_INIT-85",
                        # "TESTENV-DATABASE-WEB_GRAPH_MAPPING-301"
                        # "TESTENV-DATABASE-WS-REPORT-433"
                        for item in self.Parameters.tags:
                            if item.startswith("TESTENV-DATABASE-"):
                                resource.component = item[len("TESTENV-DATABASE-"):]
                                break

                resource.path.write_bytes(json.dumps(flaky))

            # copy from https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/build/YaMake/__init__.py?rev=8268697&blame=true#L771
            resource = sdk2.Resource["TEST_ENVIRONMENT_JSON_V2"](self, "results.json", "TestEnv Json")
            resource.path.write_bytes(json.dumps(json_data))
