import os
import requests
import logging
import json
import time
from sandbox import sdk2
from sandbox.projects.release_machine.helpers import wiki_helper
from sandbox.projects.release_machine.core import const
from sandbox.projects import resource_types
from sandbox.projects.websearch.upper import resources as upper_resources


class UpdateWikiMetricsDescriptions(sdk2.Task):
    """Updates metrics descriptions"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Component") as component:
            component.values['src_setup'] = component.Value('src_setup', default=True)
            component.values['noapache'] = component.Value('noapache')
        resources = dict(
            src_setup=(resource_types.APP_HOST_SRC_SETUP_BUNDLE_WEB, resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_WEB),
            noapache=(upper_resources.NoapacheUpper, resource_types.NOAPACHEUPPER_CONFIG)
        )
        with component.value['src_setup']:
            src_setup_binary = sdk2.parameters.Resource("Src_setup exeutable", required=True, resource_type=resources['src_setup'][0])
            src_setup_config = sdk2.parameters.Resource("Src_setup config", required=True, resource_type=resources['src_setup'][1])
        with component.value['noapache']:
            noapache_binary = sdk2.parameters.Resource("Noapache exeutable", required=True, resource_type=resources['noapache'][0])
            noapache_config = sdk2.parameters.Resource("Noapache config", required=True, resource_type=resources['noapache'][1])
        handle = sdk2.parameters.String("Command", required=True, default="_metrics_descriptions")
        wiki_page = sdk2.parameters.String("Wiki page", required=True, default="users/themanofsteel/src_setup_metrics_descriptions")

    def unzip_file(self, path):
        with sdk2.helpers.ProcessLog(self, logger='unzip') as pl:
            sdk2.helpers.subprocess.check_call(['tar', 'xvzf', path], stdout=pl.stdout, stderr=pl.stderr)
        return os.path.basename(path).split('.')[0]

    def update_config(self, path):
        config_data = ''

        with open(path, 'r') as file:
            for line in file:
                if not ('LoadLog' in line or 'EventLog' in line):
                    config_data += line

        path = 'noapacheupper.cfg'
        with open(path, 'w') as f:
            f.write(config_data)

        return path

    def get_item_from_resource(self, resource):
        return str(sdk2.ResourceData(resource).path)

    def get_resources(self):
        tool_path = self.get_item_from_resource(getattr(self.Parameters, self.Parameters.component + '_binary'))
        config_path = self.get_item_from_resource(getattr(self.Parameters, self.Parameters.component + '_config'))
        if self.Parameters.component == 'src_setup':
            tool_path = '{}/{}'.format(self.unzip_file(tool_path), '/src_setup')
            config_path = '{}/{}'.format(self.unzip_file(config_path), '/MAN_WEB_AH_SRC_SETUP_HAMSTER/config.json')
        if self.Parameters.component == 'noapache':
            config_path = self.update_config(config_path)
        return tool_path, config_path

    def get_descriptions(self, binary, config):
        tool_port = '22222'
        tool_url = 'localhost:' + tool_port
        if self.Parameters.component == 'src_setup':
            cmd = [binary, '-p', tool_port, '-c', config, '--no-mlock']
        elif self.Parameters.component == 'noapache':
            cmd = [binary, '-p', tool_port, config, '-d']
        with sdk2.helpers.ProcessLog(self, logger='tool') as pl:
            sdk2.helpers.subprocess.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr)
            while sdk2.helpers.subprocess.call(['curl', '{}/admin?action=ping'.format(tool_url)]):
                logging.info('wait')
                time.sleep(1)
            descriptions = sdk2.helpers.subprocess.check_output(['curl', '{}/{}'.format(tool_url, self.Parameters.handle)])
            logging.info(descriptions)
            sdk2.helpers.subprocess.call(['curl', '{}/admin?action=shutdown'.format(tool_url)])
        return json.loads(descriptions)

    def update_wiki(self, descriptions):
        session = requests.Session()
        token = sdk2.Vault.data(self.owner, 'WIKI_TOKEN')
        session.headers['Authorization'] = 'OAuth {}'.format(token)
        session.verify = False

        page_url = const.Urls.WIKI_API + self.Parameters.wiki_page
        headers = ['Name', 'Description']
        rows = sorted(map(lambda x: ('%%{}%%'.format(x[0]), x[1]), descriptions.items()), key=lambda x: x[0])
        params = {
            'title': 'Metrics descriptions',
            'body': wiki_helper.format_table(headers, rows)
        }

        wiki_helper.post_on_wiki_and_check(session, page_url, params)

    def on_execute(self):
        binary, config = self.get_resources()
        descriptions = self.get_descriptions(binary, config)
        self.update_wiki(descriptions)
