import logging
import re

from sandbox.common.types import task as ctt
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.constants import constants
from sandbox.projects.common.sdk_compat import task_helper as th
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import startrek_helper as rm_st
from sandbox.projects.release_machine import security as rm_sec


LOGGER = logging.getLogger(__name__)


def notify_st_on_acceptance(task, rm_component, new_build_task_id, old_build_task_id, rev, component_extra=None, use_grouped_comment=False):
    try:
        comment_st(
            task, rm_component,
            new_build_task_id=new_build_task_id,
            old_build_task_id=old_build_task_id,
            rev=rev,
            component_extra=component_extra,
            use_grouped_comment=use_grouped_comment
        )
    except Exception as e:
        eh.log_exception("Unable to write startrek comment", e)
        task.set_info("Unable to write startrek comment! See logs for details")


def comment_st(acceptance_task, c_info, new_build_task_id, old_build_task_id=None, rev=None, component_extra=None, use_grouped_comment=False):
    phases_handler = AcceptancePhases(acceptance_task)
    if phases_handler.is_notified:
        LOGGER.info("Phase %s is already notified in ticket", phases_handler.phase)
        return
    st_helper = rm_st.STHelper(token=rm_sec.get_rm_token(acceptance_task))
    issue = st_helper.find_ticket_by_release_number(
        th.input_or_ctx_field(acceptance_task, rm_const.RELEASE_NUMBER), c_info, fail=False
    )
    if not issue:
        return

    tag_regexp = re.compile(r'.*/arc/(.*)/arcadia.*')
    new_arc_url = th.ctx_field(th.task_obj(new_build_task_id), constants.ARCADIA_URL_KEY)

    if old_build_task_id:
        old_arc_url = th.ctx_field(th.task_obj(old_build_task_id), constants.ARCADIA_URL_KEY, "")
        old_tag_finder = tag_regexp.findall(old_arc_url)
        old_tag = old_tag_finder[0] if old_tag_finder else "unknown tag"
    else:
        old_tag = "production"

    text="{component} {link} //{exec_phase}//: **{new_tag} {rev}** !!vs!! **{old_tag}**\n{t_data}".format(
        component=c_info.name + (':' + component_extra if component_extra else ''),
        link=lb.task_wiki_link(acceptance_task.id, "acceptance task"),
        exec_phase=phases_handler.phase,
        new_tag=tag_regexp.findall(new_arc_url)[0],
        rev=lb.revision_link(rev, link_type=lb.LinkType.wiki) if rev else "%%unknown rev%%",
        old_tag=old_tag,
        t_data=get_subtask_table(acceptance_task),
    )
    if use_grouped_comment:
        st_helper.write_grouped_comment_in_issue(
            issue,
            group_name=rm_const.TicketGroups.PerfTest,
            title=None,
            content=text,
            notify=c_info.notify_cfg__st__notify_on_robot_comments_to_tickets,
        )
    else:
        st_helper.create_comment(
            issue,
            text=text,
            notify=c_info.notify_cfg__st__notify_on_robot_comments_to_tickets,
        )
    phases_handler.mark_phase_as_notified()
    if not th.input_or_ctx_field(acceptance_task, "st_issue_key"):
        acceptance_task.set_info("Acceptance ticket is {}".format(lb.st_link(issue.key)), do_escape=False)
        th.ctx_field_set(acceptance_task, "st_issue_key", issue.key)
    th.ctx_field_set(acceptance_task, "st_comment_id", list(issue.comments.get_all())[-1].id)


def get_subtask_table(task):
    subtasks_info = utils.subtasks_id_and_status(task.id)
    LOGGER.info("Check statuses for tasks: %s", subtasks_info)
    failed_subtasks = [th.task_obj(i['id']) for i in subtasks_info if i["status"] == ctt.Status.FAILURE]
    if failed_subtasks:
        return "#|\n|| Failed subtask | Binary Type | Reason ||\n{t_body}\n|#".format(
            t_body='\n'.join([
                "|| {link} | {binary_type} | {reason} ||".format(
                    link=lb.task_wiki_link(subtask.id),
                    binary_type=th.input_or_ctx_field(subtask, "binary_type", "None"),
                    reason=th.input_or_ctx_field(subtask, "explanation", ""),
                )
                for subtask in failed_subtasks
            ])
        )
    return ""


class AcceptancePhases(object):
    STARTED = "STARTED"
    DONE = "DONE"
    UNKNOWN = "UNKNOWN"  # exceptions in children are here
    PHASE_CTX_KEY = "notified_phases"

    def __init__(self, acceptance_task):
        self.acceptance_task = acceptance_task
        self.phase = self._get_phase()
        self.is_notified = self.phase in th.input_or_ctx_field(acceptance_task, self.PHASE_CTX_KEY, [])

    def mark_phase_as_notified(self):
        phases = th.input_or_ctx_field(self.acceptance_task, self.PHASE_CTX_KEY, [])
        th.ctx_field_set(self.acceptance_task, self.PHASE_CTX_KEY, phases + [self.phase])

    @classmethod
    def _get_phase(cls):
        if utils.amount_of_working_subtasks():
            return cls.STARTED
        if utils.check_all_subtasks_done():
            return cls.DONE
        return cls.UNKNOWN
