# -*- coding: utf-8 -*-
import os

from sandbox import sdk2

import sandbox.projects.BuildRtyServer as BRS
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.search.gdb as gdb
from sandbox.projects.common import task_env
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.mixins.build as rm_build_mixin
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.resource_types as rt
import sandbox.projects.websearch.basesearch.BuildBasesearch2 as BB2
import sandbox.projects.websearch.ChildTaskRestarter as CTR

from sandbox.projects.release_machine.components.configs.base import BaseCfg

RTY_SERVER_RESOURCES = [
    (rt.RTYSERVER, "saas/rtyserver", "rtyserver", "rtyserver"),
    (rt.RTYSERVER_INDEXER_PROXY, "saas/indexerproxy", "indexerproxy", "rtyserver_indexerproxy"),
    (rt.DISTRIBUTOR, "yweb/realtime/distributor/monolith", "monolith", "distributor"),
]
RTY_SERVER_TASK = "BUILD_RTYSERVER"
BASESEARCH_TASK = "BUILD_BASESEARCH_2"


@rm_notify.notify2()
class BuildBasesearchAndRty(rm_build_mixin.ComponentReleaseTemplate, nanny.ReleaseToNannyTask2, CTR.ChildTaskRestarter):

    """
    Build basesearch and rty server for basesearch stable releases
    """
    _LOCAL_BIN_DIR = "binaries"
    _LOCAL_RELEASE_DIR = BB2.LOCAL_RELEASE_DIR

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group(build_params.BASE_BUILD_GROUP_NAME) as bb_group:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
            # arcadia_patch = sdk2.parameters.String(
            #     "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            #     default='', multiline=True
            # )
            clear_build = sdk2.parameters.Bool("Clear build", default=True)
        with sdk2.parameters.Group(build_params.BUILD_SYSTEM_GROUP_NAME) as bs_group:
            with sdk2.parameters.String("Build type", required=True) as build_type:
                build_type.values.release = build_type.Value("Release", default=True)
                build_type.values.debug = "Debug"
                build_type.values.profile = "Profile"
                build_type.values.coverage = "Coverage"
                build_type.values.relwithdebinfo = "Release with debug info"
                build_type.values.valgrind = "Valgrind (debug)"
                build_type.values.valgrind_release = "Valgrind (release) (only for build with ya)"

            definition_flags = sdk2.parameters.String(
                "Definition flags (for example '-Dkey1=val1 ... -DkeyN=valN')", required=False
            )

            target_platform_flags = sdk2.parameters.String(
                "Target platform flags (only for cross-compilation)", required=False
            )
            musl = sdk2.parameters.Bool("Use --musl option", default=True)
            use_fuse = sdk2.parameters.Bool("Use fuse to checkout arcadia", default=False)
            thinlto = sdk2.parameters.Bool("Build with ThinLTO", default=False)

        with sdk2.parameters.Group("Projects") as p_group:
            with sdk2.parameters.CheckGroup("Build search projects") as search_projects:
                for field in BB2.build_fields:
                    setattr(
                        search_projects.values, field, search_projects.Value("Build {}".format(field), checked=True)
                    )
        build_rty_server = sdk2.parameters.Bool("Build rty server", default=True)
        with build_rty_server.value[True]:
            rty_target_platform_flags = sdk2.parameters.String("Target platform flags (only for cross-compilation)")
            build_bundle = sdk2.parameters.Bool("Build bundle (multiple archs)", default=False)
            ymake_parameters = sdk2.parameters.String("ymake parameters (all -D)", default="")
            ymake_flags = sdk2.parameters.String("ymake -W, -f flags", default="")
            save_make_paths = sdk2.parameters.Bool("Save make paths", default=False)
            write_deps_tree = sdk2.parameters.Bool("Write dependencies tree", default=False)

    def on_enqueue(self):
        setattr(self.Context, rm_const.COMPONENT_CTX_KEY, BaseCfg.name)
        if not self.Context.child_tasks:
            self.Context.child_tasks = {
                BASESEARCH_TASK: {
                    "resources": {},
                    "params": {
                        consts.ARCADIA_URL_KEY: self.Parameters.checkout_arcadia_from_url,
                        # consts.ARCADIA_PATCH_KEY: self.Parameters.arcadia_patch,
                        consts.BUILD_TYPE_KEY: self.Parameters.build_type,
                        consts.DEFINITION_FLAGS_KEY: self.Parameters.definition_flags,
                        consts.TARGET_PLATFORM_FLAGS: self.Parameters.target_platform_flags,
                        consts.MUSL: self.Parameters.musl,
                        consts.THINLTO: self.Parameters.thinlto,
                        "search_projects": self.Parameters.search_projects,
                        "use_fuse": self.Parameters.use_fuse,
                    },
                    "sdk2": True
                }
            }
            for ft in self._get_targets():
                ft_name = getattr(BB2.build_fields[ft], "arcadia_build_name", BB2.build_fields[ft].arcadia_build_path)
                res = BB2.build_fields[ft](
                    self, ft_name, os.path.join(self._LOCAL_RELEASE_DIR, BB2.build_fields[ft].arcadia_build_path)
                )
                self.Context.child_tasks[BASESEARCH_TASK]["resources"][BB2.build_fields[ft].name] = res.id
                self.Context.child_tasks[BASESEARCH_TASK]["params"]["{}_res_id".format(ft)] = res.id

            if self.Parameters.build_rty_server:
                self.Context.child_tasks[RTY_SERVER_TASK] = {
                    "resources": {},
                    "params": {
                        consts.BUILD_BUNDLE_KEY: self.Parameters.build_bundle,
                        consts.USE_AAPI_FUSE: self.Parameters.use_fuse,
                        # consts.ARCADIA_PATCH_KEY: self.Parameters.arcadia_patch,
                        consts.CLEAR_BUILD_KEY: self.Parameters.clear_build,
                        consts.BUILD_TYPE_KEY: self.Parameters.build_type,
                        consts.TARGET_PLATFORM_FLAGS: self.Parameters.rty_target_platform_flags,
                        consts.THINLTO: self.Parameters.thinlto,
                        BRS.PYmakeFlags.name: self.Parameters.ymake_flags,
                        BRS.PYmakeParameters.name: self.Parameters.ymake_parameters,
                        BRS.PSaveMakePaths.name: self.Parameters.save_make_paths,
                        BRS.PWriteDepsTree.name: self.Parameters.write_deps_tree,
                        "parent_resource_ids": {},
                    },
                    "sdk2": False
                }
                rty_server_ctx = self.Context.child_tasks[RTY_SERVER_TASK]
                for rty_resource, target_path, filename, build_field in RTY_SERVER_RESOURCES:
                    resource_name = "{} ({})".format(self.Parameters.description, filename)
                    resource_path = os.path.join(self._LOCAL_BIN_DIR, filename)
                    res = rty_resource(self, resource_name, resource_path)
                    rty_server_ctx["resources"][rty_resource.name] = res.id
                    rty_server_ctx["params"]["parent_resource_ids"][build_field] = res.id
                    rty_server_ctx["params"]["build_{}".format(build_field)] = True
        super(BuildBasesearchAndRty, self).on_enqueue()

    def _get_targets(self):
        if isinstance(self.Parameters.search_projects, list):
            return self.Parameters.search_projects
        return self.Parameters.search_projects.split(" ")

    def on_execute(self):
        gdb.append_to_stable_only(self)
        arcadia_src = self.Parameters.checkout_arcadia_from_url
        self.Context.checkout_arcadia_from_url = arcadia_src.strip("/")
        if self.Parameters.build_rty_server:
            self.Context.child_tasks[RTY_SERVER_TASK]["params"][consts.ARCADIA_URL_KEY] = (
                self.Context.checkout_arcadia_from_url
            )
        super(BuildBasesearchAndRty, self).on_execute()
