# coding: utf-8
import os
import logging

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process


class ResourceDescription(SandboxStringParameter):
    name = 'description'
    description = 'BasesearchStub build description'
    default_value = 'BasesearchStub v0.0.0'


class Tag(SandboxStringParameter):
    name = 'git_tag'
    description = 'BasesearchStub git tag or branch name'
    default_value = 'tag_for_build'
    required = True


class BuildBasesearchStub(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_BASESEARCH_STUB'

    environment = []

    input_parameters = [
        ResourceDescription,
        Tag,
    ]

    GIT_URL = 'https://git.qe-infra.yandex-team.ru/scm/nanny/base-search-stub.git'

    def _checkout(self, sources_path):
        tag = self.ctx.get('git_tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'
        run_process(
            ['git', 'clone', self.GIT_URL, sources_path],
            log_prefix='git_clone',
            shell=True,
        )
        run_process(
            ['git', 'checkout', tag],
            work_dir=sources_path,
            log_prefix='git_co',
            shell=True)

    def _make_resource(self, resource_path, type):
        logging.info('registering resource')

        self.create_resource(
            description=self.ctx['description'],
            resource_path=resource_path,
            resource_type=type,
        )

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе

            :return список из трёх значений revision, tag, branch
        """
        return None, self.ctx.get('git_tag'), None

    def on_execute(self):
        """
        Plan is:
        * git clone and checkout specified tag
        * create a resource from stub.py
        """
        sources_path = self.path('sources_download')
        self._checkout(sources_path)

        self._make_resource(os.path.join(sources_path, 'stub.py'), resource_types.BASESEARCH_STUB_EXECUTABLE)


__Task__ = BuildBasesearchStub
