# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import massif
from sandbox.projects.common.utils import splitthousands


class CompareBasesearchMemory(SandboxTask):
    type = 'COMPARE_BASESEARCH_MEMORY'
    execution_space = 50  # 50M is enough according to sandbox info
    cores = 1

    class ResourcesInfoField(parameters.SandboxInfoParameter):
        name = "resources_info_field"
        description = "Task resources"

    class ValgrindOut1ResourceId(parameters.ResourceSelector):
        name = "valgrind_out1_resource_id"
        description = "Valgrind output 1"
        resource_type = resource_types.VALGRIND_MASSIF_OUTPUT
        required = True

    class ValgrindOut2ResourceId(parameters.ResourceSelector):
        name = "valgrind_out2_resource_id"
        description = "Valgrind output 2"
        resource_type = resource_types.VALGRIND_MASSIF_OUTPUT
        required = True

    class MinDiff(parameters.SandboxIntegerParameter):
        name = "min_diff"
        description = "Min diff (Mb)"
        default_value = 5000000

    input_parameters = [ResourcesInfoField, ValgrindOut1ResourceId, ValgrindOut2ResourceId, MinDiff]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result.html', resource_types.BASESEARCH_MEMORY_COMPARE_RESULT)
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        valgrind_out1_path = self.sync_resource(self.ctx['valgrind_out1_resource_id'])
        valgrind_out2_path = self.sync_resource(self.ctx['valgrind_out2_resource_id'])

        mem_size1 = massif.get_peak_mem_size(os.path.join(valgrind_out1_path, 'massif.out'))
        mem_size2 = massif.get_peak_mem_size(os.path.join(valgrind_out2_path, 'massif.out'))

        resource_path = channel.sandbox.get_resource(self.ctx['out_resource_id']).path
        _save_diff(resource_path, mem_size1, mem_size2)

        self.mark_resource_ready(self.ctx['out_resource_id'])

        self.ctx['diff'] = mem_size2 - mem_size1
        self.ctx['has_diff'] = abs(mem_size1 - mem_size2) > self.ctx['min_diff']

    def get_results(self):
        if not self.is_completed():
            return 'Results are not ready yet.'

        return 'Diff: %s' % (splitthousands(self.ctx['diff']))

    def get_short_task_result(self):
        if not self.is_completed():
            return None

        return splitthousands(self.ctx['diff'] / 1000000)


__Task__ = CompareBasesearchMemory


def _save_diff(fileName, mem_size1, mem_size2):
    with open(fileName, "w") as file:
        file.write("<html><body><table border=1>")
        file.write("<tr><td>%s</td><td>%s</td></tr>" % ("mem 1", splitthousands(mem_size1)))
        file.write("<tr><td>%s</td><td>%s</td></tr>" % ("mem 2", splitthousands(mem_size2)))
        file.write("</table></body></html>")
