# -*- coding: utf-8 -*-

import os
import json
import logging

from sandbox.projects import resource_types

from sandbox.sandboxsdk import process as p
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk import environments

from sandbox.projects.common.differ import coloring
from sandbox.projects.common.dolbilka import stats_parser
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import stats as mst
from sandbox.projects.common import templates
from sandbox.projects.common import apihelpers

COMPATIBLE_TASK_TYPES = [
    'TEST_BASESEARCH_PERFORMANCE',
    'TEST_BASESEARCH_PERFORMANCE_BEST',
    'TEST_WIZARD_PERFORMANCE_BEST',
    'ENTITYSEARCH_SHARP_SHOOTER',
]


def compare_intervals(list_intervals1, list_intervals2):
    list_of_avg_1 = []
    for i in range(0, len(list_intervals1[0])):
        summ = 0
        for sub_list in list_intervals1:
            summ += sub_list[i]
        list_of_avg_1.append(summ/len(list_intervals1[0]))

    list_of_avg_2 = []
    for i in range(0, len(list_intervals2[0])):
        summ = 0
        for sub_list in list_intervals2:
            summ += sub_list[i]
        list_of_avg_2.append(summ/len(list_intervals2[0]))

    list_of_avg_diff = map(lambda x, y: x - y, list_of_avg_1, list_of_avg_2)
    return list_of_avg_diff


class TestResult1(parameters.TaskSelector):
    name = 'test_results1_task_id'
    description = 'Task 1'
    task_type = COMPATIBLE_TASK_TYPES
    group = 'Test basesearch performance tasks'


class TestResult2(parameters.TaskSelector):
    name = 'test_results2_task_id'
    description = 'Task 2'
    task_type = COMPATIBLE_TASK_TYPES
    group = 'Test basesearch performance tasks'


class TopResults(parameters.SandboxIntegerParameter):
    name = 'n_top_results'
    description = 'Number of top results taken into account'
    default_value = 3
    group = 'Checking parameters'


class MaxDiff(parameters.SandboxIntegerParameter):
    name = 'max_diff'
    description = 'Max diff (request/sec)'
    default_value = 10
    group = 'Checking parameters'


class MaxDiffPercent(parameters.SandboxIntegerParameter):
    name = 'max_diff_percent'
    description = 'Percentage (%)'
    default_value = 10
    group = 'Checking parameters'


class MaxMemoryDiff(parameters.SandboxIntegerParameter):
    name = 'max_memory_diff'
    description = 'Maximum possible memory difference (%)'
    default_value = 10
    group = 'Checking parameters'


class CheckPerformance(parameters.SandboxBoolParameter):
    name = 'check_performance'
    description = 'Check performance'
    default_value = True
    group = 'Checking parameters'


class CheckPerformanceExtended(parameters.SandboxBoolParameter):
    name = 'check_performance_extended'
    description = "Check performance 2d"
    default_value = False
    group = 'Checking parameters'


class CheckIntervals(parameters.SandboxBoolParameter):
    name = 'check_intervals'
    description = 'Check on intervals'
    default_value = False
    group = 'Checking parameters'


class CheckMemory(parameters.SandboxBoolParameter):
    name = 'check_memory'
    description = 'Check memory'
    sub_fields = {'true': [MaxMemoryDiff.name]}
    default_value = False
    group = 'Checking parameters'


class CompareType(parameters.SandboxStringParameter):
    name = "compare_type"
    description = 'Comparison type'
    choices = [
        ('percent', 'percent'),
        ('rps', 'rps')
    ]
    sub_fields = {
        'percent': [MaxDiffPercent.name],
        'rps': [MaxDiff.name]
    }
    default_value = 'rps'
    group = 'Checking parameters'


class CompareBasesearchPerformance(SandboxTask):
    """
    **Описание**
        Сравнение производительности двух сборок базового поиска.
        Также работает для визарда.

    **Ресурсы**
        *Необходимые для запуска ресурсы и параметры*
            * test results 1 - идентификатор задачи с тестированием производительности первой сборки
            * test results 2 - идентификатор задачи с тестированием производительности второй сборки
            * Number of top results taken into account - число результатов, по которым будет идти сравнение
            * max diff (request/sec) - максимальная допустимая разница
            * Percentage (%)
            * Comparison type
            * Check on intervals
        *Создаваемые ресурсы*
            * BASESEARCH_PERFORMANCE_COMPARE_RESULT
    """

    type = 'COMPARE_BASESEARCH_PERFORMANCE'
    execution_space = 5 * 1024  # 5 Gb
    input_parameters = [
        TestResult1,
        TestResult2,
        CheckPerformance,
        CheckPerformanceExtended,
        CheckIntervals,
        CheckMemory,
        TopResults,
        MaxMemoryDiff,
        CompareType,
        MaxDiff,
        MaxDiffPercent,
    ]
    cores = 1

    @property
    def footer(self):
        if self.ctx.get(CheckPerformanceExtended.name):
            foot = [{
                "helperName": "",
                "content": {
                    "<h3>Extended performance result</h3>": {
                        "header": [
                            {"key": "avg_rps_1", "title": "Avg RPS 1"},
                            {"key": "avg_rps_2", "title": "Avg RPS 2"},
                            {"key": "diff", "title": "Diff, %"},
                            {"key": "prob", "title": "Probability, %"},
                        ],
                        "body": self._get_extended_perf_diff_info()
                    }
                }
            }]
            return foot

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(
            self.descr, 'compare_result',
            resource_types.BASESEARCH_PERFORMANCE_COMPARE_RESULT
        )
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        task1_id = self.ctx[TestResult1.name]
        task2_id = self.ctx[TestResult2.name]
        utils.check_tasks_to_finish_correctly(task_ids_list=[task1_id, task2_id])

        # Compare rps
        stats_task1 = self.parse_stat(self.ctx[TestResult1.name])
        stats_task2 = self.parse_stat(self.ctx[TestResult2.name])

        top_results_number = self.ctx[TopResults.name]
        compare_result = True
        resource_path = channel.sandbox.get_resource(self.ctx['out_resource_id']).path
        make_folder(resource_path)
        with open(os.path.join(resource_path, 'index.html'), 'w') as resource_file:
            resource_file.write('<html><body>')
            if self.ctx.get(CheckPerformance.name):
                rps_compare_result, rps_stats = self._check_performance(stats_task1, stats_task2, top_results_number)
                compare_result = compare_result and rps_compare_result
                top1, avg1, top2, avg2, diff = rps_stats
                self.ctx['rps_diff'] = diff
                resource_file.write(self.html_diff('Comparing rps', 'request/sec', rps_stats))

            extended_perf_diff_info = self._count_t_criteria(stats_task1, stats_task2)
            if extended_perf_diff_info:
                resource_file.write('<h1>Extended perf diff info</h1>')
                for k, v in extended_perf_diff_info.iteritems():
                    resource_file.write('<p><b>{}</b>: {}</p>'.format(k, v[0]))

            if self.ctx[CheckIntervals.name]:
                intervals = self._check_intervals(stats_task1, stats_task2)
                resource_file.write(intervals)

            if self.ctx.get(CheckMemory.name):
                mem_compare_result, mem_stats = self._check_memory(stats_task1, stats_task2, top_results_number)
                compare_result = compare_result and mem_compare_result
                resource_file.write(self.html_diff('Comparing memory', 'KB', mem_stats))
            resource_file.write('</body></html>')
        self.mark_resource_ready(self.ctx['out_resource_id'])
        self.ctx['compare_result'] = compare_result

    @staticmethod
    def _write_rps_2d_stats(stats, output_file_name):
        try:
            fu.json_dump(output_file_name, stats['requests_per_sec_2d'])
        except KeyError:
            eh.check_failed(
                "No rps 2d data aggregated, maybe test results are too old and resources expired"
            )

    def _check_performance(self, stats_task1, stats_task2, top_results_number):
        """
            Сравнивает производительность
        """
        try:
            top1, avg1 = self._top_results(stats_task1['requests_per_sec'], top_results_number)
        except KeyError:
            eh.check_failed(
                "No rps data aggregated, maybe test results 1 are too old and resources expired"
            )
        try:
            top2, avg2 = self._top_results(stats_task2['requests_per_sec'], top_results_number)
        except KeyError:
            eh.check_failed(
                "No rps data aggregated, maybe test results 2 are too old and resources expired"
            )
        # much better than anything else
        diff = max(top2) - max(top1)

        if self.ctx[CompareType.name] == 'rps':
            compare_result = (abs(diff) <= self.ctx[MaxDiff.name])
        elif self.ctx[CompareType.name] == 'percent':
            if max(top1) < 1.0:
                logging.info('Diff failed -- First task RPSes are too low')
                compare_result = False
            else:
                diff_percent = 100 * abs(diff) / max(top1)
                logging.info('Diff per cent: %d', diff_percent)
                compare_result = (diff_percent <= self.ctx[MaxDiffPercent.name])
        else:
            eh.fail("Unknown compare type: {}".format(self.ctx[CompareType.name]))

        stats = top1, avg1, top2, avg2, diff
        return compare_result, stats

    def _count_t_criteria(self, stats_task1, stats_task2):
        """
            Experimental method.
            Gets top3 average rps for each shoot. Then uses Welch's t-test.
            If the p-value is smaller than the threshold,
            e.g. 1%, 5% or 10%, then we reject the null hypothesis of equal averages.
        """
        if not utils.get_or_default(self.ctx, CheckPerformanceExtended):
            logging.info("Skip extended performance stats calculation")
            return
        try:
            with environments.VirtualEnvironment() as venv:
                logging.info('Installing numpy + scipy...')
                environments.PipEnvironment('numpy', use_wheel=True, venv=venv, version="1.12.1").prepare()
                environments.PipEnvironment('scipy', use_wheel=True, venv=venv, version="0.19.0").prepare()
                logging.info("Try to calculate extended performance stats")
                stats_path1 = self.abs_path('rps_2d_1.json')
                stats_path2 = self.abs_path('rps_2d_2.json')
                self._write_rps_2d_stats(stats_task1, stats_path1)
                self._write_rps_2d_stats(stats_task2, stats_path2)
                proc = p.run_process(
                    [venv.executable, mst.get_module_path() + '/rps2d_mu_stats.py', stats_path1, stats_path2],
                    log_prefix='t_test',
                    outputs_to_one_file=False,
                    check=True,
                    wait=True,
                )
                with open(proc.stdout_path) as f:
                    t_stats = json.load(f)
                    self.ctx.update(t_stats)
                    self.ctx["diff_probability"] = round((1 - float(self.ctx.get("p_value", 0))) * 100, 2)
                    self.ctx["significant_rps_diff"] = (
                        t_stats["p_value"] < 0.05 and
                        abs(t_stats["diff_per_cent_for_avg"]) > utils.get_or_default(self.ctx, MaxDiffPercent)
                    )
                return self._get_extended_perf_diff_info()
        except Exception:
            eh.fail('Cannot calculate t-criteria:\n{}'.format(eh.shifted_traceback()))

    def _check_intervals(self, stats_task1, stats_task2):
        intervals = ''
        try:
            list_intervals1 = stats_task1['intervals_Oy']
            rage1 = stats_task1['intervals_Ox']
        except KeyError:
            eh.check_failed("Check intervals: test results 1 are too old")

        try:
            list_intervals2 = stats_task2['intervals_Oy']
            rage2 = stats_task2['intervals_Ox']
        except KeyError:
            eh.check_failed("Check intervals: test results 2 are too old")

        eh.ensure(rage1 == rage2, 'Check intervals: statistic time tables are not equal')

        list_intervals1.pop(0)
        list_intervals2.pop(0)

        eh.ensure(len(list_intervals1) > 0, "Check intervals: test results 1 contain too little results")
        eh.ensure(len(list_intervals2) > 0, "Check intervals: test results 2 contain too little results")

        list_of_avg_diff = compare_intervals(list_intervals1, list_intervals2)
        self.ctx['res_intervals'] = (len(filter(lambda x: abs(x) >= 1, list_of_avg_diff)) == 0)

        if not self.ctx['res_intervals']:
            intervals = '<b>Diff intervals</b><br><table> {} </table>'.format(
                ''.join(
                    '<tr><td width="100">%.2f</td><td width="100">%.2f</td></tr>' % (time, val)
                    for (time, val) in zip(rage2, list_of_avg_diff)
                )
            )

        return intervals

    def _check_memory(self, stats_task1, stats_task2, top_results_number):
        """
            Сравнивает использование памяти
        """
        top1, avg1 = self._top_results(stats_task1['memory_rss'], top_results_number)
        top2, avg2 = self._top_results(stats_task2['memory_rss'], top_results_number)
        diff = avg2 - avg1
        max_diff = avg1 * self.ctx[MaxMemoryDiff.name] / 100

        compare_result = (abs(diff) <= max_diff)
        stats = top1, avg1, top2, avg2, diff

        return compare_result, stats

    @staticmethod
    def html_diff(title, unit, stats):
        """
            Возвращает текстовый отчёт с результатами
            в формате html
        """
        top1, avg1, top2, avg2, diff = stats
        html_diff_template = templates.get_html_template('diff_template.html')

        return html_diff_template.format(
            title=title, unit=unit,
            top1=", ".join(str(x) for x in top1),
            top2=", ".join(str(x) for x in top2),
            avg1=avg1, avg2=avg2,
            diff=diff,
        )

    def _get_extended_perf_diff_info(self):
        return {
            "avg_rps_1": [round(float(self.ctx.get("avg_rps_1", 0)), 2)],
            "avg_rps_2": [round(float(self.ctx.get("avg_rps_2", 0)), 2)],
            "diff": [coloring.color_diff(
                float(self.ctx.get("diff_per_cent_for_avg", 0)),
                max_diff=-utils.get_or_default(self.ctx, MaxDiffPercent),
                probability=self.ctx.get("diff_probability", 0)
            )],
            "prob": [self.ctx.get("diff_probability", 0)],
        }

    @staticmethod
    def _top_results(results, num):
        """
            Возвращает последние n результатов и среднее значение для них
        """
        eh.verify(len(results) >= num, 'Test results should contain at least {} results'.format(num))

        top = sorted(results, reverse=True)[:num]
        avg = sum(top) / len(top)

        return top, avg

    def parse_stat(self, task_id):
        stats_data = {}
        list_resources = apihelpers.list_task_resources(task_id=task_id, resource_type='EXECUTOR_STAT')
        for rs in sorted(list_resources, key=lambda resource: resource.id):
            rs_path = self.sync_resource(rs.id)
            data = stats_parser.StatsParser(rs_path)
            n = data.vars.get('requests/sec', 0)
            stats_data['intervals_Ox'] = data.mcr_sec_intervals
            try:
                stats_data['requests_per_sec'].append(n)
                stats_data['intervals_Oy'].append(data.val_intervals)
            except KeyError:
                stats_data['requests_per_sec'] = [n]
                stats_data['intervals_Oy'] = [data.val_intervals]

        task = channel.sandbox.get_task(task_id)

        # *_BEST tasks do not have resources but have rps values in context
        if not list_resources:
            stats_data['requests_per_sec'] = task.ctx['requests_per_sec']
            try:
                stats_data['requests_per_sec_2d'] = task.ctx['requests_per_sec_2d']
            except KeyError:
                pass

        if self.ctx.get(CheckMemory.name):
            stats_data['memory_rss'] = task.ctx['memory_rss']

        return stats_data


__Task__ = CompareBasesearchPerformance
