# -*- coding: utf-8 -*-

import logging

from sandbox import common

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import utils
from sandbox.projects.common.base_search_quality import response_diff_task as rdt
from sandbox.projects.common.base_search_quality import response_saver

from sandbox.projects.common.geosearch import response_parser as grp
from sandbox.projects.common.noapacheupper import response_patcher as nrp
from sandbox.projects.common.rtyserver import response_patcher as rrp


_GENERATION_TASK_TYPES = (
    'GET_BASESEARCH_RESPONSES',
    'GET_WEB_BASESEARCH_RESPONSES',
    'GET_MIDDLESEARCH_RESPONSES',
    'GET_WEB_MIDDLESEARCH_RESPONSES',
    'IMAGES_GET_BASESEARCH_RESPONSES',
    'IMAGES_GET_MIDDLESEARCH_RESPONSES',
    'VIDEO_GET_MIDDLESEARCH_RESPONSES',
    'BUILD_NOAPACHEUPPER_NEH_CACHE',
    'GET_STANDALONE_NOAPACHEUPPER_RESPONSES',
    'GET_STANDALONE_NOAPACHEUPPER_RESPONSES2',
    'GET_SINGLE_HOST_NOAPACHEUPPER_RESPONSES',
)
"""
    Список задач, создающих ресурсы типа
    BASESEARCH_HR_RESPONSES
"""


class ResponseFormatParameter(parameters.SandboxStringParameter):
    name = 'response_format'
    description = 'Use responses format'
    choices = [
        ('basesearch', 'basesearch'),
        ('geosearch', 'geosearch'),
        ('noapacheupper', 'noapacheupper'),
        ('saas_oxygen', 'saas_oxygen'),
    ]
    default_value = 'basesearch'


class CompareBasesearchResponses(rdt.BaseCompareBasesearchResponses):
    """
        Сравнивает два выхлопа (базового) поиска, полученных в human-readable (hr=da) формате.
        Строит diff по запросам и по изменившимся свойствам.
        Использует результаты задачи GET_BASESEARCH_RESPONSES.
    """

    type = 'COMPARE_BASESEARCH_RESPONSES'
    cores = 1
    input_parameters = (
        (
            ResponseFormatParameter,
        ) +
        rdt.BaseCompareBasesearchResponses.input_parameters
    )

    def _get_response_patchers(self):
        patchers = rdt.BaseCompareBasesearchResponses._get_response_patchers(self)
        response_format = utils.get_or_default(self.ctx, ResponseFormatParameter)

        if response_format == 'geosearch':
            patchers.append(grp.GeosearchResponsePatcher)
        elif response_format == 'noapacheupper':
            patchers.append(nrp.response_patcher)
        elif response_format == 'saas_oxygen':
            patchers.append(rrp.response_patcher)

        return patchers

    def _find_queries(self, responses_resource):
        task_id = responses_resource.task_id
        logging.info("Retrieving responses getting task, id=%s", task_id)
        sandbox = common.rest.Client()
        task_type = sandbox.task[task_id][:]["type"]
        if task_type not in _GENERATION_TASK_TYPES:
            logging.info("task {}.{} not in allowed _GENERATION_TASK_TYPES".format(
                task_id, task_type
            ))
            return None

        queries_resource_id = sandbox.task[task_id].context[:][
            response_saver.DefaultResponseSaverParams.QueriesParameter.name
        ]
        if self.ctx.get(ResponseFormatParameter.name) == 'noapacheupper':
            queries_resource = channel.sandbox.get_resource(queries_resource_id)
            if queries_resource and queries_resource.type != 'USERS_QUERIES':
                # noapache queries are ugly and huge (200k-300k),
                # so display original user queries, if we can (or None on failure)
                queries_resource_id = channel.sandbox.get_resource_attribute(
                    queries_resource_id, 'text_queries_resource_id'
                )
        logging.info("User queries resource id: %s", queries_resource_id)
        return queries_resource_id

    def _get_custom_node_types(self):
        custom_node_types_dict = rdt.BaseCompareBasesearchResponses._get_custom_node_types(self)
        if self.ctx.get(ResponseFormatParameter.name) == 'geosearch':
            custom_node_types_dict.update(grp.GetGeosearchCustomNodeTypesDict())
        return custom_node_types_dict


__Task__ = CompareBasesearchResponses
