# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters

from sandbox.projects.common.base_search_quality import priemka
from sandbox.projects.common import utils

from sandbox.projects import MixQueriesExperimentsRegions as mq


class TestConcurrentRelevance(parameters.SandboxBoolParameter):
    name = 'test_concurrent_relevance'
    description = 'Test concurrent relevance'
    default_value = False


class NewCgiParam(parameters.SandboxStringParameter):
    name = 'new_cgi_param'
    description = 'New cgi params (must be encoded, may contain multiple params divided by \';\')'
    default_value = ''
    group = 'Queries generation setup'


class ChildTaskPriority(parameters.SandboxIntegerParameter):
    name = 'child_tasks_priority'
    description = 'Child task priority'
    default_value = 100


other_res_source = priemka.get_source_for_other_resources(fast=True)


class FastCheckBasesearchResponses(SandboxTask):
    """
        Задача определяет, для каких типов шардов в sandbox есть данные, загруженные ранее.
        Для каждого типа шарда делает следующее:

        * Подготавливает запросы с помощью запуска subtask'а MIX_QUERIES_EXPERIMENTS_REGIONS.
        * Получает выдачу бинарника путем запуска subtask'а GET_BASESEARCH_RESPONSES
        * Падает, если ответов не получилось
    """
    type = 'FAST_CHECK_BASESEARCH_RESPONSES'

    input_parameters = [
        priemka.ExecutableSource1,
        priemka.Executable1,
        priemka.DynamicModelsSource1,
        priemka.DynamicModels1,
        other_res_source,
    ] + priemka.get_other_existing_res(fast=True) + [
        priemka.GetAllFactors,
        priemka.PoliteMode,
        priemka.IgnoreGotError,
        NewCgiParam,
        ChildTaskPriority,
        TestConcurrentRelevance,
    ] + mq.exp_setup_params.params + mq.queries_gen_params.params

    def initCtx(self):
        priemka.reset_resources_ids(self)

    def on_execute(self):
        if not self.list_subtasks():
            res_by_shard = priemka.get_shard_tags_and_resources(fast=True)
            for shard_tag in res_by_shard:
                res_by_shard[shard_tag]['patched_queries_resource_id'] = self._prepare_queries(
                    res_by_shard[shard_tag]['queries'].id,
                    'Mixed queries for shard tag {}'.format(shard_tag)
                )

                self._get_basesearch_responses(
                    priemka.get_basesearch_exe(self.ctx),
                    res_by_shard[shard_tag],
                    priemka.get_models_archive(self.ctx),
                    concurrent_relevance=False,
                    task_description='Responses for mixed queries for shard tag {}'.format(shard_tag),
                )
            utils.wait_all_subtasks_stop()
        elif utils.check_all_subtasks_done():
            utils.check_subtasks_fails()
        else:
            utils.restart_broken_subtasks()

    def _prepare_queries(self, queries_source_id, task_description):
        logging.info('start preparing (mixing) queries ...')
        exp = mq.exp_setup_params
        q_gen = mq.queries_gen_params
        sub_ctx = {
            mq.Queries.name: queries_source_id,
            exp.ExpsChanged.name: self.ctx[exp.ExpsChanged.name],
            exp.SnipExpsChanged.name: self.ctx[exp.SnipExpsChanged.name],
            q_gen.NumOfSearchQueries.name: self.ctx[q_gen.NumOfSearchQueries.name],
            q_gen.NumOfXmlSearchQueries.name: self.ctx[q_gen.NumOfXmlSearchQueries.name],
            q_gen.NumSnippetQueries.name: self.ctx[q_gen.NumSnippetQueries.name],
            q_gen.QueriesWithModifiedExp.name: self.ctx[q_gen.QueriesWithModifiedExp.name],
            q_gen.QueriesWithModifiedReg.name: self.ctx[q_gen.QueriesWithModifiedReg.name],
            q_gen.QueriesVsRegRatio.name: self.ctx[q_gen.QueriesVsRegRatio.name],
            q_gen.FailOnLowAmountOfQueries.name: (
                (not self.ctx[TestConcurrentRelevance.name]) and self.ctx[q_gen.FailOnLowAmountOfQueries.name]
            )
        }

        subtask = self.create_subtask(
            task_type='MIX_QUERIES_EXPERIMENTS_REGIONS',
            input_parameters=sub_ctx,
            description=task_description,
            priority=self.ctx[ChildTaskPriority.name]
        )

        queries_resource_id = subtask.ctx['out_resource_id']

        new_cgi_param = self.ctx.get(NewCgiParam.name)
        if new_cgi_param:
            sub_ctx = {
                'queries_resource_id': queries_resource_id,
                'new_cgi_param': new_cgi_param,
            }

            subtask = self.create_subtask(
                task_type='PATCH_QUERIES',
                input_parameters=sub_ctx,
                description=task_description,
                priority=self.ctx[ChildTaskPriority.name]
            )

            queries_resource_id = subtask.ctx['out_resource_id']

        return queries_resource_id

    def _get_basesearch_responses(
        self,
        exe_res_id,
        resources,
        models_archive_res_id,
        concurrent_relevance,
        task_description,
    ):
        logging.info('Obtaining responses...')

        sub_ctx = {
            'basesearch_type': 'basesearch',
            'executable_resource_id': exe_res_id,
            'config_resource_id': resources['config'].id,
            'search_database_resource_id': resources['db'].id,
            'queries_resource_id': resources['patched_queries_resource_id'],
            'models_archive_resource_id': models_archive_res_id,
            'min_output_size': 6000000,
            'relevance_threads': 5 if concurrent_relevance else 0,
            'get_all_factors': self.ctx[priemka.GetAllFactors.name],
            'polite_mode': self.ctx[priemka.PoliteMode.name],
            'ignore_got_error': self.ctx[priemka.IgnoreGotError.name],
        }

        subtask = self.create_subtask(
            task_type='GET_BASESEARCH_RESPONSES',
            input_parameters=sub_ctx,
            description=task_description,
            priority=self.ctx[ChildTaskPriority.name],
            arch='linux',
        )

        return subtask.id


__Task__ = FastCheckBasesearchResponses
