import datetime

from sandbox import sdk2
from sandbox import common
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.websearch.basesearch import resources as bs_res
from sandbox.sdk2.helpers import subprocess as sp

GENERATED_RESOURCE_PREFIX = 'TE_web_base_attributes_{resource}_{tier}'


class GenerateAttributeQueries(sdk2.Task):
    """
        Generate attribute queries for basesearch
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024  # 2 GiB
        disk_space = 10 * 1024  # 10 GiB for PlatinumTier0 shard

        # real hostname for prs_ops
        client_tags = Tag.GENERIC & Tag.Group.LINUX & ~Tag.LXC

    class Parameters(sdk2.Task.Parameters):
        shard = sdk2.parameters.Resource(
            'Shard',
            resource_type=[resource_types.SEARCH_DATABASE],
            required=True
        )
        config = sdk2.parameters.Resource(
            'Config',
            resource_type=[bs_res.TestShardConfig],
        )
        upper = sdk2.parameters.Url(
            'Upper address',
            default='hamster.yandex.ru',
            required=True
        )
        additional_cgi = sdk2.parameters.String('Additional CGI parameters for middlesearch', default='')

        count = sdk2.parameters.Integer('Requests count', default=100)

        add_te_tag = sdk2.parameters.Bool('Add TE tag to the generated resources', default=False)

    def on_enqueue(self):
        tier = self._resolve_tier_name()
        if tier != 'PlatinumTier0':
            raise common.errors.TaskFailure('Unsupported tier {}'.format(tier))

    def on_execute(self):
        test_shard_res = bs_res.TestShardBinary.find(
            attrs={'released': 'stable'}
        ).limit(1).first()
        test_shard = sdk2.ResourceData(test_shard_res)
        shard = sdk2.ResourceData(self.Parameters.shard)
        config_path = ''
        if self.Parameters.config:
            config_path = sdk2.ResourceData(self.Parameters.config).path
        output_path = 'queries'
        with sdk2.helpers.ProcessLog(self, logger='test_shard') as pl:
            sp.check_call(
                [str(test_shard.path),
                '--generate-requests',
                '--shard', str(shard.path),
                '--number', str(self.Parameters.count),
                '--upper', self.Parameters.upper,
                '--middlesearch-params', self.Parameters.additional_cgi,
                '--save-wizarded-queries', output_path,
                '--output', '/dev/null',
                '--config', config_path],
                shell=False, stdout=pl.stdout, stderr=pl.stdout
            )
        self._update_database_resource()
        self._save_generated_queries(output_path)

    def _get_te_attrs(self, resource_type):
        attrs = {}
        if self.Parameters.add_te_tag:
            day = self._get_timestamp()
            tag = GENERATED_RESOURCE_PREFIX.format(resource=resource_type, tier=self._resolve_tier_name())
            attrs[tag] = day
        return attrs

    def _update_database_resource(self):
        for k, v in self._get_te_attrs('database').iteritems():
            setattr(self.Parameters.shard, k, v)

    def _save_generated_queries(self, queries_path):
        queries_resource = resource_types.PLAIN_TEXT_QUERIES(
            self,
            'Generated queries for basesearch',
            queries_path,
        )
        for k, v in self._get_te_attrs('queries').iteritems():
            setattr(queries_resource, k, v)
        queries_resource_data = sdk2.ResourceData(queries_resource)
        queries_resource_data.ready()

    @staticmethod
    def _get_timestamp():
        return datetime.datetime.now().strftime("_".join(["%Y", "%m", "%d"]))

    def _resolve_tier_name(self):
        return self.Parameters.shard.tier
