import os
import tempfile

from sandbox.projects.common import error_handlers as eh
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.websearch.basesearch import resources as bs_res
from sandbox.sdk2.helpers import subprocess as sp


class GetBasesearchQueriesFromPlain(sdk2.Task):
    """
        Get basesearch queries from queries in prs_ops format (TSV).
        Both queries.tsv and queries_full.tsv are supported.
        Only looks at first 3 fields: <qid> \t <user_request> \t <lr>.
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 1024
        disk_space = 5 * 1024

        # real hostname for prs_ops
        client_tags = Tag.GENERIC & Tag.Group.LINUX & ~Tag.LXC

    class Parameters(sdk2.Task.Parameters):
        queries = sdk2.parameters.Resource(
            'Plain queries',
            resource_type=[bs_res.TestShardPlainQueries],
            required=True
        )
        upper = sdk2.parameters.Url(
            'Upper address',
            default='hamster.yandex.ru',
            required=True
        )
        additional_cgi = sdk2.parameters.String('Additional CGI parameters for middlesearch', default='')

        generate_plan = sdk2.parameters.Bool('Generate plan', default=True)

        with sdk2.parameters.Group('Stages') as stages:
            stage_search = sdk2.parameters.Bool('Search', default=True)
            stage_factor = sdk2.parameters.Bool('Factor', default=False)

        with sdk2.parameters.CheckGroup('Tiers') as tiers:
            tiers.values.PlatinumTier0 = tiers.Value('PlatinumTier0', checked=True)
            tiers.values.WebTier0 = tiers.Value('WebTier0', checked=False)
            tiers.values.WebTier1 = tiers.Value('WebTier1', checked=False)

        with stage_factor.value[True]:
            with sdk2.parameters.Group('Shards for factor stage') as shards:
                shard_platinum_tier_0 = sdk2.parameters.Resource(
                    'Shard for factor stage, PlatinumTier0',
                    resource_type=[resource_types.SEARCH_DATABASE],
                    required=False,
                )
                shard_web_tier_0 = sdk2.parameters.Resource(
                    'Shard for factor stage, WebTier0',
                    resource_type=[resource_types.SEARCH_DATABASE],
                    required=False,
                )
                shard_web_tier_1 = sdk2.parameters.Resource(
                    'Shard for factor stage, WebTier1',
                    resource_type=[resource_types.SEARCH_DATABASE],
                    required=False,
                )

    def on_enqueue(self):
        sizes = {
            'PlatinumTier0': 7 * 1024,
            'WebTier0': 110 * 1024,
            'WebTier1': 540 * 1024
        }
        if self.Parameters.stage_factor:
            disk_space = 5 * 1024
            for tier in self.Parameters.tiers:
                disk_space += sizes[tier]
            self.Requirements.disk_space = disk_space

    def _save_plan(self, queries_path, plan_prefix, planner, description_template):
        plan_path = plan_prefix + '.plan'
        planner.create_plan(queries_path, plan_path, host='localhost', port=1337)
        plan_resource = sdk2.ResourceData(
            resource_types.BASESEARCH_PLAN(
                self,
                description_template.format('basesearch plan'),
                plan_path
            )
        )
        plan_resource.ready()

    def _save_queries(self, queries_path, output_path, description_template):
        os.rename(queries_path, output_path)
        plain_queries_resource = sdk2.ResourceData(
            resource_types.PLAIN_TEXT_QUERIES(
                self,
                description_template.format('plain queries'),
                output_path
            )
        )
        plain_queries_resource.ready()

    def _get_stages(self):
        stage = []
        if self.Parameters.stage_search:
            stage.append('Search')
        if self.Parameters.stage_factor:
            stage.append('Factor')
        return stage

    def _get_shards(self):
        shards = []
        if self.Parameters.stage_factor:
            def add_shard(self, name, param):
                if name in self.Parameters.tiers:
                    eh.ensure(param, 'Specify {} shard for factor queries'.format(name))
                    shards.append(sdk2.ResourceData(param).path)

            add_shard(self, 'PlatinumTier0', self.Parameters.shard_platinum_tier_0)
            add_shard(self, 'WebTier0', self.Parameters.shard_web_tier_0)
            add_shard(self, 'WebTier1', self.Parameters.shard_web_tier_1)
        return shards

    def on_execute(self):
        test_shard_res = bs_res.TestShardBinary.find(
            attrs={'released': 'stable'}
        ).limit(1).first()
        test_shard = sdk2.ResourceData(test_shard_res)

        queries = sdk2.ResourceData(self.Parameters.queries)

        output_dir = tempfile.mkdtemp()
        stages = self._get_stages()
        stage_joined = ','.join(stages)
        with sdk2.helpers.ProcessLog(self, logger='test_shard') as pl:
            cmd = [str(test_shard.path),
                '--save-requests',
                '--queries', str(queries.path),
                '--output', str(output_dir),
                '--upper', self.Parameters.upper,
                '--middlesearch-params', self.Parameters.additional_cgi,
                '--output-format', 'plain',
                '--stage', stage_joined]
            for shard in self._get_shards():
                cmd.append('--shard')
                cmd.append(str(shard))
            sp.check_call(cmd, shell=False, stdout=pl.stdout, stderr=pl.stdout)

        stage_suffix = {'Search': '', 'Factor': '.factor'}
        resources_dir = 'resources'
        os.mkdir(resources_dir)

        planner = dolbilka.DolbilkaPlanner()
        for stage in stages:
            for tier in self.Parameters.tiers:
                path_gen = lambda x: os.path.join(x, '{}{}'.format(tier, stage_suffix[stage]))
                path = path_gen(output_dir) + '.tsv'
                resource_filename = path_gen(resources_dir)
                description = 'Generated {} ' + 'for {}, {} stage'.format(tier, stage)
                if self.Parameters.generate_plan:
                    self._save_plan(path, resource_filename, planner, description)
                else:
                    self._save_queries(path, resource_filename, description)
