# -*- coding: utf-8 -*-

import re
import random
import logging
import urlparse

import sandbox.common.types.client as ctc

from sandbox.projects.common import cms
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.search import instance_resolver as ir
from sandbox.projects.common.search import shards
from sandbox.projects.common.base_search_quality import settings as bss
from sandbox.projects.common.search.eventlog import eventlog

from sandbox.projects import resource_types

from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel


class MaxQueries(SandboxIntegerParameter):
    name = 'max_queries'
    description = 'Max queries'
    default_value = 10000


class DesiredRPS(SandboxIntegerParameter):
    name = 'desired_rps'
    description = 'RPS to count delay_multiplier for'
    default_value = 500


# get request from 0th subsource from int eventlog
filter_command = '''grep "SubSourceRequest.0" %s | awk '{print $1 "\t" $8}' | sed -e "s/\thttp2:/\thttp:/"'''

# fundamental Universe constant value
_MICROSECONDS_IN_SECOND = 1000 * 1000

# default tier
_TIER = bss.DEFAULT_TIER


class GetBasesearchQueriesProdSimulation(SandboxTask):
    """
        Предназначен для получения плана обстрела базового поиска, приближенного
        к продакшен-условиям, в т.ч. с продакшен-конфигом. Реализован в рамках задачи
        SEARCH-754 по мотивам SEARCHPRODINCIDENTS-27. Базовый поиск должен обстреливаться
        полученным планом с увеличенной в N раз нагрузкой, при этом не должно возникать
        большого количества HTTP 500 и других ошибок.

        **Алгоритм работы**

        Выбирает произвольный int из продакшена, смотрящий на DEFAULT_TIER,
        дампит запросы к источнику номер 0 (это один из 36 базовых поисков),
        делает из них план, сохраняя дельты между запросами,
        также выкачивает шард, соответствующий этому базовому поиску.

        **Известные баги**

        Пока что не учитывает переключение базы в момент дампа. Это нетривиально проверить,
        т.к. в запросах инта нет timestamp-а базы, можно лишь проверять, что полученный план
        продампливается без ошибок типа "Invalid document handle..."
    """
    type = 'GET_BASESEARCH_QUERIES_PROD_SIMULATION'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = (
        MaxQueries,
        DesiredRPS,
    )

    eventlog_path = 'eventlog-int.bin'
    tmp_queries_path = 'tmp_queries.txt'

    def on_enqueue(self):
        queries_resource = self.create_resource(
            description='Queries with timing',
            resource_path='queries_with_timing.txt',
            resource_type=resource_types.PLAIN_TEXT_QUERIES,
        )
        self.ctx['output_queries_resource_id'] = queries_resource.id

    def on_execute(self):
        self._get_eventlog()
        max_line_number = utils.get_or_default(self.ctx, MaxQueries)
        maxoption = "-m %d" % max_line_number if max_line_number else ""
        try:
            eventlog.filter_eventlog(
                eventlog.get_evlogdump(),
                self.eventlog_path,
                filter_command=filter_command % maxoption,
                output_path=self.tmp_queries_path,
                evlogdump_options='-O',
            )
        except Exception as e:
            logging.error("Eventlog filtering failed:\n%s", eh.shifted_traceback())
            self.set_info("Eventlog filtering failed: {}, see logs for details".format(e))
            eh.check_failed("Eventlog filtering failed, see logs for details")

        queries_resource = channel.sandbox.get_resource(self.ctx['output_queries_resource_id'])
        bs_instance, delay_multiplier = self._make_delta_plan_from_eventlog(queries_resource.path)

        now_ts_str = tu.date_ymdhm(sep="_")

        channel.sandbox.set_resource_attribute(
            queries_resource.id,
            'delay_multiplier',
            delay_multiplier,
        )
        logging.info("Obtained plan delay multiplier: %s", delay_multiplier)
        # obtain corresponding shard name
        shard_name = None
        needed_instance = "{host}:{port}@HEAD".format(**bs_instance)

        for sh, instance in cms.instances_by_host(bs_instance["host"]):
            logging.info("Check host instance: %s, %s", sh, instance)
            if instance == needed_instance:
                shard_name = sh
                break
        if shard_name is None:
            eh.check_failed("Instance {} not found in instances list".format(needed_instance))

        channel.sandbox.set_resource_attribute(
            queries_resource.id,
            'shard_name',
            shard_name,
        )
        logging.info("Obtained shard name: %s", shard_name)

        # loading shard into Sandbox
        shard = shards.ShardDownloadTask(name=shard_name)
        shard_resource = shards.get_database_shard_resource(
            _TIER, shard,
            db_type='basesearch',
        )
        self.ctx['shard_resource_id'] = shard_resource.id

        # set attributes for queries and shard that will force TestEnv resource updating
        if self.ctx.get('update_testenv_resources'):
            channel.sandbox.set_resource_attribute(
                queries_resource.id,
                'TE_web_base_prod_simulation_{}'.format(_TIER),
                now_ts_str,
            )
            channel.sandbox.set_resource_attribute(
                shard_resource.id,
                'TE_web_base_prod_simulation_{}'.format(_TIER),
                now_ts_str,
            )

    def _make_delta_plan_from_eventlog(self, output_path):
        bs_instance = {}
        logging.info(
            "Started making plan with time deltas from %s, dump to %s",
            self.tmp_queries_path,
            output_path,
        )
        avg_delta = None
        with open(self.tmp_queries_path) as fr, open(output_path, 'w') as fw:
            prev_timestamp = None
            line_count = 0
            for line in fr:
                if line_count > self.ctx.get(MaxQueries.name, MaxQueries.default_value):
                    break
                line_count += 1
                fields = line.split()
                curr_timestamp = int(fields[0])
                if prev_timestamp is not None:
                    curr_delta = curr_timestamp - prev_timestamp
                    if curr_delta < 0:
                        # system clock shift can produce unordered events (and42@)
                        logging.error(
                            "Get unsorted requests: %s < %s at %s",
                            curr_timestamp, prev_timestamp, line,
                        )
                        continue
                    avg_delta = (avg_delta * (line_count - 1) + int(curr_delta)) / float(line_count)
                    fields[0] = str(curr_delta)
                else:
                    fields[0] = str(0)
                    avg_delta = 0

                # yet another http2 spike
                if fields[1].startswith('http2:'):
                    fields[1] = fields[1].replace('http2', 'http', 1)

                prev_timestamp = curr_timestamp

                fw.write('\t'.join(fields) + '\n')
                if line_count == 1:
                    # parse basesearch host/port
                    purl = urlparse.urlparse(fields[1])
                    # cut FQDN
                    bs_instance["host"] = re.sub(r'\..*', '', purl.hostname)
                    bs_instance["port"] = purl.port

        desired_rps = self.ctx.get(DesiredRPS.name, DesiredRPS.default_value)
        delay_multiplier = 1 / float(desired_rps) * _MICROSECONDS_IN_SECOND / avg_delta
        logging.info("Average delta between queries is %s microseconds", avg_delta)
        logging.info(
            "For {desired_rps} RPS we need multiplier is "
            "1 / {desired_rps} * MICROSECONDS_IN_SECOND / {avg_delta} = {delay_multiplier}".format(
                desired_rps=desired_rps,
                avg_delta=avg_delta,
                delay_multiplier=delay_multiplier,
            )
        )
        logging.info("Obtained basesearch instance {host}:{port}".format(**bs_instance))
        return bs_instance, delay_multiplier

    def _get_eventlog(self):
        instances = ir.get_instances_by_config("jupiter_int", "a_tier_{}".format(_TIER))
        eh.ensure(instances, "No instances for (production_int . a_tier_{})".format(_TIER))
        logging.info('Encountered %s instances', len(instances))
        for instance in random.sample(instances, min(len(instances), 10)):
            # probably if 10 random instances are unavailable, there is no point to check others
            try:
                logging.info('Try to load evlog from instance: {}:{}'.format(*instance))
                self.remote_copy(
                    "rsync://{}/logs/current-eventlog-int-{}".format(*instance),
                    self.eventlog_path,
                    protocol='rsync',
                    create_resource=True,
                    resource_type=resource_types.EVENTLOG_DUMP,
                )
                return
            except Exception:
                logging.info('Ignore exception:\n%s', eh.shifted_traceback())
                continue


__Task__ = GetBasesearchQueriesProdSimulation
