# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.common.types.client import Tag

import sandbox.projects.common.search.components.mkl as sc_mkl
from sandbox.projects import resource_types
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import settings as bss
from sandbox.projects.common import utils
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import params_handler

import sandbox.projects.saas.common.resources as saas_resources


class ExecutableResourceParameter(parameters.LastReleasedResource):
    name = 'executable_resource_id'
    description = 'Executable'
    resource_type = [
        resource_types.BASESEARCH_EXECUTABLE,
        resource_types.PEOPLESEARCH_EXECUTABLE,
        resource_types.VIDEOSEARCH_EXECUTABLE,
        resource_types.NEWS_BASESEARCH_EXECUTABLE,
        resource_types.ADDRESSNIP_SEARCH,
        resource_types.SERPAPI_SEARCH,
        resource_types.RTYSERVER,
    ]


class ConfigResourceParameter(parameters.LastReleasedResource):
    name = 'config_resource_id'
    description = 'Config'
    resource_type = [
        resource_types.SEARCH_CONFIG,
        resource_types.ADDRESSNIP_SEARCH_CONFIG,
        resource_types.VIDEO_SEARCH_CONFIG,
        resource_types.SERPAPI_SEARCH_CONFIG,
        saas_resources.SaasRtyserverConfigs
    ]
    required = False


class SearchDatabaseResourceParameter(parameters.LastReleasedResource):
    name = 'search_database_resource_id'
    description = 'DB'
    resource_type = [
        resource_types.SEARCH_DATABASE,
        resource_types.ADDRESSNIP_SEARCH_DATABASE,
        resource_types.VIDEO_SEARCH_DATABASE,
        resource_types.VIDEO_QUICK_SEARCH_DATABASE,
        resource_types.SERPAPI_SEARCH_DATABASE,
        resource_types.RTYSERVER_SEARCH_DATABASE,
    ]
    required = False


class WizardDataResourceParameter(parameters.LastReleasedResource):
    name = 'wizard_data_resource_id'
    description = 'Wizard data'
    resource_type = resource_types.WIZARD_DATA
    required = False


class ModelsArchiveResourceParameter(parameters.LastReleasedResource):
    name = 'models_archive_resource_id'
    description = 'Models archive'
    resource_type = [
        resource_types.DYNAMIC_MODELS_ARCHIVE_BASE,
        resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE,
        resource_types.DYNAMIC_MODELS_ARCHIVE,  # deprecated for basesearch
    ]
    required = False


class StaticDataResourceParameter(parameters.LastReleasedResource):
    name = 'static_data_resource_id'
    description = 'Static data'
    resource_type = resource_types.RTYSERVER_MODELS
    required = False


class PoliteModeParameter(parameters.SandboxBoolParameter):
    name = 'polite_mode'
    description = 'Polite mode'
    default_value = True


class PatchRequestThreadsParameter(parameters.SandboxBoolParameter):
    name = 'patch_request_threads'
    description = 'Patch request threads'
    default_value = True


class RelevanceThreadsParameter(parameters.SandboxIntegerParameter):
    name = 'relevance_threads'
    description = 'Relevance threads'
    default_value = 0


class BasesearchTypeParameter(parameters.SandboxStringParameter):
    name = 'basesearch_type'
    description = 'Basesearch type'
    choices = [
        ('basesearch', 'basesearch'),
        ('musicsearch', 'musicsearch'),
        ('audiosearch', 'audiosearch'),
        ('ymusic', 'ymusic'),
        ('imgsearch', 'imgsearch'),
        ('videosearch', 'videosearch'),
        ('peoplesearch', 'peoplesearch'),
        ('newssearch', 'newssearch'),
        ('addressnip', 'addressnip'),
        ('quick', 'quick'),
        ('basesearch_samohod', 'basesearch_samohod'),
    ]
    sub_fields = {
        'basesearch': [
            SearchDatabaseResourceParameter.name,
            PatchRequestThreadsParameter.name
        ],
        'ymusic': [
            SearchDatabaseResourceParameter.name,
            WizardDataResourceParameter.name
        ],
        'musicsearch': [
            SearchDatabaseResourceParameter.name
        ],
        'audiosearch': [
            SearchDatabaseResourceParameter.name
        ],
        'imgsearch': [
            SearchDatabaseResourceParameter.name
        ],
        'videosearch': [
            SearchDatabaseResourceParameter.name,
            PatchRequestThreadsParameter.name
        ],
        'peoplesearch': [
            SearchDatabaseResourceParameter.name
        ],
        'newssearch': [
            SearchDatabaseResourceParameter.name
        ],
        'addressnip': [
            SearchDatabaseResourceParameter.name
        ],
        'quick': [
            SearchDatabaseResourceParameter.name,
            StaticDataResourceParameter.name
        ],
        'basesearch_samohod': [
            SearchDatabaseResourceParameter.name,
            PatchRequestThreadsParameter.name
        ],
    }
    default_value = 'basesearch'


class GetFactorNames(parameters.SandboxBoolParameter):
    name = 'get_factor_names'
    description = 'Get factor names'
    default_value = True


class GetBasesearchResponses(SandboxTask):
    """
        Получает и сохраняет результаты обстрела базового поиска.

        **Параметры**

        * **Binary responses** - Сохранять ответы в бинарном виде (protobuf).
            В этом случае выхлоп имеет тип **SEARCH_PROTO_RESPONSES**.
            По умолчанию выдача сохраняется в текстовом виде (с использованием hr=da),
            выхлоп имеет тип **BASESEARCH_HR_RESPONSES**.
        * **Get factor names** (``get_factor_names``) - Сохранять список факторов как ресурс. Нужно для
            таска сравнения ответов, чтобы в дифф-е по факторам указывались их имена, а не номера.
    """

    type = 'GET_BASESEARCH_RESPONSES'

    input_parameters = (
        BasesearchTypeParameter,
        ExecutableResourceParameter,
        ConfigResourceParameter,
        SearchDatabaseResourceParameter,
        WizardDataResourceParameter,
        ModelsArchiveResourceParameter,
        StaticDataResourceParameter,
        PoliteModeParameter,
        PatchRequestThreadsParameter,
        RelevanceThreadsParameter,
        GetFactorNames,
        params_handler.TierName,
        params_handler.QueryType,
    ) + response_saver.create_response_saver_params(required_queries=False).params + threadPool.PARAMS

    execution_space = bss.RESERVED_SPACE
    client_tags = Tag.LINUX_PRECISE

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        response_saver.create_resources(self)

    def on_execute(self):
        out_resource = channel.sandbox.get_resource(int(self.ctx['out_resource_id']))
        if self.ctx[BasesearchTypeParameter.name] == 'basesearch':
            # get the newest TE resource (SEARCH-2352)
            params_handler.set_interrelated_params(
                response_saver.DefaultResponseSaverParams.QueriesParameter,
                "TE_web_base_prod_queries_{}_{}".format(
                    self.ctx.get(params_handler.QueryType.name),
                    self.ctx.get(params_handler.TierName.name)
                ),
                SearchDatabaseResourceParameter,
                "TE_web_base_prod_resources_{}_{}".format(
                    self.ctx.get(params_handler.QueryType.name),
                    self.ctx.get(params_handler.TierName.name)
                )
            )
            params_handler.set_optional_param(ConfigResourceParameter)
            params_handler.set_optional_param(ModelsArchiveResourceParameter)
        basesearch = self._get_basesearch_component()
        sc_mkl.configure_mkl_environment(basesearch)
        patched_queries_resource = response_saver.get_patched_queries_resource(self)

        response_saver.save_responses(
            self.ctx,
            search_component=basesearch,
            responses_resource=out_resource,
            relevance_threads=self.ctx['relevance_threads'],
            patched_queries_resource=patched_queries_resource,
            factor_names=utils.get_or_default(self.ctx, GetFactorNames),
        )

    def _get_basesearch_component(self):
        if self.ctx[BasesearchTypeParameter.name] == 'ymusic':
            return sc.get_ymusicsearch_ex(
                binary_id=self.ctx[ExecutableResourceParameter.name],
                config_id=self.ctx[ConfigResourceParameter.name],
                database_id=self.ctx[SearchDatabaseResourceParameter.name],
                polite_mode=self.ctx[PoliteModeParameter.name])
        elif self.ctx[BasesearchTypeParameter.name] == 'quick':
            return sc.get_fusion_search(
                params=sc.create_fusion_params()
            )
        else:
            component_creator = None
            if self.ctx[BasesearchTypeParameter.name] == 'basesearch_samohod':
                component_creator = sc.BasesearchWebOverSamohod
            return sc.get_basesearch_ex(
                binary_id=self.ctx[ExecutableResourceParameter.name],
                config_id=self.ctx[ConfigResourceParameter.name],
                database_id=self.ctx[SearchDatabaseResourceParameter.name],
                archive_model_id=self.ctx[ModelsArchiveResourceParameter.name],
                polite_mode=self.ctx[PoliteModeParameter.name],
                patch_request_threads=utils.get_or_default(self.ctx, PatchRequestThreadsParameter),
                component_creator=component_creator,
            )

    def get_short_task_result(self):
        if self.is_completed():
            return "unstable" if self.ctx.get(response_saver.UNSTABLE_OUTPUT_KEY) else ""
        return None


__Task__ = GetBasesearchResponses
