import datetime
import logging
import shutil

from sandbox.projects.websearch.basesearch.resources import EventLogDataGrabber
from sandbox.projects.common import dolbilka
from sandbox.projects.common import task_env
from sandbox.projects.common import time_utils as tu
from sandbox.projects.resource_types import BASESEARCH_PLAN, SEARCH_DATABASE, PLAIN_TEXT_QUERIES
from sandbox.common.share import skynet_get
from sandbox.common.types.task import Status
from sandbox import sdk2


class GetShardAndPlan(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=2496330207)

    class Parameters(sdk2.Task.Parameters):
        tiers = sdk2.parameters.List('Tiers', default=['PlatinumTier0'])
        stages = sdk2.parameters.List('Stages', default=['search', 'factors', 'snippets', 'all'])
        tracker_url = sdk2.parameters.String('Tracker url', default='http://vla-tracker-cajuper.n.yandex-team.ru')
        yt_cluster = sdk2.parameters.String('YT eventlogdata cluster', default='hahn')
        eventlog_dir = sdk2.parameters.String('YT eventlogdata directory', default='//home/eventlogdata')
        use_custom_grabber = sdk2.parameters.Bool('Use custom eventlogdata grabber', default=False)
        with use_custom_grabber.value[True]:
            grabber_resource = sdk2.parameters.Resource('YT eventlogdata grabber binary', resource_type=EventLogDataGrabber)

    def on_enqueue(self):
        load_time = tu.date_ymdhm()
        attribute_time = tu.date_ymd(sep='_')
        for tier in self.Parameters.tiers:
            shard_resource = SEARCH_DATABASE(self, '{} shard. {}'.format(tier, load_time), 'basesearch_database', load_time=load_time, tier=tier)
            setattr(self.Context, '{}_shard'.format(tier), shard_resource.id)

            for stage in self.Parameters.stages:
                setattr(shard_resource, 'TE_web_base_prod_resources_{}_{}'.format(stage, tier), attribute_time)

                queries_resource = PLAIN_TEXT_QUERIES(self, '{} {} text plan. {}'.format(tier, stage, load_time), '{}_plan.txt'.format(stage), load_time=load_time, tier=tier, stage=stage)
                setattr(self.Context, '{}_{}_plan_txt'.format(tier, stage), queries_resource.id)
                setattr(queries_resource, 'TE_web_base_prod_queries_{}_{}'.format(stage, tier), attribute_time)
                if stage == 'all':
                    setattr(queries_resource, 'TE_web_base_synt_queries_{}_{}'.format(stage, tier), attribute_time)

                plan_resource = BASESEARCH_PLAN(self, '{} {} plan. {}'.format(tier, stage, load_time), '{}_plan'.format(stage), load_time=load_time, tier=tier, stage=stage)
                setattr(self.Context, '{}_{}_plan'.format(tier, stage), plan_resource.id)
                setattr(plan_resource, 'TE_web_base_prod_queries_{}_{}'.format(stage, tier), attribute_time)

    def on_execute(self):
        import infra.callisto.deploy.tracker.client as client

        grabber_resource = None
        if self.Parameters.use_custom_grabber:
            grabber_resource = self.Parameters.grabber_resource
        else:
            build_task = sdk2.Task['BUILD_EVENT_LOG_DATA_GRABBER'](self, description='Build eventlogdata grabber for task #{}'.format(self.id)).enqueue()
            grabber_resource = sdk2.Resource['EVENT_LOG_DATA_GRABBER'].find(task_id=build_task.id).first()
            raise sdk2.WaitTask([build_task.id], [Status.Group.FINISH])

        for tier in self.Parameters.tiers:
            cmd = [
                str(sdk2.ResourceData(grabber_resource).path),
                '--yt-cluster', self.Parameters.yt_cluster,
                '--token', sdk2.Vault.data(self.owner, 'yt_token_for_testenv'),
                '--tier', tier,
                '--eventlog-dir', self.Parameters.eventlog_dir,
                '--date', datetime.datetime.today().strftime('%Y%m%d'),
                '--plan-file-suffix', '_plan.txt',
                '--shard-file', 'shard.txt']
            sdk2.helpers.subprocess.check_call(cmd)

            shard_name = open('shard.txt').readline()[:-1]
            logging.info('{} shard_name = {}'.format(tier, shard_name))

            db_timestamp = int(shard_name.split('-')[-1])
            logging.info('db_timestamp = {}'.format(db_timestamp))

            for stage in self.Parameters.stages:
                old_file = '{}_plan.txt'.format(stage)
                new_file = str(sdk2.ResourceData(sdk2.Resource[getattr(self.Context, '{}_{}_plan_txt'.format(tier, stage))]).path)
                plan_file = str(sdk2.ResourceData(sdk2.Resource[getattr(self.Context, '{}_{}_plan'.format(tier, stage))]).path)
                shutil.move(old_file, new_file)
                logging.info('moved {} {} plan from {} to {}'.format(tier, stage, old_file, new_file))
                dolbilka.convert_queries_to_plan(new_file, plan_file)
                logging.info('converted {} {} plan to dolbilo format'.format(tier, stage))

            rbtorrent = client.Client(self.Parameters.tracker_url).resolve_one('/web/prod/{}/{}/'.format(db_timestamp, tier), shard_name).rbtorrent
            logging.info('shard\'s rbtorrent = {}'.format(rbtorrent))

            shard_path = str(sdk2.ResourceData(sdk2.Resource[getattr(self.Context, '{}_shard'.format(tier))]).path)
            skynet_get(rbtorrent, shard_path)
            logging.info('got {} shard'.format(tier))
