# -*- coding: utf-8 -*-

import json
import copy
import logging

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.common import rest
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import footers
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import params_handler
from sandbox.projects.common.search import production
from sandbox.projects.common.search import memory_usage
from sandbox.projects.common.search.settings import WebSettings
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.configs.base import BaseCfg
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.websearch import acceptance
from sandbox.projects.websearch.basesearch import WebBasesearchPerformanceParallel as wbpp
from sandbox.projects.websearch.basesearch import constants as bs_const
import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects.common.search.BaseTestSuperMindTask import DisableSuperMindSharedFile as dsmsf

import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.common.sdk_compat.task_helper as rm_th

_response_saver_params = response_saver.create_response_saver_params()
_MEMUSAGE_THRESHOLD = 2.0  # per cent
PERF_MULTIRUN_TASK_KEY = "basesearch_perf_{}_parallel_multirun_task_{}_{}"
DEGRADE_TASK_KEY = "{exe}_basesearch_degrade_task_{arch}_{tier}"
CGI_PARAMS_KEY = "{exe}_basesearch_CGI_params_{arch}_{tier}"
TASK_TYPES = ["old", "new"]
DEFAULT_ARCH = "linux"
IMAGE_ADDRESS = "{}x{}:https://proxy.sandbox.yandex-team.ru/{}"
LINK_FOOTER = '<a href="https://proxy.sandbox.yandex-team.ru/{res_id}">{res_id}</a>'
ARCHS_TO_TEST = [DEFAULT_ARCH]
_DEGRADE_FOOTER = "degrade_footer"
DEGRADE_PLOTS_KEY = "degrade_plots"
_PERF_PARAMS = {
    bs_const.SEARCH_QUERY_TYPE: {
        'dolbilka_executor_max_simultaneous_requests': 10,
    },
    bs_const.FACTORS_QUERY_TYPE: {
        'dolbilka_executor_max_simultaneous_requests': 4,
    },
    bs_const.SNIPPETS_QUERY_TYPE: {
        'dolbilka_executor_max_simultaneous_requests': 4,
    },
}


class PlotImage(sdk2.Resource):
    ttl = 30


class BuildTaskParameter(parameters.TaskSelector):
    name = 'build_task_id'
    description = 'Build task'
    task_type = 'BUILD_BASESEARCH'
    required = True
    group = 'Binaries to test'


class PerfPlanTypes(parameters.SandboxBoolGroupParameter):
    name = "plan_types_for_perf_tests"
    description = "Test performance on queries"
    choices = zip(bs_const.QUERY_TYPES, bs_const.QUERY_TYPES)
    default_value = bs_const.SEARCH_QUERY_TYPE


class TestWithMiddlesearch(parameters.SandboxBoolParameter):
    name = 'test_with_middlesearch'
    description = 'Test configuration with middlesearch and two new basesearches (SEARCH-1044)'
    default_value = True


class TestMiddlesearchMemoryUsage(parameters.SandboxBoolParameter):
    name = 'test_middlesearch_memory_usage'
    description = 'Test middlesearch memory usage with two new basesearches (SEARCH-1081)'
    default_value = True


class TestDegradeLevels(parameters.SandboxBoolParameter):
    name = 'test_degrade_levels'
    description = 'Test basesearch degradation levels (SEARCH-3866)'
    default_value = True


class TestCGIParams(parameters.SandboxBoolParameter):
    name = 'test_cgi_params'
    description = 'Test basesearch CGI params'
    default_value = True


class TestPerfNew(parameters.SandboxBoolParameter):
    name = 'test_perf_new'
    description = 'Test Perf Parallel Multiruns'
    default_value = True  # SEARCH-7798


class TestFusion(parameters.SandboxBoolParameter):
    name = 'test_fusion'
    description = 'Test Fusion'
    default_value = False  # SEARCH-1534


class FtotkFlag(parameters.SandboxBoolParameter):
    name = 'rollout_binaries_to_ftotk'
    description = 'Roll out binary to FTOTK (this flag is looked up by ftotk external scripts)'
    default_value = False


class OldBasesearch(parameters.LastReleasedResource):
    name = 'old_basesearch_resource_id'
    description = 'Old basesearch'
    resource_type = [resource_types.BASESEARCH_EXECUTABLE]
    do_not_copy = True
    group = 'Binaries to test'


class OldBasesearchSource(parameters.SandboxStringParameter):
    name = "old_basesearch_source"
    description = 'Old basesearch source'
    required = True
    choices = [
        ('from production', 'from_production'),
        ('last released', 'last_released'),
        ('existing', 'existing')
    ]
    sub_fields = {
        'existing': [
            OldBasesearch.name,
        ]
    }
    default_value = 'from_production'
    group = 'Binaries to test'


class RevisionTested(parameters.SandboxStringParameter):
    name = "revision_tested"
    description = "Revision tested"


@rm_notify.notify2()
class PriemkaBasesearchBinary(bugbanner.BugBannerTask):
    """
        Приёмка базового поиска на производительность.
    """
    type = 'PRIEMKA_BASESEARCH_BINARY'

    client_tags = task_env.TaskTags.startrek_client
    cores = 1

    input_parameters = [
        PerfPlanTypes,
        TestPerfNew,
        TestWithMiddlesearch,
        TestMiddlesearchMemoryUsage,
        TestDegradeLevels,
        TestCGIParams,
        TestFusion,
        BuildTaskParameter,
        OldBasesearchSource,
        OldBasesearch,
        FtotkFlag,
        RevisionTested,
    ]

    environment = [
        task_env.TaskRequirements.startrek_client,
        PipEnvironment("matplotlib", '1.5.1', use_wheel=True),
    ]
    default_performance_task_cpu_model = {
        DEFAULT_ARCH: 'e5-2650',
    }

    @property
    def footer(self):
        client = rest.Client()
        foot = []
        foot.extend(self._get_memusage_foot(client))
        foot.extend(self._get_degrade_footer())
        foot.extend(self._get_new_perf_foot(client))
        return foot

    def _get_new_perf_foot(self, client):
        new_perf_foots = []
        if not utils.get_or_default(self.ctx, TestPerfNew):
            return new_perf_foots
        for tier in _BASESEARCH_TIERS:
            for q_type in self._perf_plan_types:
                new_perf_task_ctx_key = PERF_MULTIRUN_TASK_KEY.format(q_type, DEFAULT_ARCH, tier.name)
                task_id = self.ctx.get(new_perf_task_ctx_key)
                if task_id:
                    task_foot = client.task[task_id].custom.footer.read()[0]["content"].values()[0]
                    new_perf_foots.append({
                        "content": {
                            "<h3>{} {} performance multiruns</h3>".format(tier.name, q_type): task_foot
                        }
                    })
        return new_perf_foots

    def _get_degrade_footer(self):
        try:
            foot = [{
                "content": {
                    "<h3>Test degrade footer:</h3>": "<br/>".join(self.ctx[_DEGRADE_FOOTER])
                }
            }, ]
        except Exception:
            foot = []
        return foot

    def _get_memusage_foot(self, client):
        if not utils.get_or_default(self.ctx, TestMiddlesearchMemoryUsage):
            return []
        return [footers.memusage_footer(self._memusage_task_ids(), client)]

    def _memusage_task_ids(self):
        return [self.ctx.get("memusage_task_{}_id".format(n + 1)) for n in xrange(2)]

    @property
    def _perf_plan_types(self):
        return utils.get_or_default(self.ctx, PerfPlanTypes).split()

    def on_execute(self):
        self.ctx[rm_const.COMPONENT_CTX_KEY] = BaseCfg.name
        if _DEGRADE_FOOTER not in self.ctx:
            self.ctx[_DEGRADE_FOOTER] = []
        self.add_bugbanner(bugbanner.Banners.WebBaseSearch)
        if not self.list_subtasks():
            self.check_test_parameters()

            first_base, second_base = self.get_all_bs_resources()

            if utils.get_or_default(self.ctx, TestWithMiddlesearch) and not self.ctx.get('test_configuration_task_id'):
                self.ctx['test_configuration_task_id'] = self._test_configuration_with_middle()

            self.test_middle_memusage(first_base, second_base)

            self.test_info_requests()
            self.perf_test_new(first_base, second_base)
            self.test_degrade_levels()
            self.test_cgi()
            if not self.ctx.get('fusion_test_task_id') and utils.get_or_default(self.ctx, TestFusion):
                self.ctx['fusion_test_task_id'] = self._create_fusion_tests()

            self.comment_in_st()
            utils.wait_all_subtasks_stop()
        elif not utils.check_all_subtasks_done():
            utils.restart_broken_subtasks()
        else:
            self.draw_degrade_info()
            self.comment_results_to_st()
            utils.check_subtasks_fails(stop_on_broken_children=True)

        logging.debug("Task context: %s", self.ctx)
        self.check_memusage()

    def test_info_requests(self):
        for arch_to_test in ARCHS_TO_TEST:
            base_exe = self._get_base_exe("new", arch_to_test)
            self._shoot_info_reqs(base_exe, "new")

    def test_cgi(self):
        if utils.get_or_default(self.ctx, TestCGIParams):
            for arch_to_test in ARCHS_TO_TEST:
                base_exe = self._get_base_exe("new", arch_to_test)
                for tier in _BASESEARCH_TIERS:
                    ctx_key = CGI_PARAMS_KEY.format(exe="new", arch=arch_to_test, tier=tier.name)
                    if not self.ctx.get(ctx_key):
                        self.ctx[ctx_key] = self._create_test_basesearch_cgi_params(base_exe, tier)

    def test_degrade_levels(self):
        if utils.get_or_default(self.ctx, TestDegradeLevels):
            for arch_to_test in ARCHS_TO_TEST:
                for exe_gen in TASK_TYPES:
                    base_exe = self._get_base_exe(exe_gen, arch_to_test)
                    for tier in _BASESEARCH_TIERS:
                        ctx_key = DEGRADE_TASK_KEY.format(exe=exe_gen, arch=arch_to_test, tier=tier.name)
                        if not self.ctx.get(ctx_key):
                            self.ctx[ctx_key] = self._test_degrade_levels(base_exe, arch_to_test, tier)

    def perf_test_new(self, first_base, second_base):
        if not utils.get_or_default(self.ctx, TestPerfNew):
            return
        for tier in _BASESEARCH_TIERS:
            for q_type in self._perf_plan_types:
                new_perf_task_ctx_key = PERF_MULTIRUN_TASK_KEY.format(q_type, DEFAULT_ARCH, tier.name)
                sub_ctx = {
                    wbpp.BASESEARCH1_PARAMS.Binary.name: first_base.executable,
                    wbpp.BASESEARCH2_PARAMS.Binary.name: second_base.executable,
                    wbpp.BASESEARCH1_PARAMS.ArchiveModel.name: first_base.models_base,
                    wbpp.BASESEARCH2_PARAMS.ArchiveModel.name: second_base.models_base,
                    wbpp.BASESEARCH1_PARAMS.Database.name: self._get_basesearch_database_id(tier, q_type),
                    wbpp.BASESEARCH2_PARAMS.Database.name: self._get_basesearch_database_id(tier, q_type),
                    wbpp.BASESEARCH1_PARAMS.Config.name: first_base.get_cfg(tier.name),
                    wbpp.BASESEARCH2_PARAMS.Config.name: second_base.get_cfg(tier.name),
                    wbpp.BASESEARCH1_PARAMS.PoliteMode.name: False,
                    wbpp.BASESEARCH2_PARAMS.PoliteMode.name: False,
                    dsmsf.name: True,
                    'number_of_runs': 25,
                    'dolbilo_plan1_resource_id': self._get_basesearch_plan_id(tier, q_type),
                    'dolbilo_plan2_resource_id': self._get_basesearch_plan_id(tier, q_type),
                    'dolbilka_executor_requests_limit': 200000,
                    'dolbilka_executor_mode': 'finger',
                    'dolbilka_executor_max_simultaneous_requests': 10,
                    'notify_via': '',
                }
                sub_ctx.update(_PERF_PARAMS.get(q_type))
                self.ctx[new_perf_task_ctx_key] = self.create_subtask(
                    task_type='WEB_BASESEARCH_PERFORMANCE_MULTIRUNS',
                    description="Performance parallel multirun on {} plan for {}".format(q_type, tier.name),
                    input_parameters=sub_ctx,
                ).id

    def comment_results_to_st(self):
        try:
            rm_token = rm_sec.get_rm_token(self)
            st_helper = STHelper(token=rm_token)
            text = '{}\n{}'.format(
                self._new_perf_info_for_st(), self._degrade_info_for_st(),
            )
            st_helper.comment_results_to_st(self.ctx["st_issue_key"], self.ctx.get("st_comment_id", 0), text)
        except Exception:
            logging.info("Can't update comment: %s", eh.shifted_traceback())

    def comment_in_st(self):
        old_res_id = self.ctx.get(OldBasesearch.name)
        old_task_id = rm_th.resource_obj(old_res_id).task_id if old_res_id else None
        acceptance.notify_st_on_acceptance(
            self, rmc.get_component(BaseCfg.name),
            new_build_task_id=self.ctx[BuildTaskParameter.name],
            old_build_task_id=old_task_id,
            rev=self.ctx.get(RevisionTested.name),
        )

    def check_memusage(self):
        if not utils.get_or_default(self.ctx, TestMiddlesearchMemoryUsage):
            logging.info("Memory usage isn't tested here. Nothing to check")
            return
        client = rest.Client()
        old_mem_id, new_mem_id = self._memusage_task_ids()
        new_mem = footers.get_task_fields(client, new_mem_id, "context.memory_bytes").get("context.memory_bytes")
        old_mem = footers.get_task_fields(client, old_mem_id, "context.memory_bytes").get("context.memory_bytes")
        if new_mem is None or old_mem is None:
            eh.check_failed("Unable to check memory usage: no info from subtasks")
        for i in ["rss", "vms", "uss", "pss", "anon", "shared", "vmlck"]:
            diff = 100 * (new_mem[0][i] - old_mem[0][i]) / float(old_mem[0][i])
            failed = coloring.get_color_diff(diff, _MEMUSAGE_THRESHOLD) == coloring.DiffColors.bad
            eh.ensure(not failed, "Limit of '{}' exceeded!".format(i))
            logging.info("%s usage for is OK", i)

    def _test_configuration_with_middle(self):
        """
            Создать задачу GET_MIDDLESEARCH_RESPONSES.
            Проверяет, нормально ли работает базовый для всех типов запросов
            :return: идентификатор созданной задачи
        """
        logging.info('Create task GET_MIDDLESEARCH_RESPONSES')
        basesearch_resource_id = apihelpers.list_task_resources(
            self.ctx.get(BuildTaskParameter.name),
            'BASESEARCH_EXECUTABLE',
            arch=DEFAULT_ARCH
        )[0].id
        last_stable_models_base_res = apihelpers.get_last_released_resource(
            resource_types.DYNAMIC_MODELS_ARCHIVE_BASE
        )
        last_stable_models_res = apihelpers.get_last_resource_with_attribute(
            resource_type=resource_types.DYNAMIC_MODELS_ARCHIVE,
            attribute_name="current_production",
        )

        sub_ctx = {
            _response_saver_params.QueriesParameter.name: apihelpers.get_last_resource_with_attribute(
                resource_type=ms_resources.WebMiddlesearchPlainTextQueries,
                attribute_name=WebSettings.testenv_middle_resources_attr_name("mmeta", "reqs"),
            ).id,
            wbpp.BASESEARCH1_PARAMS.Binary.name: basesearch_resource_id,
            wbpp.BASESEARCH2_PARAMS.Binary.name: basesearch_resource_id,
            wbpp.BASESEARCH1_PARAMS.Config.name: self._get_basesearch_config_id(_BASESEARCH_TIERS[0]),
            wbpp.BASESEARCH2_PARAMS.Config.name: self._get_basesearch_config_id(_BASESEARCH_TIERS[-1]),
            wbpp.BASESEARCH1_PARAMS.Database.name: self._get_basesearch_database_id(_BASESEARCH_TIERS[0]),
            wbpp.BASESEARCH2_PARAMS.Database.name: self._get_basesearch_database_id(_BASESEARCH_TIERS[-1]),
            wbpp.BASESEARCH1_PARAMS.ArchiveModel.name: last_stable_models_base_res.id,
            wbpp.BASESEARCH2_PARAMS.ArchiveModel.name: last_stable_models_base_res.id,
            sc.DefaultMiddlesearchParams.ArchiveModel.name: last_stable_models_res.id,
            sc.DefaultMiddlesearchParams.Binary.name: apihelpers.get_last_released_resource(
                resource_type=ms_resources.RankingMiddlesearchExecutable,
                arch=DEFAULT_ARCH,
            ).id,
            sc.DefaultMiddlesearchParams.Data.name: apihelpers.get_last_resource_with_attribute(
                resource_type=resource_types.MIDDLESEARCH_DATA,
                attribute_name=WebSettings.testenv_middle_resources_attr_name("mmeta", "data"),
                arch=DEFAULT_ARCH,
            ).id,
            sc.DefaultMiddlesearchParams.Config.name: apihelpers.get_last_resource_with_attribute(
                resource_type=resource_types.MIDDLESEARCH_CONFIG,
                attribute_name=WebSettings.testenv_middle_cfg_attr_name("jupiter_mmeta"),
                arch=DEFAULT_ARCH,
            ).id,
            'notify_via': '',
            'notify_if_failed': self.owner,
            'process_count': 4,  # do not overshoot basesearch (SEARCH-1044)
        }

        description = "{}, test basesearch with all types of queries".format(self.descr)

        return self.create_subtask(
            task_type='GET_MIDDLESEARCH_RESPONSES',
            description=description,
            input_parameters=sub_ctx,
            arch=DEFAULT_ARCH
        ).id

    def _test_degrade_levels(self, base_exe, arch, tier):
        """
            Создать задачу TEST_BASESEARCH_EXPERIMENT, проверяющую производительность
            базового поиска в разных режимах supermind-ручки.
            Сделано в рамках SEARCH-3866.
            :return: идентификатор созданной задачи.
        """
        logging.info('Create task TEST_BASESEARCH_EXPERIMENT')
        basesearch_resource_id = base_exe.id
        last_stable_models_base_res = apihelpers.get_last_released_resource(
            resource_types.DYNAMIC_MODELS_ARCHIVE_BASE
        )
        new_cgi_params_str = ""
        degrade_level_steps = 10
        for smm in xrange(degrade_level_steps, -1, -1):
            new_cgi_params_str += "&pron=smm_{}\n".format(smm / float(degrade_level_steps))

        sub_ctx = {
            "basesearch_executable_resource_id": basesearch_resource_id,
            "models_archive_resource_id": last_stable_models_base_res.id,
            "dolbilo_plan_resource_id": self._get_basesearch_plan_id(tier),
            "basesearch_config_resource_id": self._get_basesearch_config_id(tier),
            "new_cgi_params": new_cgi_params_str,
            "test_baseline_performance": False,
            "test_memory_usage": False,
            "tier": tier.name,
            'notify_via': '',
            'notify_if_failed': self.owner,
        }
        logging.debug("TEST_BASESEARCH_EXPERIMENT context: %s", json.dumps(sub_ctx))
        description = "{}, basesearch degrade levels test for {}".format(self.descr, tier.name)

        return self.create_subtask(
            task_type='TEST_BASESEARCH_EXPERIMENT',
            description=description,
            input_parameters=sub_ctx,
            arch=arch,
        ).id

    def test_middle_memusage(self, first_base, second_base):
        """
            Создать задачи MEASURE_MIDDLESEARCH_MEMORY_USAGE
            они получают значения rss (Resident Set Size) и vsz (Virtual Memory Size)
            для конфигурации из среднего и двух базовых
            :return: идентификаторы созданных задач
        """
        if (
            utils.get_or_default(self.ctx, TestMiddlesearchMemoryUsage) and
            not self.ctx.get('memusage_task_1_id') and
            not self.ctx.get('memusage_task_2_id')
        ):
            logging.info('Create tasks MEASURE_MIDDLESEARCH_MEMORY_USAGE')

            middle = _Middlesearch(utils.get_and_check_last_released_resource_id(
                resource_type=ms_resources.RankingMiddlesearchExecutable,
                arch=DEFAULT_ARCH,
            ))
            middle.config = utils.get_and_check_last_resource_with_attribute(
                resource_type=resource_types.MIDDLESEARCH_CONFIG,
                attr_name=WebSettings.testenv_middle_cfg_attr_name("jupiter_mmeta")
            ).id
            middle.data = utils.get_and_check_last_resource_with_attribute(
                resource_type=resource_types.MIDDLESEARCH_DATA,
                attr_name=WebSettings.testenv_middle_resources_attr_name("mmeta", "data")
            ).id
            middle.models = utils.get_and_check_last_resource_with_attribute(
                resource_type=resource_types.DYNAMIC_MODELS_ARCHIVE,
                attr_name="current_production",
            ).id
            middle.plan = utils.get_and_check_last_resource_with_attribute(
                resource_type=ms_resources.MiddlesearchPlan,
                attr_name=WebSettings.testenv_middle_resources_attr_name("mmeta", "plan"),
            ).id
            middle.verify_stderr = False

            self.ctx['memusage_task_1_id'], self.ctx['memusage_task_2_id'] = memory_usage.compare_memory_usage(
                first_base, middle,
                second_basesearch=second_base,
                description=self.descr,
            )

    def get_all_bs_resources(self):
        first_base = _Basesearch(self._get_old_bs_resource(DEFAULT_ARCH).id)
        first_base.config1 = self._get_basesearch_config_id(_BASESEARCH_TIERS[0])
        first_base.config2 = self._get_basesearch_config_id(_BASESEARCH_TIERS[-1])
        first_base.database1 = self._get_basesearch_database_id(_BASESEARCH_TIERS[0])
        first_base.database2 = self._get_basesearch_database_id(_BASESEARCH_TIERS[-1])
        first_base.models_base = utils.get_and_check_last_released_resource_id(
            resource_types.DYNAMIC_MODELS_ARCHIVE_BASE
        )
        second_base = copy.copy(first_base)
        second_base.executable = apihelpers.get_task_resource_id(
            self.ctx.get(BuildTaskParameter.name),
            resource_types.BASESEARCH_EXECUTABLE,
            arch=DEFAULT_ARCH,
        )
        return first_base, second_base

    def _create_test_basesearch_cgi_params(self, base_exe, tier):
        """
            Создать таск для проверки базового поиска с cgi-параметрами
            :param base_exe: объект ресурса бинарника базового поиска
            :return: идентификатор созданного таска
        """
        description = "basesearch cgi_params test for {} [{}, {}]".format(
            utils.get_binary_version(base_exe), tier.name, base_exe.arch
        )
        logging.info("CREATE TASK: %s", description)
        models_res_id = utils.get_and_check_last_released_resource_id(resource_types.DYNAMIC_MODELS_ARCHIVE_BASE)

        get_queries_task = self.create_subtask(
            task_type='GET_QUERIES_FROM_PLAN',
            input_parameters={
                "plan_resource_id": self._get_basesearch_plan_id(tier)
            },
            description="get queries from plan for {} tier".format(tier.name)
        )

        queries_resource_id = get_queries_task.ctx['out_resource_id']

        sub_ctx = {
            sc.DefaultBasesearchParams.Binary.name: base_exe.id,
            sc.DefaultBasesearchParams.Config.name: self._get_basesearch_config_id(tier),
            sc.DefaultBasesearchParams.Database.name: self._get_basesearch_database_id(tier),
            sc.DefaultBasesearchParams.ArchiveModel.name: models_res_id,
            sc.DefaultBasesearchParams.PoliteMode.name: False,
            'queries_resource_id': queries_resource_id,
            'parse_and_check_response': False,
            'max_queries': 50,
            'max_fetch_queries': 50,
            'response_timeout': 20000000,  # will be ignored when use_dolbilka = True
            'save_cgi_info': False,
            'save_testing_queries': False,
            'use_dolbilka': True,
            'use_multiprocessing': False,
            'process_count': 1,
        }
        return self.create_subtask(
            task_type='TEST_BASESEARCH_CGI_PARAMS',
            # ??model=self.default_performance_task_cpu_model[base_exe.arch],
            description=description,
            input_parameters=sub_ctx,
            arch=base_exe.arch
        ).id

    def _get_base_exe(self, base_descr, arch):
        if base_descr == "old":
            return self._get_old_bs_resource(arch)
        elif base_descr == "new":
            return apihelpers.list_task_resources(
                self.ctx.get(BuildTaskParameter.name),
                resource_types.BASESEARCH_EXECUTABLE,
                arch=arch
            )[0]
        eh.check_failed("Unknown basesearch description: {}".format(base_descr))

    def _get_old_bs_resource(self, arch):
        old_bs_source = self.ctx.get(OldBasesearchSource.name, 'from_production')
        if old_bs_source == 'from_production':
            binary = production.get_prod_binary_nanny(resource_types.BASESEARCH_EXECUTABLE, 'vla_web_tier1_base', arch)
            eh.verify(binary, "Failed to get prod binary")
            return binary
        elif old_bs_source == 'last_released':
            return apihelpers.get_last_released_resource(
                resource_types.BASESEARCH_EXECUTABLE,
                arch=arch,
            )
        else:
            return channel.sandbox.get_resource(self.ctx.get(OldBasesearch.name))

    def _get_basesearch_configs(self):
        get_cfg_task = sdk2.Task['GET_BASESEARCH_CONFIG_2']
        get_cfg_task_id = get_cfg_task(
            get_cfg_task.current,
            description='Get basesearch production config for priemka basesearch task #{}'.format(self.id),
            owner=self.owner,
            priority=self.priority,
            db_types=[tier.name for tier in _BASESEARCH_TIERS]
        ).enqueue().id
        enqueued_cfg_task = sdk2.Task[get_cfg_task_id]
        for tier in _BASESEARCH_TIERS:
            self.ctx[tier.config_key()] = getattr(
                enqueued_cfg_task.Context, 'search_config_for_{}'.format(tier.name)
            )

    def _get_plans_and_dbs(self):
        for tier in _BASESEARCH_TIERS:
            for q_type in self._perf_plan_types:
                plan_key = tier.plan_key(q_type)
                db_key = tier.database_key(q_type)
                if plan_key not in self.ctx or db_key not in self.ctx:
                    db, plan = params_handler.get_last_interrelated_resources(
                        resource_types.SEARCH_DATABASE, tier.shard_res_attr_name(q_type),
                        resource_types.BASESEARCH_PLAN, tier.plan_res_attr_name(q_type),
                    )
                    self.ctx[db_key] = db.id
                    self.ctx[plan_key] = plan.id

    def _get_basesearch_config_id(self, tier):
        return self.ctx[tier.config_key()]

    def _get_basesearch_plan_id(self, tier, q_type=bs_const.SEARCH_QUERY_TYPE):
        return self.ctx[tier.plan_key(q_type)]

    def _get_basesearch_database_id(self, tier, q_type=bs_const.SEARCH_QUERY_TYPE):
        return self.ctx[tier.database_key(q_type)]

    def check_test_parameters(self):
        """
            Проверяем наличие всех нужных для задачи ресурсов: конфиги, планы, шарды
        """
        self._get_basesearch_configs()
        self._get_plans_and_dbs()

    def _create_fusion_tests(self):
        new_fusion_binary = apihelpers.list_task_resources(
            self.ctx.get(BuildTaskParameter.name),
            'RTYSERVER_EXECUTABLE',
            arch=DEFAULT_ARCH
        )
        eh.ensure(new_fusion_binary, "Cannot find RTYSERVER_EXECUTABLE")

        if self.ctx.get(OldBasesearchSource.name, 'from_production') == 'from_production':
            old_fusion_binary = apihelpers.get_last_released_resource(
                resource_types.RTYSERVER_EXECUTABLE,
                arch=DEFAULT_ARCH,
            )
        else:
            old_fusion_binary = channel.sandbox.get_resource(self.ctx.get(OldBasesearch.name))

        task_description = "test new fusion binary for {0} [new]".format(
            utils.get_binary_version(new_fusion_binary[0])
        )

        task_id = self.create_subtask(
            task_type='RUN_FUSION_ACCEPTANCE',
            description=task_description,
            input_parameters={
                sc.OldFusionBinary.name: old_fusion_binary.id,
                sc.NewFusionBinary.name: new_fusion_binary[0].id,
                'notify_via': 'email',
                'notify_if_failed': 'czech,fusion-releases@yandex-team.ru',
                'notify_if_finished': 'czech,fusion-releases@yandex-team.ru',
            },
            arch=DEFAULT_ARCH
        ).id

        return task_id

    def _shoot_info_reqs(self, base_exe, exe_gen):
        """
            SEARCH-1346
        """
        ctx_key = '{}_basesearch_functional_task_{}'.format(exe_gen, base_exe.arch)
        if not self.ctx.get(ctx_key):
            sub_ctx = {
                'notify_via': '',
                'notify_if_failed': self.owner,
                "search_type": "base",
                "search_subtype": "web",
                sc.DefaultBasesearchParams.Binary.name: base_exe.id,
                sc.DefaultBasesearchParams.Config.name: self._get_basesearch_config_id(_BASESEARCH_TIERS[0]),
                # any tier is suitable
                sc.DefaultBasesearchParams.Database.name: self._get_basesearch_database_id(_BASESEARCH_TIERS[0])
            }
            self.ctx[ctx_key] = self.create_subtask(
                task_type="FUNCTIONAL_TEST_SEARCH",
                description="Test simple requests to check {} binary".format(exe_gen),
                input_parameters=sub_ctx,
                arch=base_exe.arch
            ).id
        return self.ctx[ctx_key]

    def _new_perf_info_for_st(self):
        cmp_info = []
        try:
            for tier in _BASESEARCH_TIERS:
                for q_type in self._perf_plan_types:
                    perf_task_id = self.ctx[PERF_MULTIRUN_TASK_KEY.format(q_type, DEFAULT_ARCH, tier.name)]
                    perf_task = rm_th.task_obj(perf_task_id)
                    baseline_median_rps = rm_th.ctx_field(perf_task, "median_rps_1", 0)
                    test_median_rps = rm_th.ctx_field(perf_task, "median_rps_2", 0)
                    diff = rm_th.ctx_field(perf_task, "diff_per_cent_median", 0)
                    diff_color = coloring.get_color_diff(diff, max_diff=-1.2)
                    task_link = 'https://sandbox.yandex-team.ru/task/{}/view {}'.format(perf_task_id, perf_task_id)
                    cmp_info.append(
                        "**{} {}**\n  Baseline median rps = **!!{:.2f}!!**\n  Test median rps = **!!{:.2f}!!**\n  Diff,% = **!!({}){:.2f} %!!**\n  Task : (({}))".format(
                            tier.name, q_type, baseline_median_rps, test_median_rps, diff_color, diff, task_link
                        )
                    )

        except Exception:
            logging.info("Unable to get new perf info for startrek: %s", eh.shifted_traceback())
        cmp_info = "\n".join(cmp_info)
        logging.debug("Compare perf info: %s", cmp_info)
        return cmp_info

    def _degrade_info_for_st(self):
        try:
            if utils.get_or_default(self.ctx, TestDegradeLevels):
                result = ["<{Degrade levels"]
                for tier in _BASESEARCH_TIERS:
                    resource_id = self.ctx.get(DEGRADE_PLOTS_KEY, {}).get(tier.name)
                    if resource_id:
                        result.append(IMAGE_ADDRESS.format(500, 300, resource_id))
                result.append("}>")
                return "\n".join(result)
        except Exception as e:
            eh.log_exception("Unable to send degrade levels info", e)

        return ""

    def draw_degrade_info(self):
        if not utils.get_or_default(self.ctx, TestDegradeLevels):
            return
        from matplotlib import pyplot as plt
        for tier in _BASESEARCH_TIERS:
            degrade_info = self._count_degrade_info(DEFAULT_ARCH, tier)
            rps = {}
            for task_type in TASK_TYPES:
                x, y = zip(*degrade_info[task_type])
                rps[task_type] = map(lambda val: max(val, 1), y)
                plt.plot(x, y, label=task_type)
                plt.xlabel("pron=smm")
                plt.ylabel("rps")
                plt.title("{}".format(tier.name))
                plt.legend()
            rps_diffs = map(lambda a, b: abs(a - b) / b, rps["old"], rps["new"])
            threshold = 0.3
            if max(rps_diffs) > threshold:
                self.set_info("Degrade level for {} looks strange: Max rps diff = {} > {}".format(
                    tier.name, max(rps_diffs), threshold
                ))
            plot_name = "{}_degrade_level_plot".format(tier.name)
            resource = self.create_resource(plot_name, "{}.png".format(plot_name), PlotImage)
            plt.savefig(resource.path)
            plt.close()
            self.ctx.setdefault(DEGRADE_PLOTS_KEY, {})[tier.name] = resource.id
            self.ctx[_DEGRADE_FOOTER].append(LINK_FOOTER.format(res_id=resource.id))
            self.mark_resource_ready(resource.id)

    def _count_degrade_info(self, arch_to_test, tier):
        degrade_info_tier = {}
        for exe_gen in TASK_TYPES:
            degrade_ctx = channel.sandbox.get_task(
                self.ctx[DEGRADE_TASK_KEY.format(exe=exe_gen, arch=arch_to_test, tier=tier.name)]
            ).ctx

            exp_data = json.loads(degrade_ctx.get('exp_data', '{}'))
            exp_count = len(exp_data) - 1

            for exp_index in range(exp_count):
                exp = exp_data[str(exp_index)]
                cgi_params = exp['cgi_params']
                exp_rps = int(float(exp['rps']) * 10) / 10.0
                degrade_info_tier.setdefault(exe_gen, []).append((float(cgi_params[-3:]), exp_rps))
        logging.debug("Degrade info for %s:\n%s", tier.name, json.dumps(degrade_info_tier, indent=2))
        return degrade_info_tier


__Task__ = PriemkaBasesearchBinary


class _DataTier:
    def __init__(self, name):
        self.name = name

    def config_key(self):
        return 'basesearch_config_{}'.format(self.name)

    def plan_key(self, q_type):
        return 'basesearch_plan_{}_{}'.format(self.name, q_type)

    def database_key(self, q_type):
        return 'basesearch_database_{}_{}'.format(q_type, self.name)

    @property
    def db_type_attr_name(self):
        return 'db_type'

    @property
    def db_type_attr_value(self):
        return 'basesearch'

    def shard_res_attr_name(self, q_type):
        return 'TE_web_base_prod_resources_{}_{}'.format(q_type, self.name)

    @property
    def shard_res_attr_value(self):
        return None

    def plan_res_attr_name(self, q_type):
        return "TE_web_base_prod_queries_{}_{}".format(q_type, self.name)


class _Middlesearch(object):
    def __init__(self, executable):
        self.executable = executable


class _Basesearch(object):
    INDEX = {
        'PlatinumTier0': 2,
        'WebTier1': 3,
    }

    def __init__(self, executable):
        self.executable = executable

    def get_cfg(self, tier_name):
        return getattr(self, "config{}".format(self.INDEX[tier_name]))


_BASESEARCH_TIERS_NAMES = ["PlatinumTier0"]
_BASESEARCH_TIERS = [_DataTier(name) for name in _BASESEARCH_TIERS_NAMES]
