import logging
import itertools
import json

import sandbox.sandboxsdk.parameters as sp
import sandbox.sandboxsdk.environments as env
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search.basesearch import task as search_task
from sandbox.projects.common import stats as mst
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.websearch.basesearch import constants as bs_const
from sandbox.projects import resource_types

BASESEARCH1_PARAMS = sc.create_basesearch_params(n=1)
BASESEARCH2_PARAMS = sc.create_basesearch_params(n=2)
_FIRST = "#1"
_SECOND = "#2"
_THIRD = "#1 (retest)"  # https://st.yandex-team.ru/SEARCH-4025#1507806917000
DB_TYPES = ["platinum", "web0", "web1"]
COMBINATIONS = ["{}_{}".format(*i) for i in list(itertools.product(DB_TYPES, bs_const.QUERY_TYPES))] + ["unknown"]


class Plan1Parameter(sp.ResourceSelector):
    name = 'dolbilo_plan1_resource_id'
    description = 'Plan for basesearch #1'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


class Plan2Parameter(sp.ResourceSelector):
    name = 'dolbilo_plan2_resource_id'
    description = 'Plan for basesearch #2'
    resource_type = resource_types.BASESEARCH_PLAN
    required = True


class RunType(sp.SandboxStringParameter):
    """
        Type of running task
    """
    name = 'run_type'
    description = "Run type"
    choices = zip(COMBINATIONS, COMBINATIONS)
    default_value = "unknown"


class WebBasesearchPerformanceParallel(
    search_performance.NewShootingTask,
    search_task.BasesearchComponentTask
):
    """
        Parallel version of basesearch performance task.
        1) Run first basesearch (to map data into memory)
        2) Run second basesearch, then run first basesearch again and compare their results (diff_retest in footer)
    """
    type = 'WEB_BASESEARCH_PERFORMANCE_PARALLEL'
    client_tags = search_performance.NewShootingTask.client_tags & search_task.BasesearchComponentTask.client_tags
    execution_space = 110 * 1024
    required_ram = 96 * 1024
    input_parameters = (
        (RunType,) +
        BASESEARCH1_PARAMS.params + (Plan1Parameter,) +
        BASESEARCH2_PARAMS.params + (Plan2Parameter,) +
        search_task.BasesearchComponentTask.basesearch_input_parameters +
        search_performance.NewShootingTask.shoot_input_parameters
    )
    new_stat_types = search_performance.NewShootingTask.new_stats_types + (
        ("shooting.latency_0.95", "Latency P95", "{:0.2f}"),
    )

    def on_execute(self):
        search_task.BasesearchComponentTask.on_execute(self)
        bs1 = self._basesearch(BASESEARCH1_PARAMS)
        bs2 = self._basesearch(BASESEARCH2_PARAMS)

        self._init_virtualenv()
        search_performance.NewShootingTask._dolbilo_shoot(self, bs1, self.ctx[Plan1Parameter.name], _FIRST)
        search_performance.NewShootingTask._dolbilo_shoot(self, bs2, self.ctx[Plan2Parameter.name], _SECOND)
        search_performance.NewShootingTask._dolbilo_shoot(self, bs1, self.ctx[Plan1Parameter.name], _THIRD)

        stats = self.ctx[self.new_stats_key]
        for key in stats[_FIRST]:
            if self._in_stats(key):
                stats.setdefault("diff", {})[key] = search_performance.delta_percent(
                    stats[_FIRST][key],
                    stats[_SECOND][key]
                )
                stats.setdefault("diff_retest", {})[key] = search_performance.delta_percent(
                    stats[_THIRD][key],
                    stats[_SECOND][key]
                )
                stats.setdefault("diff_same", {})[key] = search_performance.delta_percent(
                    stats[_FIRST][key],
                    stats[_THIRD][key]
                )
        rps50_diff = float(stats["diff_retest"].get("shooting.rps_0.5", 0))
        latency50_diff = float(stats["diff_retest"].get("shooting.latency_0.5", 0))
        self.ctx["rps50_diff"] = rps50_diff
        self.ctx["latency50_diff"] = latency50_diff
        self.ctx["significant_diff"] = abs(rps50_diff) > 1.2 and abs(latency50_diff) > 0.8
        self.ctx[self.bold_stats_key] = "diff_retest"
        self._save_run_data_to_yt(stats)

    def _save_run_data_to_yt(self, result):
        try:
            perf_data = {
                "run_sandbox_task_id": self.id,
                "run_sandbox_task_host": self.host,
                "testenv_database": self.ctx.get("testenv_database", "-"),
                "run_type": utils.get_or_default(self.ctx, RunType),
                "plan_resource_ids": [self.ctx[Plan1Parameter.name], self.ctx[Plan2Parameter.name]]
            }
            perf_data.update({
                k: {_convert_to_yt_compatible(i): j for i, j in v.iteritems()} for k, v in result.iteritems()
            })
            perf_data_path = "perf_data.json"
            with open(perf_data_path, "w") as f:
                json.dump(perf_data, f)
            with env.VirtualEnvironment() as venv:
                logging.info('Installing yt client')
                env.PipEnvironment('yandex-yt', version="0.8.11-0", venv=venv, use_wheel=True).prepare()
                env.PipEnvironment(
                    'yandex-yt-yson-bindings-skynet', version="0.2.25.post1", venv=venv, use_wheel=True
                ).prepare()
                script_path = mst.get_module_path() + '/dump_data_to_yt.py'
                cmd = [
                    venv.executable, script_path,
                    "--token", self.get_vault_data("SEARCH-ROBOT", "yt_blender_perf_token"),
                    "--data", perf_data_path,
                    "--table", "//home/search-runtime/test_data/web_basesearch/performance",
                ]
                run_process(" ".join(cmd), log_prefix="dump_perf_data_to_yt", shell=True)
        except Exception:
            logging.error("Unable to dump info to YT: %s", eh.shifted_traceback())


def _convert_to_yt_compatible(s):
    return s.replace("0.5", "50").replace("0.95", "95").replace("0.99", "99").replace(".", "_")


__Task__ = WebBasesearchPerformanceParallel
