import logging
import os

from sandbox import sdk2

from sandbox.projects.common import config_patcher_tool as cfg_patcher
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search.components import component as components_common


class WebBasesearch(
    components_common.WaitPortComponentMixin,
    components_common.ProcessComponentMixinWithShutdownSDK2,
    components_common.Component,
):
    name = "web_basesearch"

    LOGGER = logging.getLogger("components.WebBasesearch")
    _LOG_TMPL = name + "_{port}_{log_type}.log"
    _EMERGENCY_PATH = "degrade"
    _EMERGENCY_CHECK_PERIOD = 5  # period can't be less than 5 sec (see TCommonSearch::InitEmergencyCgi)
    _CONFIG_NAME_TMPL = name + "_{}_patched.cfg"

    def __init__(
        self, task, binary, config, models, database,
        port=search_components.DEFAULT_BASESEARCH_PORT,
        start_timeout=search_components.DEFAULT_START_TIMEOUT,
        cfg_patch_dict=None,
        patch_request_threads=True,
        polite_mode=None,
        ignore_index_generation=False
    ):
        self.LOGGER.info("Starting init WebBasesearch")
        self.task = task
        self.binary_data = sdk2.ResourceData(binary)
        self.config_data = sdk2.ResourceData(config)
        self.models_data = sdk2.ResourceData(models)
        self.database_data = sdk2.ResourceData(database)
        self.port = port
        self.start_timeout = start_timeout
        self.cfg_patch_dict = cfg_patch_dict or {}
        self.patch_request_threads = patch_request_threads
        self.polite_mode = polite_mode
        self.ignore_index_generation = ignore_index_generation

        patched_cfg = cfg_patcher.patch_cfg(
            task, str(self.config_data.path), self.cfg_patch_path(), self.cfg_name, self.name
        )
        self.LOGGER.debug("Patched cfg:\n%s", patched_cfg)

        components_common.WaitPortComponentMixin.__init__(
            self, endpoints=[("localhost", port)], wait_timeout=start_timeout
        )
        components_common.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=[str(self.binary_data.path), '-d', '-p', str(self.port), self.cfg_name],
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self.port),
            log_prefix=self.name,
        )
        self.LOGGER.info("Component '%s' initialized successfully", self.name)

    @property
    def cfg_name(self):
        return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.port))

    def path_to_log(self, log_type):
        return os.path.abspath(self._LOG_TMPL.format(port=self.port, log_type=log_type))

    def cfg_patch_path(self):
        self.cfg_patch_dict.update({
            'Collection.IndexDir': str(self.database_data.path),
            'Collection.MXNetFile': str(self.models_data.path),
            'Collection.EmergencyFile': os.path.abspath(self._EMERGENCY_PATH),
            'Collection.EmergencyCheckPeriod': str(self._EMERGENCY_CHECK_PERIOD),
            'Collection.UserParams.Polite': 'true' if self.polite_mode else None,
            'Collection.UserParams.NoIndexGeneration': '' if self.ignore_index_generation else "__remove__",
            'Server.ClientTimeout': 300,
            'Server.LoadLog': self.path_to_log("load"),
            'Server.LogExceptions': 'True',
            'Server.PassageLog': self.path_to_log("passage"),
            'Server.Port': self.port,
            'Server.ServerLog': self.path_to_log("server"),
        })

        if self.patch_request_threads:
            self.LOGGER.debug("Patch requests threads")
            self.cfg_patch_dict.update({
                'Collection.RequestThreads': '${1 + NCPU}',
                'Collection.SnippetThreads': '${1 + NCPU}',
                'Collection.FactorsThreads': '${1 + NCPU}',
            })
        patch_path = os.path.abspath("cfg_patch.json")
        fu.json_dump(patch_path, self.cfg_patch_dict, indent=1)
        return patch_path
