import datetime
import os
import json
import copy

try:
    import six
except ImportError:
    # To avoid `ya make` in small scripts and still work
    import sys

    class six:
        @staticmethod
        def _iter(name, container):
            if sys.version_info.major == 2:
                name = 'iter' + name
            return getattr(container, name)()

        @staticmethod
        def iteritems(x):
            return six._iter('items', x)

        @staticmethod
        def itervalues(x):
            return six._iter('values', x)

        @staticmethod
        def iterkeys(x):
            return six._iter('keys', x)


class BegemotService(object):
    @staticmethod
    def _upper_snake_case(s):
        parts = []
        b = 0
        for e, c in enumerate(s):
            if c.isupper() or c.isdigit():
                if e > 0 and (s[e - 1].isupper() or s[e - 1].isdigit()):
                    continue
                if b != e:
                    parts.append(s[b:e])
                b = e
        parts.append(s[b:])
        return '_'.join(p.upper() for p in parts)

    def _gen_resource_name(self, prefix, suffix):
        if suffix in [None, False]:
            return None
        elif suffix is True:
            return '{prefix}_{suffix}'.format(prefix=prefix, suffix=BegemotService._upper_snake_case(self.name))
        else:
            return suffix

    def __init__(
        self,
        name,
        apphost_source_name=None,  # backend name for srcrwr
        shard_resource_name=None,  # True: use default name; None: there is no shard
        fast_build_config_resource_name=True,  # True: use default name; None: there is no shard
        fresh=None,                # True: use default resource name; None: there is no fresh; string: use a custom resource name
        fresh_fast_build_config_resource_name=None,  # If fresh is true, default name is used
        prj=True,                  # for golovan
        shard_size_gb=1,
        release=False,
        test_fresh=None,           # Enable per-commit tests with fresh data in testenv (default: yes if there is a fresh)
        release_fresh=None,        # True: release this fresh via RELEASE_BEGEMOT_FRESH.
                                   # By default equals to the 'release' arg (see above). Is always False if there is no fresh.
        test_with_merger=False,
        beta=None,                 # generate resources for yappy betas for this shard.
        perf_beta=False,           # generate resources for yappy perf betas for this shard.
                                   #  Ask in begemoting how to add your component, 'True' or name here is required but is not enough.
        nanny_name=None,           # nanny service name, without '_production_{location}' suffix
        beta_parent=True,          # True: derive from nanny_name if there is beta. May be overridden manually
        testenv_apphost_queries=True,  # Name of testenv global resource with apphost queries for shard. True: use BEGEMOT_APPHOST_QUERIES
        testenv_cgi_queries=None,  # Name of global resource with cgi queries
        cpu_range=[20, 40],        # Normal range for service CPU usage
        binary='BEGEMOT',          # the resource used will be '{}_EXECUTABLE'
        released_by='search',      # Release cycle. 'search' is the default one, others are excluded from main tests&releases
        all_apphost_sources=None,  # List of all apphost sources using backend apphost_cource_name
    ):
        self.name = name
        self.apphost_source_name = apphost_source_name

        self.shard_resource_name_test = self._gen_resource_name('BEGEMOT_ISS', True)
        if shard_resource_name not in (True, None):
            raise RuntimeError("Begemot shard {}. Please don't use custom shard resourse name.")
        self.shard_resource_name = self._gen_resource_name('BEGEMOT_ISS', shard_resource_name)

        self.fast_build_config_resource_name = self._gen_resource_name('BEGEMOT_FAST_BUILD_CONFIG', fast_build_config_resource_name)
        self.fresh_resource_name = self._gen_resource_name('BEGEMOT_FRESH_DATA_FOR', fresh)
        self.test_with_merger = test_with_merger
        self.prj = [prj, self.name.lower()][prj is True]
        self.test_fresh = test_fresh
        self.released_by = released_by
        if self.fresh_resource_name:
            self.fresh_fast_build_config_resource_name = self.fresh_resource_name + '_FAST_BUILD'
            if test_fresh is None:
                self.test_fresh = True
        else:
            self.fresh_fast_build_config_resource_name = None
        self.shard_size_gb = shard_size_gb
        self.release = release
        if self.fresh_resource_name is None:
            self.release_fresh = False
        else:
            self.release_fresh = release_fresh if release_fresh is not None else self.release
        self.nanny_name = name.lower() if nanny_name is True else nanny_name
        self.cpu_range = cpu_range
        self.binary = binary
        self.binary_resource_name = "{}_EXECUTABLE".format(binary) if binary != 'BEGEMOT' else None
        self.beta = "worker-" + BegemotService._upper_snake_case(self.name).lower().replace("_", "-") if beta is True else beta
        self.perf_beta = name.lower() if perf_beta is True else perf_beta
        if self.beta or self.perf_beta:
            self.beta_parent = "begemot_hamster_%s" % self.nanny_name if beta_parent is True else beta_parent
            assert self.beta_parent
        self.testenv_apphost_queries = 'BEGEMOT_APPHOST_QUERIES' if testenv_apphost_queries is True else testenv_apphost_queries
        self.testenv_cgi_queries = testenv_cgi_queries
        self.all_apphost_sources = [self.apphost_source_name] if all_apphost_sources is None else all_apphost_sources

    def add_res(self, resources):
        self.data_resource_type = resources.__dict__[self.shard_resource_name] if self.shard_resource_name else None
        self.data_resource_type_test = resources.__dict__[self.shard_resource_name_test] if self.shard_resource_name_test else None
        self.fast_build_config_resource_type = resources.__dict__[self.fast_build_config_resource_name] if self.fast_build_config_resource_name else None
        self.binary_resource = resources.__dict__['{}_EXECUTABLE'.format(self.binary)]

        if self.fresh_resource_name:
            self.fresh_data_resource_type = resources.__dict__[self.fresh_resource_name]
            self.fresh_data_resource_packed_type = resources.__dict__[self.fresh_resource_name + '_PACKED']
            self.fresh_fast_build_config_resource_type = resources.__dict__[self.fresh_resource_name + '_FAST_BUILD']
        else:
            self.fresh_data_resource_type = None
            self.fresh_data_resource_packed_type = None
            self.fresh_fast_build_config_resource_type = None


class BegemotAllServices(object):
    def __init__(self):
        self.shards_path = 'search/begemot/data'
        self.Service = {}
        self.add(
            'Advq',
            binary='BEGEMOT_ADVQ',
            cpu_range=[0, 100],
            release=True,
            nanny_name='begemot_advq',
            shard_size_gb=45,
            testenv_apphost_queries=None,
            testenv_cgi_queries=None,
        )
        self.add(
            'AliceAramusic',
            apphost_source_name='BEGEMOT_ALICE_ARA_MUSIC',
            shard_size_gb=40,
            test_with_merger=True,
        )
        self.add(
            'Antirobot',
            binary='BEGEMOT_ANTIROBOT',
            cpu_range=[10, 100],
            release=True,
            nanny_name='begemot_antirobot',
            shard_size_gb=5,
            testenv_apphost_queries=None,
            testenv_cgi_queries='BEGEMOT_ANTIROBOT_QUERIES',
        )
        self.add(
            'Bert',
            apphost_source_name='BEGEMOT_BERT',
            beta=True,
            beta_parent='begemot_bert_hamster',
            binary='BEGEMOT_BERT',
            cpu_range=[0, 100],
            fresh=None,
            nanny_name='begemot_bert',
            perf_beta=False,
            release=True,
            shard_size_gb=15,
            test_with_merger=True,
            testenv_apphost_queries='BEGEMOT_BERT_APPHOST_QUERIES',
        )
        self.add(
            'Bravo',
            apphost_source_name='BEGEMOT_WORKER_BRAVO',
            beta=True,
            beta_parent='begemot_bravo_yp_hamster',
            fresh=True,
            nanny_name='begemot_bravo_yp',
            perf_beta=True,
            release=True,
            shard_size_gb=25,
            test_with_merger=True,
        )
        self.add(
            'Ethos',
            apphost_source_name='BEGEMOT_WORKER_E',
            beta=True,
            beta_parent="begemot_ethos_yp_hamster",
            fresh=True,
            nanny_name='begemot_ethos_yp',
            perf_beta=True,
            release=True,
            shard_size_gb=60,
            test_with_merger=True,
        )
        self.add(
            'Extractors',
            cpu_range=[0, 100],
            shard_size_gb=30,
        )
        self.add(
            'FailCache',
            apphost_source_name='BEGEMOT_FAILCACHE_BACKENDS',
            beta_parent='begemot_failcache_yp_prod_sas',
            fast_build_config_resource_name='BEGEMOT_FAST_BUILD_CONFIG_FAILCACHE',
            nanny_name='begemot_failcache_yp',
            perf_beta=True,
            release=True,
            testenv_apphost_queries='BEGEMOT_FAILCACHE_APPHOST_QUERIES',
        )
        self.add(
            'FakeShard',
            binary='BEGEMOT_FAKESHARD',
            release=False,
            shard_size_gb=10,
        )
        self.add(
            'Geo',
            binary='BEGEMOT_GEO',
            nanny_name='begemot_geo',
            release=True,
            shard_resource_name=True,
            shard_size_gb=55,
            testenv_apphost_queries=None,
        )
        self.add(
            'ImagesCV',
            apphost_source_name='IMAGES_RESULT_FILTER_BEGEMOT',
            beta='worker-alisa',
            beta_parent='begemot_cv_yp_hamster',
            cpu_range=[10, 30],
            fresh=True,
            nanny_name='begemot_cv_yp',
            perf_beta='cv',
            prj='cv',
            release=True,
            shard_size_gb=40,
            testenv_apphost_queries='BEGEMOT_IMAGES_CV_APPHOST_QUERIES',
        )
        self.add(
            'ImagesLogger',
            binary='BEGEMOT_LOGGER',
            release=True,
            testenv_apphost_queries='BEGEMOT_LOGGER_APPHOST_QUERIES',
        )
        self.add(
            'LingBoost',
            apphost_source_name='BEGEMOT_WORKER_LB',
            beta="worker-lingboost",
            beta_parent="begemot_lingboost_yp_hamster",
            fast_build_config_resource_name='BEGEMOT_FAST_BUILD_CONFIG_LINGBOOST',
            perf_beta=True,
            prj='lingboost',
            nanny_name='begemot_lingboost_yp',
            release=True,
            shard_size_gb=16,
            test_with_merger=True,
            testenv_apphost_queries='BEGEMOT_LINGBOOST_APPHOST_QUERIES',
            all_apphost_sources=['LINGBOOST_INIT', 'LINGBOOST_MERGER'],
        )
        self.add(
            'Logger',
            binary='BEGEMOT_LOGGER',
            release=True,
            testenv_apphost_queries='BEGEMOT_LOGGER_APPHOST_QUERIES',
        )
        self.add(
            'MarketWizard',
            apphost_source_name='BEGEMOT_WIZARD',
            cpu_range=[10, 30],
            fresh=True,
            nanny_name='bg_market_wizard_yp',
            prj='market-wizard',
            release=True,
            shard_size_gb=30,
        )
        self.add(
            'Megamind',
            apphost_source_name='BEGEMOT_WORKER_MEGAMIND',
            beta_parent='begemot_megamind_hamster',
            binary='BEGEMOT_MEGAMIND',
            cpu_range=[10, 30],
            fresh=True,
            nanny_name='begemot_megamind_yp',
            release=True,
            released_by='megamind',
            shard_size_gb=35,
            testenv_apphost_queries='BEGEMOT_MEGAMIND_APPHOST_QUERIES',
            testenv_cgi_queries='BEGEMOT_MEGAMIND_QUERIES',
        )
        self.add(
            'Beggins',
            apphost_source_name='BEGEMOT_WORKER_BEGGINS',
            binary='BEGEMOT_BEGGINS',
            cpu_range=[0, 30],
            fresh=True,
            nanny_name='begemot_megamind_yp',
            release=True,
            released_by='megamind',
            shard_size_gb=35,
        )
        self.add(
            'Merger',
            apphost_source_name='BEGEMOT_MERGER',
            beta="merger",
            beta_parent='begemot_merger_yp_hamster',
            cpu_range=[10, 30],
            fresh=True,
            nanny_name='begemot_merger_yp',
            perf_beta=True,
            release=True,
            test_with_merger=False,
            testenv_apphost_queries='BEGEMOT_MERGER_APPHOST_QUERIES',
            all_apphost_sources=['BEGEMOT_MERGER', 'BEGEMOT_MERGER_OPTIMISTIC', 'BEGEMOT_MERGER_MISSPELL'],
        )
        self.add(
            'MisspellFeatures',
            apphost_source_name='BEGEMOT_WORKER_MF',
            beta="worker-misspell-f",
            beta_parent="begemot_mf_yp_hamster",
            perf_beta='mf',
            prj='misspell',
            nanny_name='begemot_mf_yp',
            release=True,
            shard_size_gb=55,
            test_with_merger=True,
        )
        self.add(
            'MisspellFeatures2',
            apphost_source_name='BEGEMOT_WORKER_MF2',
            beta="worker-misspell-f-2",
            beta_parent="begemot_mf2_yp_hamster",
            fresh=True,
            perf_beta='mf2',
            prj='misspell2',
            nanny_name='begemot_mf2_yp',
            release=True,
            shard_size_gb=35,
            test_with_merger=True,
        )
        self.add(
            'RequestInit',
            apphost_source_name='REQUEST_INIT',
            binary='BEGEMOT_REQUEST_INIT',
            prj='init',
            release=True,
            released_by='init',
            shard_size_gb=2,
            testenv_apphost_queries='BEGEMOT_REQUEST_INIT_APPHOST_QUERIES',
        )
        self.add(
            'Spellchecker',
            fresh=True,
            prj='spellchecker-http',
            release=True,
            release_fresh=False,  # released separately manually
            nanny_name='spellchecker_http',
            shard_size_gb=200,
            test_fresh=False,     # Not in arcadia currently, not versioned
            testenv_apphost_queries='BEGEMOT_SPELLCHECKER_APPHOST_QUERIES',
        )
        self.add(
            'Wizard',
            apphost_source_name='BEGEMOT_WORKER_P',
            beta="worker-wizard-2",
            beta_parent="begemot_wizard_yp_hamster",
            fresh=True,
            nanny_name='begemot_wizard_yp',
            perf_beta=True,
            prj='wizard',
            release=True,
            shard_size_gb=40,
            test_with_merger=True,
        )

        self.add(
            'ServiceWizard',
            beta_parent='begemot_service_yp_hamster',
            cpu_range=[20, 50],
            fresh=True,
            perf_beta='service',
            prj='service-wizard',
            nanny_name='begemot_service_yp',
            release=True,
            shard_size_gb=self.Service['Wizard'].shard_size_gb + 2,
            testenv_apphost_queries='BEGEMOT_SERVICE_APPHOST_QUERIES',
            testenv_cgi_queries='BEGEMOT_SERVICE_CGI_QUERIES',
        )

        self.add(
            'SrcSetup',
            apphost_source_name='BEGEMOT_WORKER_SRC_SETUP',
            beta='worker-webfresh',
            beta_parent='begemot_webfresh_hamster',
            fresh=True,
            perf_beta='webfresh',
            prj='webfresh',
            nanny_name='begemot_webfresh_yp',
            release=True,
            shard_size_gb=2,
            test_with_merger=True,
            testenv_apphost_queries='BEGEMOT_WEBFRESH_APPHOST_QUERIES',
            all_apphost_sources=['BEGEMOT_WORKER_SRC_SETUP', 'BEGEMOT_WORKER_WEBFRESH_POST_SETUP'],
        )

        for size, lang in [
            (128, 'RUS'),
            (73, 'ENG'),
            (120, 'Misc1'),
        ]:
            llang = lang.lower()
            for suffix in ['', 'Exp']:
                self.add(
                    'Spellchecker' + lang + suffix,
                    beta='worker-spellchecker' + llang,
                    beta_parent='spellchecker%s_hamster' % llang,
                    cpu_range=[0, 100],  # irregular testing traffic
                    fresh=True,
                    perf_beta=not suffix and llang == 'rus',
                    prj='spellchecker' + llang,
                    release=True,
                    nanny_name='spellchecker{}{}'.format(lang.lower(), '' if not suffix else '_{}'.format(suffix.lower())),
                    release_fresh=False,  # released separately manually
                    shard_size_gb=size,
                    test_fresh=False,     # Not in arcadia currently, not versioned
                    testenv_apphost_queries='BEGEMOT_SPELLCHECKER_APPHOST_QUERIES',
                )

        self.Services = list(self.Service.keys())
        # Workers with fresh
        self.BegemotFreshShards = [s.name for s in self.Service.values() if s.fresh_resource_name]

    def add(self, name, **kwargs):
        assert name not in self.Service
        self.Service[name] = BegemotService(name, **kwargs)

    def _gen_resources_list(self, res_type, condition=lambda x: True, superset=None):
        return [
            ("begemot_{shard}_{res}".format(shard=name, res=res_type.replace('_resource_name', '')), getattr(s, res_type))
            for name, s in six.iteritems(superset or self.Service) if condition(s) and getattr(s, res_type)
        ]

    def gen_shard_resources_list(self, shard, resources):
        assert shard in self.Services
        ret = []
        if not isinstance(resources, list) and not isinstance(resources, tuple):
            resources = [resources]
        for r in resources:
            ret += self._gen_resources_list(r, superset={shard: self.Service[shard]})
        return ret

    def __iter__(self):
        return six.iteritems(self.Service).__iter__()

    def keys(self):
        return six.iterkeys(self.Service)

    def values(self):
        return six.itervalues(self.Service)

    def __getitem__(self, key):
        return self.Service[key]


Begemots = BegemotAllServices()

BegemotTestEnvQueries = {i.name: [i.testenv_cgi_queries, i.testenv_apphost_queries] for i in Begemots.values()}


class BegemotResources(object):
    Paths = {
        'BEGEMOT': 'search/daemons/begemot/default/begemot',
        'BEGEMOT_ADVQ': 'search/daemons/begemot/advq/advq',
        'BEGEMOT_ANTIROBOT': 'search/daemons/begemot/antirobot/antirobot',
        'BEGEMOT_BERT': 'search/daemons/begemot/bert/bert',
        'BEGEMOT_GEO': 'search/daemons/begemot/geo/geo',
        'BEGEMOT_LOGGER': 'search/daemons/begemot/logger/logger',
        'BEGEMOT_MEGAMIND': 'search/daemons/begemot/megamind/begemot',
        'BEGEMOT_REQUEST_INIT': 'search/daemons/begemot/request_init/request_init',
    }

    MainBinary = ("begemot", "BEGEMOT_EXECUTABLE")
    LoggerBinary = ("begemot_logger_res_id", "BEGEMOT_LOGGER_EXECUTABLE")
    AntirobotBinary = ("begemot_antirobot_res_id", "BEGEMOT_ANTIROBOT_EXECUTABLE")

    # Resources needed by all interesting begemot services
    Common = [
        MainBinary,
        ("worker.cfg", "BEGEMOT_CONFIG"),
        ("instancectl.conf", "BEGEMOT_INSTANCECTL_CONF"),
        ("fast_build_data_downloader", "BEGEMOT_FAST_BUILD_DOWNLOADER"),
        ("evlogdump", "BEGEMOT_EVLOGDUMP"),
        ("args", "BEGEMOT_ARGUMENTS_PARSER"),
    ]

    # proto files for unified agent: BEGEMOT-2663
    UnifiedAgentFiles = [
        ("events.ev", "BEGEMOT_PROTO_EVENTS"),
        ("descriptor.proto", "BEGEMOT_PROTO_DESCRIPTOR"),
        ("events_extension.proto", "BEGEMOT_PROTO_EVENTS_EXTENSION"),
        ("evlogdump_for_ua", "BEGEMOT_EVLOGDUMP_FOR_UNIFIED_AGENT"),
    ]
    Common += UnifiedAgentFiles

    Bstr = [
        ("bstr", "BEGEMOT_BSTR"),
        ("bstr_caller", "BEGEMOT_BSTR_CALLER"),
        ("fast_data_callback", "BEGEMOT_FAST_DATA_CALLBACK"),
    ]

    class LegacyWizard(object):
        Binary = ("wizard_executable_res_id", "REMOTE_WIZARD")
        Shard = ("wizard_shard_res_id", "WIZARD_SHARD")
        Fresh = ("wizard_fresh_id", "WIZARD_RUNTIME_PACKAGE_UNPACKED")
        GeoBinary = ("geowizard_binary_res_id", "REMOTE_WIZARD")

        Geo = [
            GeoBinary,
            Shard,
            ("wizard_geo_config", "WIZARD_GEO_CONFIG_NEW"),
        ]

        Misc = [
            Binary,
            Shard,
            ("wizard_config", "WIZARD_CONFIG"),
        ]

    Binaries = [
        MainBinary,
        LegacyWizard.Binary,
        LoggerBinary,
    ]
    AllCommon = Common + Bstr

    class Release(object):
        Fastbuild = Begemots._gen_resources_list('fast_build_config_resource_name', lambda x: x.release and (x.released_by == 'search' or x.released_by == 'megamind'))
        Fresh = Begemots._gen_resources_list('fresh_resource_name', lambda x: x.release_fresh and (x.released_by == 'search' or x.released_by == 'megamind'))
    Release.All = AllCommon + Release.Fastbuild + Release.Fresh

    class Betas(object):
        Fastbuild = Begemots._gen_resources_list('fast_build_config_resource_name', lambda x: x.beta and x.released_by == 'search')
        Fresh = Begemots._gen_resources_list('fresh_resource_name', lambda x: x.beta and x.release_fresh and x.released_by == 'search')
        SpecialBinaries = Begemots._gen_resources_list('binary_resource_name', lambda x: x.beta and x.binary != "BEGEMOT" and x.released_by == 'search')
    Betas.All = AllCommon + Betas.Fastbuild + Betas.Fresh + Betas.SpecialBinaries

    class PerfBetas(object):
        Fastbuild = Begemots._gen_resources_list('fast_build_config_resource_name', lambda x: x.perf_beta)
        Fresh = Begemots._gen_resources_list('fresh_resource_name', lambda x: x.perf_beta and x.release_fresh)
    PerfBetas.All = AllCommon + PerfBetas.Fastbuild + PerfBetas.Fresh

    class ServiceBeta(object):
        Fastbuild = Begemots._gen_resources_list('fast_build_config_resource_name', lambda x: x.name == 'ServiceWizard')
        Fresh = Begemots._gen_resources_list('fresh_resource_name', lambda x: x.name == 'ServiceWizard')
    ServiceBeta.All = AllCommon + ServiceBeta.Fastbuild + ServiceBeta.Fresh

    class Spellchecker(object):
        Fastbuild = Begemots.gen_shard_resources_list('Spellchecker', 'fast_build_config_resource_name')
        Fresh = [(k, v + "_PACKED") for k, v in Begemots.gen_shard_resources_list('Spellchecker', 'fresh_resource_name')]
        AllSpecific = Fastbuild + Fresh


class ApphostTestGraph(object):
    """
    Simple graph generator imports testing graph from arcadia
    Put graphs for this generator to arcadia/apphost/conf/graphs/TEST
    """
    def __init__(self, graph_path, backends):
        """
        backends should be a list of tuples: (backend_file_path, port, backend_name)
        """
        self.graph_path = graph_path
        self.backends = backends

    def generate_graph(self):
        graph, backends = {}, {}
        with open(self.graph_path, 'r') as gf:
            graph = json.loads(gf.read())['settings']
            for node in graph['nodes']:
                backend_name = graph['nodes'][node].get('backend_name', '')
                if backend_name:
                    graph['nodes'][node]['backend_name'] = backend_name.split('__')[-1]

        for path, port, name in self.backends:
            with open(path, 'r') as bf:
                backends[name] = json.loads(bf.read())
                backends[name]["instanceGroups"][0]["instances"][0]["port"] = port

        return graph, backends


class ApphostGraph(object):
    """
    Advanced graph generator builds testing graph and backends
    from production graphs web5b/begemot-workers
    """
    def __init__(self, graphs_path, worker_type, worker_port, merger_port, backends_path=None, separate_backends=True, merger_type='BEGEMOT_MERGER'):
        self._graphs = {}
        self._backends = {}
        self.graphs_path = graphs_path
        self.worker_type = worker_type
        self.merger_type = merger_type
        self.worker_port = worker_port
        self.merger_port = merger_port
        self.backends_path = backends_path
        self._separate_backends = separate_backends
        self._read_necessary_graphs()
        self._read_backends()

    def _read_necessary_graphs(self):
        for graph_name in ('web5b', 'begemot-workers'):
            with open(os.path.join(self.graphs_path, '{}.json'.format(graph_name))) as gf:
                self._graphs[graph_name] = json.loads(gf.read())

    def _read_backends(self):
        try:
            with open(self.backends_path) as backends_file:
                self._backends = json.load(backends_file)
                if not self._separate_backends:
                    self._backends = self._backends['MAN_WEB_APP_HOST_HAMSTER']
        except Exception:
            pass

    @staticmethod
    def _fill_backend_info(options, backend_ports):
        use_grpc = options.get('use_grpc', 1)
        options['backend_config'] = {
            'backend_descrs': [{
                'ip': '[::]',
                'port': str(port + use_grpc),
                'protocol': 'grpc' if use_grpc else 'post',
            } for port in backend_ports
            ]
        }
        options['timeout'] = 60000

    @staticmethod
    def _process_source(options, backend_descrs):
        use_grpc = options.get('use_grpc', 0)
        use_handler = options.get('handler', '/')

        for backend in backend_descrs:
            backend['port'] += use_grpc
            backend['path'] = use_handler

        return backend_descrs

    def _generate_backends(self):
        worker_backend = copy.deepcopy(self._backends)
        worker_backend["instanceGroups"][0]["instances"][0]["port"] = self.worker_port + 1
        merger_backend = copy.deepcopy(self._backends)
        merger_backend["instanceGroups"][0]["instances"][0]["port"] = self.merger_port + 1
        return {
            "BEGEMOT_WORKER.json": worker_backend,
            "BEGEMOT_MERGER.json": merger_backend
        }

    def get_source_backends(self, options):
        source_name = options['source']
        try:
            return self._process_source(options, self._backends[source_name]['backend_descrs'])
        except KeyError:
            pass

        try:
            return self._process_source(options, self._backends[source_name]['meta_backend_descrs'][1]['backend_descrs'])
        except KeyError:
            pass

    def generate_graph(self):
        if self.worker_type in self._graphs['begemot-workers']['settings']['nodes']:
            worker_options = self._graphs['begemot-workers']['settings']['nodes'][self.worker_type]
        elif self.worker_type in self._graphs['web5b']['settings']['nodes']:
            worker_options = self._graphs['web5b']['settings']['nodes'][self.worker_type]
        else:
            worker_options = {}
        if not self._separate_backends:
            self._fill_backend_info(worker_options, [self.worker_port])
        else:
            worker_options['backend_name'] = 'BEGEMOT_WORKER'
            if 'alias_config' in worker_options and 'addr_alias' in worker_options['alias_config']:
                for item in worker_options['alias_config']['addr_alias']:
                    if item == self.worker_type:
                        item = 'BEGEMOT_WORKER'

        try:
            merger_options = self._graphs['web5b']['settings']['nodes'][self.merger_type]
        except:
            merger_options = self._graphs['begemot-workers']['settings']['nodes'][self.merger_type]
            merger_options['backend_name'] = 'BEGEMOT_WORKER'

        if not self._separate_backends:
            self._fill_backend_info(merger_options, [self.merger_port])
        sources = {
            'BEGEMOT_WORKER': worker_options,
            'BEGEMOT_MERGER': merger_options,
            'BEGEMOT_CONFIG_ORIGINAL': self._graphs['web5b']['settings']['nodes']['BEGEMOT_CONFIG_ORIGINAL'],
            'BEGEMOT_CONFIG_MERGER': self._graphs['web5b']['settings']['nodes']['BEGEMOT_CONFIG_MERGER']
        }
        graph = {
            'version': 2,
            'settings': {
                'node_deps': {
                    'BEGEMOT_WORKER': {'input_deps': ['INIT', 'BEGEMOT_CONFIG_ORIGINAL']},
                    'BEGEMOT_MERGER': {'input_deps': ['BEGEMOT_CONFIG_MERGER', 'BEGEMOT_WORKER']},
                    'RESPONSE': {'input_deps': ['BEGEMOT_MERGER']}
                },
                'nodes': sources,
            }
        }
        if not self._separate_backends:
            return graph

        backends = self._generate_backends()
        return graph['settings'], backends

    def generate_lingboost_graph(self):
        graph, backends = self.generate_graph()
        graph['node_deps'] = {
            'BEGEMOT_WORKER': {'input_deps': ['INIT', 'LINGBOOST_CONFIG', 'LINGBOOST_INIT_FLAG', 'LINGBOOST_INIT_CONFIG']},
            'BEGEMOT_MERGER': {'input_deps': ['LINGBOOST_CONFIG', 'LINGBOOST_MERGER_FLAG', 'BEGEMOT_WORKER']},
            'RESPONSE': {'input_deps': ['BEGEMOT_MERGER']}
        }
        graph['nodes']['LINGBOOST_INIT_CONFIG'] = self._graphs['begemot-workers']['settings']['nodes']['LINGBOOST_INIT_CONFIG']
        graph['nodes']['LINGBOOST_CONFIG'] = self._graphs['begemot-workers']['settings']['nodes']['LINGBOOST_CONFIG']
        graph['nodes']['LINGBOOST_INIT_FLAG'] = self._graphs['begemot-workers']['settings']['nodes']['LINGBOOST_INIT_FLAG']
        graph['nodes']['LINGBOOST_MERGER_FLAG'] = self._graphs['begemot-workers']['settings']['nodes']['LINGBOOST_MERGER_FLAG']
        return graph, backends


def add_realtime_version_file(data_dir, version, new_binary=True):
    FILENAME = 'realtime_version.pb.txt'
    file_path = os.path.join(data_dir, FILENAME)
    field = 'RealtimeVersion' if new_binary else 'Task'
    with open(file_path, 'w') as f:
        f.write('GenerationTime: "{}"\n'.format(datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')))
        f.write('{}: {}\n'.format(field, version))
    return FILENAME

if __name__ == '__main__':
    from json import JSONEncoder

    class MyEncoder(JSONEncoder):
        def default(self, o):
            return o.__dict__
    json.dump(Begemots.Service, indent=2, fp=sys.stdout, cls=MyEncoder)
    sys.exit(0)
    graphs = os.path.join(
        os.path.abspath(__file__).split('/sandbox/', 1)[0],
        'apphost', 'conf', 'verticals', 'WEB'
    )
    worker_port = 10
    g = ApphostGraph(graphs_path=graphs,
                     worker_type='BEGEMOT_WORKER_E',
                     worker_port=worker_port,
                     merger_port=20).generate_graph()
    worker_e = g['settings']['nodes']['BEGEMOT_WORKER']
    use_grpc = worker_e.get('use_grpc', 1)
    assert worker_e['backend_config']['backend_descrs'][0]['port'] == str(worker_port + use_grpc)
    print(json.dumps(g, indent=2))
