# -*- coding: utf-8 -*-

import datetime
import json
import logging
import math
import os
import shutil
import subprocess
from multiprocessing.pool import ThreadPool

import sandbox.projects.common.constants as consts

from .deps_tree import get_deps_tree
from .hasher import ArcadiaDependencyHasher
from .versions_storage import VersionsStorage
from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.common.utils import get_task_link
from sandbox.common.types.notification import Transport
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot.resources import BEGEMOT_FAST_BUILD_RULE_DATA, BEGEMOT_FAST_BUILD_FRESH_CONFIG
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake

YaMake = YaMake.YaMake
FRESH_DIR = "search/wizard/data/fresh"


class _ConfigEntry:
    def __init__(self, name, convert=str, validate=lambda x: x is not None):
        self.name = name
        self.convert = convert
        self.validate = validate


class FastBuildRuleLogs(sdk2.Resource):
    ttl = 12
    releasable=False


def _validate_resource(res_id):
    if res_id is None:
        return False
    resource = channel.sandbox.get_resource(int(res_id))
    if not resource or not resource.is_ready():
        return False
    Client().resource[int(res_id)].update()
    return True


def _get_expiration_time():
    LOCKE_INFO_TTL = 14
    expiration_time = datetime.datetime.utcnow() + datetime.timedelta(days=LOCKE_INFO_TTL)
    return expiration_time.isoformat() + 'Z'


class _Config:
    schema = [
        _ConfigEntry('cypress_path'),
        _ConfigEntry('name'),
        _ConfigEntry('torrent'),
        _ConfigEntry('resource_id', convert=int, validate=_validate_resource),
        _ConfigEntry('resource_size_kb', convert=int),
        _ConfigEntry('arcadia_revision'),
    ]


class FastBuilder(object):

    RULE_RESOURCE_TTL = 30

    def __init__(self, yt_path, arcadia_path, shard_names, with_testpatches, fresh=False, one_config=False, rebuild=False, additional_flags="",
                 with_cypress=True, checkout=True, separate_build=False, threads=4):
        self.is_fresh = fresh
        self.is_separate_build = separate_build
        self.threads = threads
        self.arcadia_path = arcadia_path
        self.shard_names = shard_names.split()
        self.shard_name = self.shard_names[0]

        self.shard_path = os.path.join(Begemots.shards_path, self.shard_name) if not fresh else FRESH_DIR

        self.build_path = self.shard_path

        self.pack_path = "pack"
        if not fresh:
            self.shard_pack_path = os.path.join(self.pack_path, self.shard_name)
        else:
            self.shard_pack_path = os.path.join(self.pack_path, "rules")

        self.rules_to_build = []
        self.rules_ready = []
        self.rebuild_mode = rebuild
        self.one_config = one_config
        self.checkout = checkout

        self.yt_cluster = 'locke'
        self.yt_path = yt_path
        self.yt_token = sdk2.Vault.data('BEGEMOT', 'robot-bg-fastbuild-oauth-token')
        self.check_cypress_shards = with_cypress

        self.transaction_timeout = 60
        self.check_timeout = 10
        self.update_timeout = 20

        self.with_testpatches = with_testpatches

        self.rule_resource_type = "BEGEMOT_FAST_BUILD_RULE_DATA"

        if not fresh:
            self.config_resource_types = {name: Begemots[name].fast_build_config_resource_name for name in self.shard_names}
        else:
            self.config_resource_types = {name: AllBegemotServices[name].fresh_fast_build_config_resource_type.name for name in self.shard_names}

        if not os.path.exists(self.shard_pack_path):
            os.makedirs(self.shard_pack_path)

        self.pure_path = None
        self.additional_flags = additional_flags

        self.failed_rules_info = {}
        self.failed_rules_logs_path = 'logs'
        os.mkdir(self.failed_rules_logs_path)

    def make_checkout(self, arcadia, clear_build=False):
        logging.debug("FAST_BUILD: performing checkout...")

        # TODO Use svn.Arcadia.checkout to checkout ya.make files only
        if self.is_fresh:
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                self.arcadia_path,
                ["search/begemot/data/{}".format(s) for s in self.shard_names],
                build_threads=0,
                checkout=self.checkout,
                clear_build=clear_build
            )

        if self.is_fresh and self.is_separate_build:
            return

        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM,
            self.arcadia_path,
            [self.shard_path],
            build_threads=0,
            checkout=self.checkout,
            clear_build=clear_build
        )

        logging.debug("FAST_BUILD: checkout done")

    def _create_versions_storage(self):
        return VersionsStorage(
            self.yt_cluster,
            self.yt_path,
            self.yt_token,
            transaction_timeout=self.transaction_timeout
        )

    def _check_rule_to_build(self, rule):
        logging.debug("FAST_BUILD: checking rule %s", rule['name'])
        vs = self._create_versions_storage()
        if vs.lock_resource(rule['storage_node'], timeout=self.check_timeout):
            logging.debug("FAST_BUILD: validating prebuilt results for %s", rule['name'])
            update = {}
            for entry in _Config.schema:
                if entry.name == 'cypress_path' and not self.check_cypress_shards:
                    continue
                value = vs.get_resource_attribute(entry.name)
                if not entry.validate(value):
                    logging.debug("FAST_BUILD: failed validation of %s, key: %s, value: %s", rule['name'], entry.name, entry.convert(value))
                    break
                update[entry.name] = entry.convert(value)
            else:
                rule.update(update)
                logging.debug("FAST_BUILD: OK for rule %s", rule)
                vs.set_resource_attribute('expiration_time', _get_expiration_time())

            vs.unlock_resource()

    def _have_patches(self, path):
        full_path = os.path.join(self.arcadia_path, path)
        for curr, dirs, files in os.walk(full_path):
            for f in files:
                origin_file = os.path.join(curr, f)
                patch_file = origin_file + ".testpatch"
                if os.path.exists(patch_file):
                    return True
        return False

    def _get_rule_name(self, path):
        if not self.is_fresh:
            return os.path.basename(path)
        rule_make = YaMake(os.path.join(self.arcadia_path, path, 'ya.make'))
        try:
            set_macroses = json.loads(rule_make.dumps_json())['SET']
            for s in set_macroses:
                if s[0] == 'RESULT':
                    return os.path.basename(s[1])
        except:
            raise Exception("Failed to find rule name for path {}".format(path))

    def _resolve_rules_for_shards(self):
        self.rules_to_put = {}
        for shard in self.shard_names:
            shard_makefile = YaMake(os.path.join(self.arcadia_path, "search/begemot/data", shard, "ya.make"))
            self.rules_to_put[shard] = set([os.path.basename(path) for path in YaMake.skip_comments(shard_makefile.peerdir)])
            if self.is_fresh:
                released_config = sdk2.Resource.find(
                    type=AllBegemotServices.Service[shard].fast_build_config_resource_name,
                    attrs={'released': 'stable'}
                ).first()
                config_path = str(sdk2.ResourceData(released_config).path)
                config = json.load(open(config_path, "r"))
                for res in config["resources"]:
                    if res["name"] not in self.rules_to_put[shard]:
                        logging.debug("FAST_BUILD: ({}) add rule {} from production shard version".format(shard, res["name"]))
                        self.rules_to_put[shard].add(res["name"])
            logging.debug("FAST BUILD: Shard {}: put rules {}".format(shard, self.rules_to_put[shard]))

    def resolve_rules_to_build(self):
        exclude = [u'dict/gazetteer/compiler'] if self.is_fresh else []
        deps_tree = get_deps_tree(self.arcadia_path, self.shard_path, exclude=exclude, additional_flags=self.additional_flags)
        hasher = ArcadiaDependencyHasher(self.arcadia_path, deps_tree)
        shard = YaMake(os.path.join(self.arcadia_path, self.shard_path, 'ya.make'))
        if self.is_fresh:
            rule_paths = ["{}/{}".format(FRESH_DIR, rule) for rule in YaMake.skip_comments(shard.recurse)]
        else:
            rule_paths = YaMake.skip_comments(shard.peerdir)

        for rule_path in rule_paths:
            logging.debug("FAST BUILD: resolving rule {}".format(rule_path))
            rule_name = self._get_rule_name(rule_path)

            logging.debug("FAST BUILD: Rule name is {}".format(rule_name))

            if rule_name == "pure":
                self.pure_path = rule_path
                continue

            h = hasher.get_target_hash(rule_path)
            logging.debug("FAST_BUILD: hash for rule %s: %s", rule_name, h)

            node_suffix = ""
            if self.with_testpatches and self._have_patches(rule_path):
                node_suffix = "_test"

            # TODO: Build rules Thesaurus and ThesaurusLarge from different directories
            rule_names = [rule_name]
            if self.is_fresh and rule_name == "Thesaurus":
                rule_names.append("ThesaurusLarge")

            for name in rule_names:
                rule = {
                    'name': name,
                    'path': rule_path,
                    'storage_node': "{}/{}{}".format(name, h, node_suffix),
                }

                if not self.rebuild_mode:
                    self._check_rule_to_build(rule)
                if 'resource_id' in rule:
                    logging.debug("FAST_BUILD: found a prebuilt rule %s, resource_id is %s, rbtorrent is %s", rule, rule["resource_id"], rule.get("torrent"))
                    self.rules_ready.append(rule)
                else:
                    logging.debug("FAST_BUILD: going to build rule %s", rule)
                    self.rules_to_build.append(rule)

        if self.is_fresh:
            self._resolve_rules_for_shards()

        return [r['name'] for r in self.rules_ready + self.rules_to_build]

    def get_rules_for_shard(self, shard):
        return self.rules_to_put[shard] if self.is_fresh else None

    def _get_last_prebuild_rule_version(self, rule):
        vs = self._create_versions_storage()
        if vs.lock_resource(rule['name'], timeout=self.check_timeout):
            rule_hash = vs.get_rule_last_hash()
            vs.unlock_resource()
            cypress_path = '{}/{}'.format(rule['name'], rule_hash)
            if vs.lock_resource(cypress_path, timeout=self.check_timeout):
                for entry in _Config.schema:
                    value = vs.get_resource_attribute(entry.name)
                    rule[entry.name] = entry.convert(value)
                self.failed_rules_info[rule['name']] = {'revision': rule['arcadia_revision'], 'updated': vs.get_resource_attribute('modification_time')}
                vs.unlock_resource()

    def _write_faield_rule_logs(self, rule_name, out, err):
        path = os.path.join(self.failed_rules_logs_path, rule_name)
        os.mkdir(path)

        with open(os.path.join(path, 'ya.make.out'), 'w') as fout:
            fout.write(out.decode('utf-8'))

        with open(os.path.join(path, 'ya.make.err'), 'w') as ferr:
            ferr.write(err.decode('utf-8'))

    def _build_fresh_rule(self, rule):
        logging.debug("FAST_BUILD: start build {}".format(rule['name']))
        cmd = '{}/ya make -r {} -o={} {}'.format(self.arcadia_path, os.path.join(self.arcadia_path, rule['path']),
                                                 os.path.join(self.arcadia_path, self.build_path),
                                                 '--checkout'if self.checkout else '')
        logging.debug("FAST_BUILD: build rule cmd: {}".format(cmd))
        build_pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = build_pipe.communicate()
        returncode = build_pipe.poll()

        logging.debug("FAST_BUILD: {} build code: {}".format(rule['name'], returncode))

        if int(returncode) != 0:
            self._write_faield_rule_logs(rule['name'], out, err)
            self._get_last_prebuild_rule_version(rule)

    def _separate_fresh_build(self):
        logging.debug("FAST_BUILD: Separate build perfoming")
        self.rules_failed = set()
        p = ThreadPool(self.threads)
        p.map(self._build_fresh_rule, self.rules_to_build)
        logging.debug("FAST_BUILD: separate build done")

    def create_targets(self):
        self.build_path = os.path.join(self.build_path, "fast_build")
        build_full_path = os.path.join(self.arcadia_path, self.build_path)
        os.mkdir(build_full_path)

        rules_to_build_paths = [rule['path'] for rule in self.rules_to_build]

        if self.pure_path:
            rules_to_build_paths.append(self.pure_path)

        if self.is_fresh and self.is_separate_build:
            self._separate_fresh_build()
            rules_to_build_paths = []

        yamake_path = os.path.join(build_full_path, "ya.make")
        with open(yamake_path, 'w') as yamake:
            yamake.write("PACKAGE()\n\n")
            yamake.write("OWNER(g:begemot)\n\n")
            yamake.write("PEERDIR(\n")
            yamake.write("    " + "\n    ".join(rules_to_build_paths))
            yamake.write("\n)\n\n")
            yamake.write("END()\n")
        with open(yamake_path, 'r') as yamake:
            logging.debug(''.join(["Here is the generated ya.make file:"] + yamake.readlines()))

    def get_targets(self):
        return [self.build_path]

    def get_failed_rules_logs(self, task):
        for rule in os.listdir(self.failed_rules_logs_path):
            logs_resource = sdk2.Resource['FAST_BUILD_RULE_LOGS'](task, 'Ya make {} build logs'.format(rule), os.path.join(self.failed_rules_logs_path, rule))
            sdk2.ResourceData(logs_resource).ready()
            url = logs_resource.url
            task.set_info(
                '<p style="color:red;">Build {} failed</p><a href="{}">Ya make logs</a><p style="color:red;">Used cached data build on {}, arcadia_revision: {}</p>'.format(
                    rule,
                    url,
                    self.failed_rules_info[rule]['updated'],
                    self.failed_rules_info[rule]['revision']
                ),
                do_escape=False
            )

        if len(self.failed_rules_info) > 0:
            channel.sandbox.send_email(
                mail_to=['gluk47', 'bgsavko'],
                mail_cc=[],
                mail_subject='BUILD_BEGEMOT_DATA: Failed fast build rules',
                mail_body='<br/>'.join([
                    'Fresh build of these rules had failed:',
                    ' '.join([rule for rule in os.listdir(self.failed_rules_logs_path)]),
                    'Cached data build was taken.',
                    'You can find ya make logs in fask info: <a href="{url}">{task_id}</a>'.format(url=get_task_link(task.id), task_id=task.id)
                ]),
                content_type='text/html'
            )
            client = Client()
            client.notification(
                body="Fresh build failed. Rules: {}".format(' '.join([rule for rule in os.listdir(self.failed_rules_logs_path)])),
                recipients=["howcanunot"],
                transport=Transport.TELEGRAM
            )

    def _pack_rule(self, rule, output_dir):
        if rule['name'] in self.failed_rules_info:
            return
        full_build_path = os.path.join(output_dir, self.build_path)
        if self.is_fresh:
            rule_output_path = os.path.join(full_build_path, '{}/package'.format(FRESH_DIR), rule['name'])
        else:
            rule_output_path = os.path.join(full_build_path, rule['path'])
        rule_pack_path = os.path.join(self.shard_pack_path, rule['name'])

        logging.debug("FAST_BUILD: copying %s to %s", rule_output_path, rule_pack_path)
        shutil.copytree(rule_output_path, rule_pack_path)

    def pack_rules(self, output_dir):
        output_dir = self.arcadia_path if self.is_fresh and self.is_separate_build else output_dir
        for rule in self.rules_to_build:
            self._pack_rule(rule, output_dir)

    def make_own_pack(self):
        new_pack_path = "fast_build_pack"
        new_shard_pack_path = os.path.join(new_pack_path, self.shard_name)
        os.makedirs(new_shard_pack_path)

        for rule in self.rules_to_build:
            old_rule_path = os.path.join(self.shard_pack_path, rule['name'])
            new_rule_path = os.path.join(new_shard_pack_path, rule['name'])
            shutil.copytree(old_rule_path, new_rule_path)

        self.pack_path = new_pack_path
        self.shard_pack_path = new_shard_pack_path

    def _make_rule_resource(self, rule, task, revision):
        if rule['name'] in self.failed_rules_info:
            return
        description = "Rule '{}' for begemot".format(rule['name'])
        rule_pack_path = os.path.join(self.shard_pack_path, rule['name'])

        attrs = {'ttl': FastBuilder.RULE_RESOURCE_TTL, 'rule': rule['name']}

        try:
            res = BEGEMOT_FAST_BUILD_RULE_DATA(task, description, rule_pack_path)
            res.rule_ttl = FastBuilder.RULE_RESOURCE_TTL
            res.rule = rule['name']
            sdk2.ResourceData(res).ready()
            rbtorrent = res.skynet_id
        except Exception as e:
            logging.error('Cannot create resource using sdk2: {}'.format(e))
            res = task.create_resource(
                description,
                rule_pack_path,
                self.rule_resource_type,
                attributes=attrs
            )
            task.mark_resource_ready(res.id)
            rbtorrent = channel.sandbox.get_resource(res.id).skynet_id
        logging.debug("FAST_BUILD: resource id for rule %s is %s, rbtorrent: %s", rule['name'], res.id, rbtorrent)

        rule['resource_id'] = res.id
        rule['torrent'] = str(rbtorrent)
        rule['arcadia_revision'] = revision

        du_output = subprocess.check_output(['du', '-skL', rule_pack_path])
        rule['resource_size_kb'] = int(du_output.split()[0].decode('utf-8'))

    def make_resources(self, task, version_info):
        for rule in self.rules_to_build:
            self._make_rule_resource(rule, task, version_info["Revision"])

        only_config = self.is_fresh and self.one_config
        required_shards = self.shard_names if not only_config else ['ALL_SHARDS_FRESH']
        for shard in required_shards:
            if self.is_fresh:
                os.mkdir(os.path.join(self.pack_path, shard))
                config_path = os.path.join(self.pack_path, shard, "fast_build_config.json")
            else:
                config_path = os.path.join(self.pack_path, "fast_build_config.json")

            version_info["ShardName"] = '"{}"'.format(shard)
            config = {
                'shard_name': shard,
                'resources': [],
                'version_info': version_info,
            }

            data_size = 0
            for rule in self.rules_to_build + self.rules_ready:
                if not only_config and self.is_fresh and rule['name'] not in self.rules_to_put[shard]:
                    continue

                logging.debug("RULE: {}".format(rule))
                config['resources'].append({
                    entry.name: rule[entry.name] for entry in _Config.schema if entry.name in rule
                })
                data_size += int(rule['resource_size_kb'])

            with open(config_path, 'w') as config_file:
                json.dump(config, config_file, indent=4)

            description = "FastBuild config for begemot shard {}".format(shard) if not only_config else "FastBuild config for begemot fresh"

            if isinstance(version_info["Revision"], int):
                revision = version_info["Revision"]
            else:
                revision = -1

            attrs = {
                'data_size_kb': data_size,
                'version': revision,
            }

            resource = None
            if only_config:
                resource = BEGEMOT_FAST_BUILD_FRESH_CONFIG(task, description, config_path)
            else:
                try:
                    resource = sdk2.Resource[self.config_resource_types[shard]](task, description, config_path)
                except AttributeError:
                    task.create_resource(
                        description,
                        config_path,
                        self.config_resource_types[shard] if not only_config else 'BEGEMOT_FAST_BUILD_FRESH_CONFIG',
                        attributes=attrs
                    )
            if resource:
                resource.version = attrs['version']
                resource.data_size_kb = attrs['data_size_kb']

    def get_existing_cypress_paths(self):
        return (rule.get('cypress_path') for rule in self.rules_ready)

    def extend_existing_cypress_paths_ttl(self, cache_config):
        import yt.wrapper as yw

        client = yw.YtClient(proxy=cache_config['proxy'], token=cache_config['token'])
        expiration_time = datetime.datetime.utcnow() + datetime.timedelta(days=14)
        expiration_time = expiration_time.isoformat()

        for path in self.get_existing_cypress_paths():
            parts = path.split('/')
            if len(parts) < 2:
                logging.error("Unexpected cypress path (too short): '%s'" % path)
            res_hash = parts[-2]
            attr_path = os.path.join(cache_config['path'], res_hash, '@expiration_time')
            client.set(attr_path, expiration_time)

    def update_cypress_paths(self, file_path):
        with open(file_path, 'r') as file:
            paths = [path.rstrip() for path in file.readlines()]

        path_dict = {
            os.path.basename(path): path for path in paths
        }
        logging.debug("FAST_BUILD: New cypress paths:")
        for name, path in path_dict.items():
            logging.debug("### %s: %s", name, path)

        for rule in self.rules_to_build:
            if rule['name'] in path_dict:
                rule['cypress_path'] = path_dict[rule['name']]
            else:
                logging.debug("FAST_BUILD: Missing cypress path for %s, skipping", rule['name'])

    def update_cache(self):
        for rule in self.rules_to_build:
            vs = self._create_versions_storage()
            if not vs.lock_resource(rule['storage_node'], timeout=self.update_timeout):
                continue
            for entry in _Config.schema:
                value = rule.get(entry.name)
                if value:
                    vs.set_resource_attribute(entry.name, value)
            vs.set_resource_attribute('expiration_time', _get_expiration_time())
            vs.unlock_resource()

    def get_rules_size_mb(self):
        return {
            rule['name']: math.ceil(int(rule['resource_size_kb']) / 1024.0)
            for rule in self.rules_to_build + self.rules_ready
        }
