import os
import hashlib


class ArcadiaDependencyHasher:
    def __init__(self, root_path, deps_tree):
        self._root_path = root_path
        self._deps_tree = deps_tree
        self._physical_hashes = {}
        self._target_hashes = {}

    def _get_file_md5(self, full_path):
        h = hashlib.md5()
        with open(full_path, "rb") as f:
            for chunk in iter(lambda: f.read(4096 * 1000), b""):
                h.update(chunk)
        return h.hexdigest()

    def _get_str_md5(self, s):
        h = hashlib.md5()
        h.update(s.encode('utf-8'))
        return h.hexdigest()

    def count_physical_hash(self, path):
        full_path = os.path.join(self._root_path, path)
        if os.path.isfile(full_path):
            return self._get_file_md5(full_path)
        else:
            childs = sorted(os.listdir(full_path))
            child_hashes = []
            for child in childs:
                child_path = os.path.join(path, child)
                h = self.get_physical_hash(child_path)
                child_hashes.append((child, h))
            return self._get_str_md5(str(child_hashes))

    def get_physical_hash(self, path):
        if path not in self._physical_hashes:
            self._physical_hashes[path] = self.count_physical_hash(path)
        return self._physical_hashes[path]

    def _scan_deps(self, path, deps_set):
        if path in deps_set:
            return
        deps_set.add(path)
        for dep_path in self._deps_tree.get(path, []):
            self._scan_deps(dep_path, deps_set)

    def _count_target_hash(self, path):
        deps_set = set()
        self._scan_deps(path, deps_set)
        deps_hashes = []
        for dep in sorted(list(deps_set)):
            deps_hashes.append((dep, self.get_physical_hash(dep)))
        return self._get_str_md5(str(deps_hashes))

    def get_target_hash(self, path):
        if path not in self._target_hashes:
            self._target_hashes[path] = self._count_target_hash(path)
        return self._target_hashes[path]
