import logging
import os

import json

from .lock import Lock, SharedNodeLock

logger = logging.getLogger(__name__)


class VersionsStorage:
    def __init__(self, proxy, path, token,
                 transaction_timeout=5, logger=logger):
        import yt.wrapper as yt

        self._client = yt.YtClient(proxy=proxy, token=token)

        self._path = path
        self._transaction_timeout = transaction_timeout * 1000
        self._logger = logger
        self._format = yt.format.JsonFormat()

        self._reset_lock_variables()

    def _reset_lock_variables(self):
        self._lock = None
        self._lock_client = None
        self._resource_path = None

    def lock_resource(self, node, timeout=None, exclusive=True):
        if self._lock is not None:
            raise RuntimeError("Error: attempt to make second lock.")

        transaction_kwargs = {'timeout': self._transaction_timeout}
        self._resource_path = os.path.join(self._path, node)
        LockType = Lock if exclusive else SharedNodeLock
        self._lock = LockType(
            self._resource_path,
            client=self._client,
            transaction_kwargs=transaction_kwargs,
            identifier=node)

        self._lock_client = self._lock._yt

        if self._lock.acquire(timeout=timeout):
            return True

        self._logger.debug("Unable to lock resource for node {}".format(self._resource_path))
        self._reset_lock_variables()
        return False

    def unlock_resource(self):
        self._lock.release()
        self._reset_lock_variables()

    def get_resource_attribute(self, attribute):
        client = self._lock_client if self._lock else self._client
        path = "{}/@{}".format(self._resource_path, attribute)
        if client.exists(path):
            return client.get(path)
        else:
            msg = "Resource {} was not found, returning None.".format(self._resource_path)
            self._logger.debug(msg)
            return None

    def get_rule_last_hash(self):
        client = self._lock_client if self._lock else self._client
        cmp = lambda x: x['$attributes']['modification_time']
        rules_attributes = sorted(json.loads(client.list(self._resource_path, attributes=['modification_time', 'arcadia_revision'], format=self._format)), key=cmp, reverse=True)
        for rule in rules_attributes:
            if 'arcadia_revision' in rule['$attributes']:
                return rule['$value']
        else:
            raise RuntimeError("Cached data doesn't exist")

    def set_resource_attribute(self, attribute, value):
        if self._lock:
            path = "{}/@{}".format(self._resource_path, attribute)
            self._lock_client.set(path, value)
        else:
            raise RuntimeError("Error: attempt to set resource without lock.")
