# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common.types import resource as ctr
import sandbox.projects.resource_types.releasers as rr
from sandbox.projects import resource_types
from sandbox.projects.websearch.begemot.common import Begemots, BegemotResources
import sandbox.projects.runtime_models.releasers as runtime_models_releasers


class BegemotFreshData(sdk2.resource.AbstractResource):
    """
    Base begemot fresh data (packed or unpacked) resource
    """
    releasable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    runtime_data = sdk2.Attributes.Integer('Robots repository revision', required=True)
    version = sdk2.Attributes.Integer("Package version", default_value=0)


def _generate_bg_resource_type(shard, attr_name, class_dict={}, base_class=sdk2.Resource, suffix=''):
    res_name = getattr(shard, attr_name)
    if not res_name:
        return
    res_name += suffix
    class_dict["__module__"] = __name__
    globals()[res_name] = type(res_name, (base_class, ), class_dict)


class BEGEMOT_ISS_SHARD(sdk2.resource.AbstractResource):
    """
    Some begemot data
    """
    releasable = True
    any_arch = True
    executable = True
    auto_backup = True
    releasers = rr.wizard_releasers


class BEGEMOT_FAST_BUILD_SHARD_CONFIG(sdk2.Resource):
    """
    Json config for begemot shard containing resources to fetch,
    generated by fast build
    """
    releasable = True
    any_arch = True
    executable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    data_size_kb = sdk2.Attributes.Integer("Size of shard data", default_value=0)
    version = sdk2.Attributes.Integer("Arcadia revision", default_value=0)


class BEGEMOT_FAST_BUILD_FRESH_CONFIG(BEGEMOT_FAST_BUILD_SHARD_CONFIG):
    """
    Json config for all begemot fresh data containing resources to fetch,
    generated by fast build
    """
    dir_name = 'ALL_SHARDS_FRESH'


class BEGEMOT_STABLE_RELEASE_CONFIG(sdk2.Resource):
    """
    Json config for begemot prod&hamster services, contains names of every service and
    list of resources (section `files` in Nanny) that should be on it
    """
    releasable = True
    any_arch = True
    releasers = rr.wizard_releasers


class BEGEMOT_TESTING_RELEASE_CONFIG(sdk2.Resource):
    """
    Json config for begemot testing services, contains names of every service and
    list of resources (section `files` in Nanny) that should be on it
    """
    releasable = True
    any_arch = True
    releasers = rr.wizard_releasers


for name, bg in Begemots:
    class_dict = {'dir_name': name}
    _generate_bg_resource_type(bg, 'fresh_resource_name', class_dict)
    _generate_bg_resource_type(bg, 'fresh_resource_name', class_dict, suffix='_PACKED')
    _generate_bg_resource_type(bg, 'fresh_resource_name', class_dict, BEGEMOT_FAST_BUILD_SHARD_CONFIG, suffix='_FAST_BUILD')
    _generate_bg_resource_type(bg, 'shard_resource_name_test', class_dict, BEGEMOT_ISS_SHARD)
    _generate_bg_resource_type(bg, 'fast_build_config_resource_name', class_dict, BEGEMOT_FAST_BUILD_SHARD_CONFIG)


class BEGEMOT_REALTIME_PACKAGE(sdk2.Resource):
    """
    Begemot realtime data.
    """
    shard = sdk2.parameters.String('Begemot shard name')
    releasable = True
    any_arch = True
    auto_backup = True
    version = sdk2.Attributes.Integer("Package version", default_value=0)


class BEGEMOT_SHARD_SIZE_DIFF(sdk2.Resource):
    """
    diff between two size_shard_mb
    """
    any_arch = True


class BEGEMOT_RESPONSES_DIFF(sdk2.resource.AbstractResource):
    """
    html diff between two BEGEMOT_RESPONSES_RESULT
    """
    any_arch = True


class BEGEMOT_FAILCACHE_HIT_DIFF(sdk2.resource.AbstractResource):
    """
    double diff between two BEGEMOT_FAILCACHE_HIT_RESULT
    """
    any_arch = True


class BEGEMOT_CONFIG_WITH_CGI(sdk2.resource.AbstractResource):
    """
    begemot testing config dir
    """
    auto_backup = True
    any_arch = True
    arcadia_build_path = 'web/daemons/begemot'
    config_file = 'worker.cfg'


class BEGEMOT_CONFIG(sdk2.resource.AbstractResource):
    """
    begemot related resource
    """
    releasable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/worker.cfg'


class BEGEMOT_INSTANCECTL_CONF(sdk2.resource.Resource):
    """
    begemot instancectl.conf file
    """
    restart_policy = ctr.RestartPolicy.IGNORE
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/instancectl.conf'


class BEGEMOT_PUSHCLIENT_CONF(sdk2.resource.Resource):
    """
    Begemot perf logger pushclient config
    """
    restart_policy = ctr.RestartPolicy.IGNORE
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'search/daemons/begemot/logger/push-client.conf'


class BEGEMOT_APP_HOST_PLAN(sdk2.resource.AbstractResource):
    """
    Begemot apphost plan
    """


class BEGEMOT_EXECUTABLE(sdk2.resource.AbstractResource):
    """
    begemot binary
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = "search/daemons/begemot/default/begemot"


class BEGEMOT_EXECUTABLE_FUZZY(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'search/fuzzing/fuzz_begemot/fuzz_begemot'
    arcadia_build_name = 'begemot_fuzzy'


class BEGEMOT_BERT_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/bert/bert"
    releasers = BEGEMOT_EXECUTABLE.releasers + ['gotmanov']


class BEGEMOT_CAESAR_MODELS_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/yabs_caesar_models/yabs_caesar_models"
    releasers = runtime_models_releasers.caesar_models_releasers


class BEGEMOT_LOGGER_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/logger/logger"
    releasers = BEGEMOT_EXECUTABLE.releasers


class BEGEMOT_MEGAMIND_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/megamind/megamind"
    releasers = rr.megamind_wizard_releasers


class BEGEMOT_BEGGINS_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/beggins/beggins"
    releasers = rr.megamind_wizard_releasers


class BEGEMOT_ADVQ_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = BegemotResources.Paths['BEGEMOT_ADVQ']
    releasers = BEGEMOT_EXECUTABLE.releasers + ['abordonos', 'maxim-kuzin', 'd-frolov', 'mafanasev']


class BEGEMOT_ANTIROBOT_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/antirobot/antirobot"
    releasers = BEGEMOT_EXECUTABLE.releasers + ['ashagarov', 'toshchakov']


class BEGEMOT_GEO_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/geo/geo"
    releasers = BEGEMOT_EXECUTABLE.releasers + ['evelkin', 'yurakura']


class BEGEMOT_REQUEST_INIT_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/request_init/request_init"
    releasers = BEGEMOT_EXECUTABLE.releasers + ['alex-ersh', 'dima-zakharov', 'elshiko', 'kozunov', 'vmordovin', 'mya-engineer']


class BEGEMOT_FAKESHARD_EXECUTABLE(BEGEMOT_EXECUTABLE):
    arcadia_build_path = "search/daemons/begemot/fakeshard/fakeshard"
    releasers = BEGEMOT_EXECUTABLE.releasers


class BEGEMOT_FULL_EXECUTABLE(BEGEMOT_EXECUTABLE):
    """
    begemot full(with debug info) binary
    """


class BEGEMOT_EVLOG(sdk2.resource.AbstractResource):
    """
    begemot event log
    """


class BEGEMOT_EVLOGDUMP(sdk2.resource.AbstractResource):
    """
    begemot event log dumper
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers + ["alborisov"]
    arcadia_build_path = 'web/daemons/begemot/event_log_dump/event_log_dump'


class BEGEMOT_EVLOGSTAT(sdk2.resource.AbstractResource):
    """
    begemot event log stat dumper
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/event_log_stat/event_log_stat'


class BEGEMOT_EVLOG_UPLOADER(sdk2.resource.AbstractResource):
    """
    begemot event log uploader
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/scripts/eventlog_uploader/eventlog_uploader'


class BEGEMOT_ARGUMENTS_PARSER(sdk2.resource.AbstractResource):
    """
    begemot argument parser
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/scripts/args/args'


class BEGEMOT_AH_CONVERTER(sdk2.resource.AbstractResource):
    """
    apphost context converter
    """
    executable = True
    any_arch = False
    auto_backup = True
    arcadia_build_path = 'apphost/tools/converter/converter'


class BEGEMOT_CGI_PLAN(sdk2.resource.AbstractResource):
    """
    begemot input for tests in cgi format
    """
    any_arch = True
    arcadia_build_path = 'web/daemons/begemot/test/worker_test_input.txt'


class BEGEMOT_TEST_AH_CONTEXT_INPUT(sdk2.resource.AbstractResource):
    """
    begemot tests' input in app_host context format
    """
    any_arch = True


class BEGEMOT_TEST_AH_GRAPH(sdk2.resource.AbstractResource):
    """
    begemot tests' app_host graph, copy of production one
    """
    any_arch = True
    releasable = True
    releasers = rr.wizard_releasers


class BEGEMOT_TEST_AH_CONTEXT_OUTPUT(sdk2.resource.AbstractResource):
    """
    begemot tests' needed output in app_host context format
    """
    any_arch = True


class BEGEMOT_RESPONSES_RESULT(sdk2.resource.AbstractResource):
    """
    begemot tests' output in cgi format
    """
    any_arch = True
    responses_for_revision = sdk2.parameters.Integer('Revison number')


class BEGEMOT_FAILCACHE_HIT_RESULT(sdk2.resource.AbstractResource):
    """
    begemot failcache tests' cache hit
    """
    any_arch = True
    responses_for_revision = sdk2.parameters.Integer('Revison number')


class BEGEMOT_SHARD_UPDATER(sdk2.resource.AbstractResource):
    """
    tool for updating begemot shards in Cypress
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    arcadia_build_path = 'tools/wizard_yt/shard_updater/shard_updater'


class BEGEMOT_YT_EVENTLOG_MAPPER(sdk2.resource.AbstractResource):
    """
    Binary for building input table from eventlog table in YT
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    arcadia_build_path = 'tools/wizard_yt/begemot_reducer/eventlog_mapper/eventlog_mapper'


class BEGEMOT_YT_MAPPER(sdk2.resource.AbstractResource):
    """
    begemot binary for tests in YT.
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    arcadia_build_path = 'tools/wizard_yt/begemot_reducer/mapper/default/mapper'


class BEGEMOT_CYPRESS_SHARD(sdk2.resource.AbstractResource):
    """
    File with shard files paths in Cypress
    """
    releasable = True
    executable = False
    any_arch = True
    shard_name = sdk2.Attributes.String('Shard name')
    is_fresh = sdk2.Attributes.Bool('Is fresh', default_value=False)
    is_broken = sdk2.Attributes.Bool('If loading to YT failed, generate resource for testenv', default_value=False)


class BEGEMOT_APPHOST_QUERIES(sdk2.resource.AbstractResource):
    """
    begemot input for tests in apphost json format
    """


class BEGEMOT_CGI_QUERIES(sdk2.resource.AbstractResource):
    """
    begemot input for tests in cgi format
    """


class BEGEMOT_HTTP_QUERIES(sdk2.resource.AbstractResource):
    """
    begemot input queries for tests, each line is a query
    """


class BEGEMOT_SPELLCHECKER_APPHOST_QUERIES(BEGEMOT_APPHOST_QUERIES):
    """
    begemot input in apphost json format for spellchecker
    """


class BEGEMOT_LOGGER_APPHOST_QUERIES(BEGEMOT_APPHOST_QUERIES):
    """
    begemot input in apphost json format for logger
    """


class BEGEMOT_FAILCACHE_APPHOST_QUERIES(BEGEMOT_APPHOST_QUERIES):
    """
    begemot input in apphost json format for failcache
    """


class BEGEMOT_FAILCACHE_HIT_APPHOST_QUERIES(BEGEMOT_APPHOST_QUERIES):
    """
    begemot input in apphost json format for failcache hit
    """


class SAMOGON_BEGEMOT_FRESH_PACKAGE(sdk2.resource.AbstractResource):
    """
    Samogon service for deploying begemot fresh
    """
    releasable = True
    auto_backup = True
    any_arch = True
    releasers = rr.wizard_releasers


class YT_DOWNLOAD_COREDUMP(sdk2.resource.AbstractResource):
    """
    Binary for downloading operation coredumps.
    """
    executable = True


class BEGEMOT_BSTR_CALLER(sdk2.resource.AbstractResource):
    """
    Binary for downloading fast data in nanny services
    """
    executable = True
    releasable = True
    arcadia_build_path = 'web/daemons/begemot/scripts/bstr_fresh_deploy/bstr_caller/bstr_caller'


class BEGEMOT_BSTR(sdk2.resource.AbstractResource):
    """
    Bstr binary for begemot releases
    Is used to pull fresh data
    """
    executable = True
    releasable = True
    arcadia_build_path = 'tools/bstr/bstr'


class BEGEMOT_BSTR_CALLBACK(sdk2.resource.AbstractResource):
    """
    Bstr callback binary
    """
    executable = True
    releasable = True
    arcadia_build_path = 'web/daemons/begemot/scripts/bstr_fresh_deploy/callback/callback'


class BEGEMOT_FAST_DATA_CALLBACK(sdk2.resource.AbstractResource):
    """
    Bstr callback binary
    """
    executable = True
    releasable = True
    arcadia_build_path = 'search/tools/fast_data_deployment/callback/callback'


class BEGEMOT_BSTR_PUSHER(sdk2.Resource):
    """
    pusher
    """
    executable = True
    releasable = True
    arcadia_build_path = 'web/daemons/begemot/scripts/bstr_fresh_deploy/pusher/pusher'


class WIZARD_FULL_EXECUTABLE(resource_types.REMOTE_WIZARD):
    """
    wizard full (with debug info) binary
    """


class WIZARD_DEBUG(sdk2.resource.AbstractResource):
    """
    debug info from begemot binary
    """
    releasable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers


class BEGEMOT_EVENTLOG(sdk2.Resource):
    pass


class BEGEMOT_SPELLCHECKER_HUSQVARNA_API_LOG(sdk2.Resource):
    """
    log of begemot husqwarna api
    """
    executable = False


class BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT(sdk2.Resource):
    """
    json with husqvarna test report
    """
    executable = False
    releasable = True

    begemot_release = sdk2.Attributes.Integer("Begemot release number", required=False, default=0)


class BEGEMOT_FAST_BUILD_RULE_DATA(sdk2.Resource):
    """
    Data for single begemot rule generated by fast build
    """
    releasable = True
    rule_ttl = sdk2.Attributes.Integer("Fasr build rule ttl")
    rule = sdk2.Attributes.String("Fast build rule name")


class BEGEMOT_FAST_BUILD_DOWNLOADER(sdk2.resource.Resource):
    """
    begemot fastbuild rules downloader binary
    """
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'search/begemot/tools/fast_build/resource_downloader/deploy/deploy'


class BEGEMOT_RESPONSE_PARSER(sdk2.resource.AbstractResource):
    """
    begemot response-parser tool binary
    """
    releasable = False
    executable = True
    any_arch = False
    auto_backup = True
    arcadia_build_path = 'search/begemot/tools/response-parser/response-parser'


class BEGEMOT_CACHE_GUESS_SCRIPT(sdk2.resource.Resource):
    """
    begemot script for compare diff on old and new fresh date
    """
    releaseable = False
    executable = True
    any_arch = False


class BEGEMOT_NANNY_TIDINESS(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'web/daemons/begemot/scripts/nanny_tidiness/tidiness'
    arcadia_build_name = 'nanny_tidiness'


class WizardQueriesJson(sdk2.Resource):
    """Wizard/begemot queries in json format"""
    any_arch = True
    TE_web_production_wizard_reqs = sdk2.parameters.String("Testenv prod attribute")


class BEGEMOT_PROTO_EVENTS(sdk2.resource.Resource):
    """
    events file for unified agent
    """
    releasable = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'search/begemot/status/events.ev'


class BEGEMOT_PROTO_DESCRIPTOR(sdk2.resource.Resource):
    """
    dependency of events.ev
    """
    releasable = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'contrib/libs/protobuf/src/google/protobuf/descriptor.proto'


class BEGEMOT_PROTO_EVENTS_EXTENSION(sdk2.resource.Resource):
    """
    dependency of events.ev
    """
    releasable = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'library/cpp/eventlog/proto/events_extension.proto'


class BEGEMOT_EVLOGDUMP_FOR_UNIFIED_AGENT(sdk2.resource.AbstractResource):
    """
    evlogdump for unified agent
    """
    releasable = True
    executable = True
    any_arch = False
    auto_backup = True
    releasers = rr.wizard_releasers
    arcadia_build_path = 'search/tools/setrace/tools/evlogdump_for_ua/evlogdump_for_ua'


class BEGEMOT_UT_LOG_DUMP(sdk2.Resource):
    """
    dumps file for single begemot rule
    """
    apphost_source = 'BEGEMOT_WORKER_P'
