# -*- coding: utf-8 -*-
import os
import logging

from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.websearch.begemot import resources as br
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.common import error_handlers as eh
from sandbox.common.types.resource import State
from sandbox import sdk2


class BegemotCreateResponsesDiffMulti(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        responses_without_patch = sdk2.parameters.List(
            "List of first GetBegemotResponses result resources",
            description="",
            default=[],
            required=True
        )
        responses_with_patch = sdk2.parameters.List(
            "List of second GetBegemotResponses result resources",
            description="",
            default=[],
            required=True
        )
        verbose = sdk2.parameters.Bool('Check the box only if you want to debug', default=False)
        fail_on_any_error = True

    def verbose(self, *args, **kwargs):
        if self.Parameters.verbose:
            logging.debug(*args, **kwargs)
        else:
            pass

    def on_execute(self):
        import sandbox.projects.websearch.begemot.tasks.BegemotCreateResponsesDiff as BegemotCreateResponsesDiff
        diff_class = sdk2.Task[BegemotCreateResponsesDiff.BegemotCreateResponsesDiff.type]
        responses_without_patch = self.Parameters.responses_without_patch
        responses_with_patch = self.Parameters.responses_with_patch

        with self.memoize_stage.get_begemot_responses(commit_on_entrance=False):
            broken_shards = []
            diff_tasks = []
            shard_list = []

            for i in range(len(responses_without_patch)):
                try:
                    old_resp = sdk2.Resource[responses_without_patch[i]]
                    new_resp = sdk2.Resource[responses_with_patch[i]]
                except Exception as e:
                    eh.check_failed('Error on collecting some resources: {}'.format(e))

                if old_resp.state != State.READY and new_resp.state != State.READY:
                    broken_shards.append((old_resp.Shard, "both"))
                elif old_resp.state != State.READY:
                    broken_shards.append((old_resp.Shard, "old"))
                elif new_resp.state != State.READY:
                    broken_shards.append((old_resp.Shard, "new"))
                else:
                    diff_task = diff_class(
                        self,
                        begemot_responses_1=old_resp.id,
                        begemot_responses_2=new_resp.id,
                        fail_rate=False
                    )
                    diff_tasks.append(diff_task.save().enqueue().id)
                    shard_list.append(old_resp.Shard)

            self.Context.shard_list = shard_list
            self.Context.begemot_create_responses_diff_tasks = diff_tasks
            self.Context.broken_shards = broken_shards
            self.Context.save()

            raise sdk2.WaitTask(self.Context.begemot_create_responses_diff_tasks, Status.Group.FINISH | Status.Group.BREAK, wait_all=True)

        bad_statuses = [Status.FAILURE, Status.EXCEPTION, Status.TIMEOUT]
        sub_tasks = self.find(diff_class)
        for shard, sub_task_id in enumerate(self.Context.begemot_create_responses_diff_tasks):
            sub_task = sdk2.Task[sub_task_id]
            if sub_task.status in bad_statuses:
                self.set_info("BegemotCreateResponsesDiff task {} for shard {} have failed.".format(sub_task_id, self.Context.shard_list[shard]))

        self.Context.diffs_resources = [task.Context.diffs for task in sub_tasks]
        output_path = sdk2.paths.make_folder("diffs_multi", delete_content=True)
        self.Context.has_diff = False

        if len(self.Context.broken_shards) != 0:
            self.Context.has_diff = True
            self.set_info("Broken shards: " + ', '.join(list(zip(*self.Context.broken_shards))[0]))
            for name, state in self.Context.broken_shards:
                sdk2.paths.make_folder(output_path + "/" + name)
                fout = open(output_path + "/" + name + "/" + "response.diff", "w")
                if state == "both":
                    fout.write("Both old and new responses are not ready, the shard {} is broken.".format(name))
                else:
                    fout.write("Got no {} responses, the shard {} is broken".format(state, name))
                fout.close()
            eh.check_failed('Error on collecting some resources. Below is a list of broken shards.')

        self.verbose("Broken shards:\n%r", self.Context.broken_shards)
        self.verbose("Resources ids:\n%r", self.Context.diffs_resources)

        for res in self.Context.diffs_resources:
            try:
                sync_resource(res)
            except errors.TaskError:
                self.verbose("Tried to download null resource")
            else:
                self.verbose("Correct resource, shard: %r", sdk2.Resource[res].Shard)
                self.Context.has_diff = True
                folder_name = "/" + sdk2.Resource[res].Shard
                is_cgi = sdk2.Resource[res].is_cgi
                in_path = sync_resource(res)
                if os.path.exists(output_path + folder_name):
                    if is_cgi:
                        folder_name += '.cgi'
                    else:
                        folder_name += '2'
                sdk2.paths.copy_path(in_path, output_path + folder_name)
        if self.Context.has_diff:
            multi_diff_res = br.BEGEMOT_RESPONSES_DIFF(self, "", output_path)
            self.Context.out_resource_id = multi_diff_res.id
            resource_data = sdk2.ResourceData(multi_diff_res)
            resource_data.ready()
