# -*- coding: utf-8 -*-

from __future__ import print_function

import os
import shutil
import logging

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia, Svn
from sandbox.sandboxsdk.channel import channel

from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common import utils

from sandbox.projects.common.wizard.wizard_builder import WizardBuilder
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CypressShardUpdater
import sandbox.projects.common.build.parameters as build_params

from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake

YaMake = YaMake.YaMake


class ResourceNotFound(Exception):
    pass


class BuildForProduction(parameters.SandboxBoolParameter):
    """
        Create begemot packs based on production-branch shards
        instead of current shards from Arcadia.
        Set this to False in per-commit checks,
        set this to true for sandbox release scheduler.
    """
    name = "production_build"
    description = "Build for production binary and shards"
    default_value = True


class ArcadiaRevision(parameters.SandboxIntegerParameter):
    """
    default: latest
    """
    name = 'arcadia_revision'
    description = 'Arcadia revision to checkout'
    default = None


class ParentResources(parameters.DictRepeater, parameters.SandboxStringParameter):
    # This parameter is used by BUILD_WIZARD_2 and is not visible in the UI.
    name = 'parent_resources'
    description = 'Transfer resources to parent task (resource_type, resource_id)'


class BegemotShards(parameters.SandboxBoolGroupParameter):
    name = 'begemot_shards'
    description = 'Begemot shards to build'
    choices = [(s.name, s.name) for s in AllBegemotServices.Service.values() if s.fresh_data_resource_type is not None]
    default_value = ' '.join(i[0] for i in choices)


class CreateUnpacked(parameters.SandboxBoolParameter):
    name = 'create_unpacked'
    description = 'Create unpacked resources'
    default_value = False


class ShardBuilder:
    def __init__(self, service):
        self.service = service
        self.path = 'fresh_for_%s' % service
        self.package_path = 'fresh_for_%s.tar' % service
        if service == 'wizard':
            self.package_resource_type = resource_types.WIZARD_RUNTIME_PACKAGE
            self.resource_type = resource_types.WIZARD_RUNTIME_PACKAGE_UNPACKED
        else:
            self.package_resource_type = AllBegemotServices.Service[self.service].fresh_data_resource_packed_type
            self.resource_type = AllBegemotServices.Service[self.service].fresh_data_resource_type
        self._static_data_yamake_path = ''

    def set_static_data_dir(self, dir):
        self._static_data_yamake_path = os.path.join(dir, 'ya.make')

    def build_from_shard(self, pre_build_path):
        for item in self._rules_in_yamake(self._static_data_yamake_path):
            shard_item_path = os.path.join(pre_build_path, item)
            if os.path.isdir(shard_item_path):
                shutil.copytree(shard_item_path, os.path.join(self.path, item))
        with open(os.path.join(self.path, 'version.pb.txt'), 'a') as fp:
            fp.write('ShardName: "%s"\n' % self.service)
        process.run_process(['tar', '-cf', self.package_path, self.path], check=True, wait=True)

    @staticmethod
    def _rules_in_yamake(yamake_path):
        if not os.path.exists(yamake_path):
            # A shard has not yet appeared in any released branch
            return []
        shard = YaMake(yamake_path)
        return [r.split('search/wizard/data/wizard/', 1)[1] for r in shard.peerdir]


class BegemotFreshBuild(YaMakeTask, CypressShardUpdater):
    type = 'BEGEMOT_FRESH_BUILD'
    fresh_subpath = 'search/wizard/data/fresh'
    execution_space = 100 * 1024  # MB
    input_parameters = [build_params.ArcadiaUrl, BuildForProduction, ArcadiaRevision, BegemotShards, CreateUnpacked] + \
        CypressShardUpdater.input_parameters
    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1
    client_tags = Tag.CUSTOM_BEGEMOT
    cores = 24

    logger = logging.getLogger('TASK_LOGGER')
    logger.setLevel(logging.DEBUG)

    def _make_begemot_shard_builders(self):
        if 'begemot_shard_resource_ids' not in self.ctx:
            self.ctx['begemot_shard_resource_ids'] = dict()
        return [
            ShardBuilder(s)
            for s in ['wizard'] + utils.get_or_default(self.ctx, BegemotShards).split()
        ]

    def get_resources_attrs(self):
        return {'runtime_data': self.ctx['runtime_data']}

    def pre_build(self, source_dir):
        with open(os.path.join(source_dir, self.fresh_subpath, '.merged_revision')) as fp:
            self.ctx['runtime_data'] = fp.read()
        self.ctx[build_params.EnvironmentVarsParam.name] = '''
            SANDBOX_TASK=%s
            ARCADIA_REVISION=%s
            ROBOTS_REVISION=%s
        ''' % (str(self.id), str(self.ctx[ArcadiaRevision.name]), self.ctx['runtime_data'])

    def get_targets(self):
        return [self.fresh_subpath]

    @classmethod
    def _checkout_arcadia_branch(cls):
        arc = WizardBuilder.get_production_wizard_task().ctx[build_params.ArcadiaUrl.name]
        arcadia_path = Arcadia.checkout(arc, 'arcadia', depth=Svn.Depth.IMMEDIATES)
        Arcadia.update(os.path.join(arcadia_path, 'search', 'begemot', 'data'), depth=Svn.Depth.INFINITY, parents=True)
        Arcadia.update(os.path.join(arcadia_path, 'search', 'wizard', 'data', 'wizard'), depth=Svn.Depth.INFINITY,
                       parents=True)
        return arcadia_path

    def post_build(self, source_dir, output_dir, pack_dir):
        shard_builders = self._make_begemot_shard_builders()

        if utils.get_or_default(self.ctx, BuildForProduction):
            arcadia_with_yamake = self._checkout_arcadia_branch()
        else:
            arcadia_with_yamake = source_dir

        pre_build_path = os.path.join(output_dir, self.fresh_subpath, 'package')

        for b in shard_builders:
            if b.service == 'wizard':
                b.set_static_data_dir(os.path.join(arcadia_with_yamake, 'search', 'wizard', 'data', b.service))
            else:
                b.set_static_data_dir(os.path.join(arcadia_with_yamake, 'search', 'begemot', 'data', b.service))
            b.build_from_shard(pre_build_path)

        for b in shard_builders:
            self.update_cypress_shard(b.path, b.service, is_fresh=True)

        for b in shard_builders:
            attrs = {'runtime_data': self.ctx['runtime_data']}
            if self.ctx[CreateUnpacked.name]:
                self.create_resource(self.description, b.path, b.resource_type, attributes=attrs)
            self.create_resource(self.description, b.package_path, b.package_resource_type, attributes=attrs)

        parent_resources = self.ctx.get(ParentResources.name) or {}
        for res_type, res_id in parent_resources.items():
            have = channel.sandbox.list_resources(resource_type=res_type, task_id=self.id)
            if have:
                self.save_parent_task_resource(have[0].path, int(res_id), save_basename=True)
            else:
                self.info += 'Warning: parent task requested %s, but no such resource generated' % res_type


__Task__ = BegemotFreshBuild
