# -*- encoding: utf-8 -*-
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.websearch.begemot.common import Begemots
import sandbox.projects.websearch.begemot.resources as br

import json
import logging
import os
from sandbox.projects.common import requests_wrapper


# Graphs conatining more lines will most likely cause timeout while rendering
_HUGE_GRAPHS_LINES = 256


class BegemotRulegraph(sdk2.Task):
    '''Generate rules graph for all begemot shards and post the result to wiki'''

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.String('OAuth token for writing to wiki', default=None, description='default: from sandbox vault')
        page = sdk2.parameters.String("Wiki page to write to", default='Begemot/graph')
        width = sdk2.parameters.Integer("Thumbnail width. Set to -1 for no restriction (this will also disable full-graph generation hidden in a spoiler)", default=1800)
        begemot_binary = sdk2.parameters.Resource(
            'Begemot binary to describe',
            resource_type=br.BEGEMOT_EXECUTABLE,
            required=True,
        )
        begemot_binaries = sdk2.parameters.Dict('Begemot binaries')
        arcadia_url = sdk2.parameters.String('Svn url for arcadia', default=sdk2.svn.Arcadia.trunk_url())

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1.5 * 1024

    @staticmethod
    def format_graph(graph, width=-1):
        decorator = 'graphviz'
        if width > 0:
            decorator += ' width=%s' % width
        return '%%%%(%s)\n%s\n%%%%' % (decorator, graph)

    def on_execute(self):
        params = self.Parameters

        services = Begemots.Service
        arc_url = self.Parameters.arcadia_url
        if not arc_url.endswith('/arcadia'):
            arc_url += '/arcadia'
        shards = sdk2.svn.Arcadia.get_arcadia_src_dir('%s/search/begemot/data' % arc_url)

        images = []
        valid_shards = []
        for shard in sorted(os.listdir(shards)):
            image = 'Объявление шарда: [search/begemot/data/{shard}/ya.make](https://a.yandex-team.ru/arc/trunk/arcadia/search/begemot/data/{shard}/ya.make).\n'.format(
                shard=shard
            )
            yamake = os.path.join(shards, shard, 'ya.make')
            if shard == 'ut' or not os.path.exists(yamake):
                continue
            valid_shards.append(shard)

            name = ''
            if shard in services:
                name = services[shard].binary or ''
            resource_id = params.begemot_binaries.get(name.lower(), None)
            if resource_id is not None:
                resource = sdk2.Resource[int(resource_id)]
            else:
                resource = params.begemot_binary
            resource_data = sdk2.ResourceData(resource)

            image += 'Был использован бинарь {name}.\n\n'.format(name=resource.type.name)

            begemot_executable = str(resource_data.path)
            args = [begemot_executable, '--print-rulegraph', '--data', yamake]
            try:
                graph = subprocess.check_output(args)
            except subprocess.CalledProcessError as exc:
                logging.error("Generating graph for {shard} failed, exit code {code}.\n{msg}\n========\n".format(
                    shard=shard,
                    code=exc.returncode,
                    msg=exc.output
                ))
                raise

            image += '<{Graph sources\n%%%%\n%s\n%%%%}>\n' % graph
            if graph.count('\n') > _HUGE_GRAPHS_LINES:
                image += 'The graph is too huge to be rendered\n'
            elif params.width > 0:
                image += '\n'.join([
                    self.format_graph(graph, params.width),
                    '<{Full-size picture',
                    self.format_graph(graph),
                    '}>',
                    ''
                ])
            else:
                image += self.format_graph(graph) + '\n'
            images.append(image)

        head_info = '\n'.join([
            'Обновлено скриптом [BegemotGraph](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/websearch/begemot/tasks/BegemotGraph/__init__.py).',
            'Исходники графов удобно рассматривать в [xdot](https://github.com/jrfonseca/xdot.py).',
            ''
        ]).format(shard=shard)
        links = []
        for index, shard in enumerate(valid_shards):
            links.append('{idx}. [{shard}](!/{shard})'.format(idx=index + 1, shard=shard))

        data_main = {'title': 'Begemot rule dependency graph',
                     'body': head_info + '\n'.join(links)}
        data_shards = [{'title': shard, 'body': head_info + img}
                       for img, shard in zip(images, valid_shards)]

        API = 'https://wiki-api.yandex-team.ru/_api/frontend/'
        oauth = params.oauth_token or sdk2.Vault.data('Begemot wiki token')
        headers = {'Authorization': 'OAuth %s' % oauth,
                   'Content-Type': 'application/json'}
        logging.debug('Wiki API: requesting POST %s' % (API + params.page))
        responses = []
        for shard, data in zip([''] + valid_shards, [data_main] + data_shards):
            response = requests_wrapper.post(API + params.page + '/' + shard, headers=headers, data=json.dumps(data))
            logging.debug('Wiki API response (%s):\n+++\n%s---\n' % (response.status_code, response.text))
            responses.append(response)
        href = 'http://wiki/' + params.page.lstrip('/')
        self.set_info('<a href={href}>{href}</a>'.format(**locals()))
        for response in responses:
            response.raise_for_status()
