# -*- coding: utf-8 -*-
import logging
import re
import os
import json
import jinja2

import sandbox.projects.release_machine.helpers.startrek_helper as sh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import link_builder as lb
from sandbox.projects.websearch.begemot.tasks.BegemotSpellcheckerHusqvarnaQuality import BegemotSpellcheckerHusqvarnaQuality
from sandbox.projects.websearch.begemot.resources import BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT
from sandbox.projects.websearch.begemot.common.spellchecker import QualityDiffViewer

MAX_BRANCHES_BACK = 10


class BegemotHusqvarnaQualityDiff(sdk2.Task):
    '''
    Get diff between run husqvarna test on current branch and previous.
    '''
    __logger = logging.getLogger('TASK_LOGGER')
    __logger.setLevel(logging.DEBUG)

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Parameters):
        kill_timeout = 12 * 60 * 60

        current_branch_report = sdk2.parameters.Resource(
            'Test run report for current branch',
            resource_type=BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT,
            name="current_branch_report"
        )

        report_to_ticket = sdk2.parameters.Bool(
            'Report to release ticket',
            default=False,
            name="report_to_ticket"
        )

    def get_branch_n(self, resource_description):
        pattern = re.compile(r"begemot-spellchecker-(\d+)")
        logging.debug("Resource description: %s", resource_description)
        branch_n =  int(re.search(pattern, resource_description).group(1))
        logging.info("Parse branch number as %s, from report description \"%s\"", branch_n, resource_description)
        return branch_n

    def get_sb_tag(self, branch_n):
        return "TESTENV-DATABASE-WS-BEGEMOT-{}".format(branch_n)

    def get_previous_report(self, branch_n):
        for old_branch_n in range(branch_n - 1, branch_n - MAX_BRANCHES_BACK, -1):
            task = sdk2.Task.find(BegemotSpellcheckerHusqvarnaQuality,
                                   input_parametrs={"husqvarna_test_group": "release"},
                                   hidden=True,
                                   status=(ctt.Status.SUCCESS),
                                   tags=[self.get_sb_tag(old_branch_n)]).order(-sdk2.Task.id).first()
            if task is None:
                logging.warning("Task BEGEMOT_SPELLCHECKER_HUSQVARNA_QUALITY for branch %s not found", old_branch_n)
                continue

            logging.info("Task BEGEMOT_SPELLCHECKER_HUSQVARNA_QUALITY for branch %s has id %s", old_branch_n, task.id)
            resource =  sdk2.Resource.find(
                BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT,
                task=task
            ).first()

            if resource is not None:
                logging.info("Task found resource %s", resource.id)
                return resource
        raise TaskFailure("Resource BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT for branches [{start}, {end}] not found".format(start=branch_n - MAX_BRANCHES_BACK + 1, end=branch_n - 1))

    def on_execute(self):
        branch_n = self.get_branch_n(self.Parameters.current_branch_report.description)
        previous_report = self.get_previous_report(branch_n)

        st_helper = sh.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        c_info = rmc.COMPONENTS["begemot"]()
        if self.Parameters.report_to_ticket:
            st_helper.write_grouped_comment(
                "Begemot spellchecker quality diff",
                None,
                "Task started, see details in {}".format(lb.task_wiki_link(self.id)),
                branch_n,
                c_info,
            )


        current_report_path = str(sdk2.ResourceData(self.Parameters.current_branch_report).path)
        previous_report_path = str(sdk2.ResourceData(self.Parameters.current_branch_report).path)

        current_report_json = json.load(open(current_report_path, "r"))
        previous_report_json = json.load(open(previous_report_path, "r"))
        viewer = QualityDiffViewer(previous_report_json, current_report_json)
        self.Context.quality_diff_report = viewer.get_quality_diff_report()
        self.Context.save()

        if not viewer.is_zerodiff():
            if self.Parameters.report_to_ticket:
                st_helper.write_grouped_comment(
                    "Begemot spellchecker quality diff",
                    None,
                    "Task !!FAILED!!, see details in {}".format(lb.task_wiki_link(self.id)),
                    branch_n,
                    c_info,
                )
            raise TaskFailure("There is quality diff. It is time to ask responsible engineer.")
        if self.Parameters.report_to_ticket:
            st_helper.write_grouped_comment(
                "Begemot spellchecker quality diff",
                None,
                "It is zerodiff, we can release. See details in {}".format(lb.task_wiki_link(self.id)),
                branch_n,
                c_info,
            )

    @sdk2.footer()
    def quality_table(self):
        if self.Context.quality_diff_report:
            template_path = os.path.dirname(os.path.abspath(__file__))
            env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
            return env.get_template('footer.tmpl').render({'results': self.Context.quality_diff_report,
                                                           'METRICS': QualityDiffViewer.METRICS,
                                                           'RESPONSE_TYPES': QualityDiffViewer.RESPONSE_TYPES,
                                                           'URL_TEMPLATE': QualityDiffViewer.URL_TEMPLATE})


