# -*- coding: utf-8 -*-

import json
import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.projects.websearch.begemot import parameters as begemot_parameters
from sandbox.projects.websearch.begemot import resources as begemot_resources
from sandbox.projects.websearch.rtmodels import resources as rtmodels_resources
from sandbox.projects.websearch.cfg_models import resources as cfgmodels_resources
from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.common.utils import Enum


class ServiceType(Enum):
    BEGEMOT = "begemot"
    RTMODELS = "rtmodels"
    CFG_MODELS = "cfg_models"


class BegemotParseEvlog(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Get begemot statistics from eventlog.
    """

    __logger = logging.getLogger("TASK_LOGGER")
    __logger.setLevel(logging.DEBUG)

    class Context(sdk2.Context):
        critical_path_metrics = {}
        bgschema = {}
        statistics = {}
        templates = {}
        batch_requests = 0

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Service type") as service_type:
            service_type.values[ServiceType.BEGEMOT] = service_type.Value(value=ServiceType.BEGEMOT)
            service_type.values[ServiceType.RTMODELS] = service_type.Value(value=ServiceType.RTMODELS)
            service_type.values[ServiceType.CFG_MODELS] = service_type.Value(value=ServiceType.CFG_MODELS)

        with service_type.value[ServiceType.BEGEMOT]:
            begemot_executable = begemot_parameters.BegemotExecutableResource(required=True)

        with service_type.value[ServiceType.RTMODELS]:
            rtmodels_executable = sdk2.parameters.Resource(
                "RtModels executable",
                required=True,
                resource_type=rtmodels_resources.RtmodelsExecutable
            )

        with service_type.value[ServiceType.CFG_MODELS]:
            cfgmodels_executable = sdk2.parameters.Resource(
                "CfgModels executable",
                required=True,
                resource_type=cfgmodels_resources.CfgModelsExecutable
            )

        evlogstat_binary = begemot_parameters.BegemotEvlogstatBinaryResource(required=False)
        eventlog_dump = sdk2.parameters.Resource(
            "Eventlog dump",
            required=True,
            resource_type=begemot_resources.BEGEMOT_EVENTLOG
        )

        tasks_archive_resource = binary_task.binary_release_parameters(stable=True)

    def _collect_templates(self, templates=[]):
        path = str(self.path("templates"))
        Arcadia.export(Arcadia.trunk_url("sandbox/projects/websearch/begemot/tasks/BegemotParseEvlog"), path)
        for template in templates:
            with open(os.path.join(path, template), 'rb') as fd:
                self.Context.templates[template] = fd.read().decode('utf-8')

    def _get_template(self, template_name):
        if template_name in self.Context.templates:
            return self.Context.templates[template_name]
        else:
            return ""

    def _render_table_row(self, row):
        return '<tr>{}</tr>'.format(''.join('<td>{}</td>'.format(cell) for cell in row))

    def _render_table(self, header, rows):
        return self._get_template('table.html').format(
            header=self._render_table_row(header),
            rows=''.join(map(self._render_table_row, rows)),
        )

    def _parse_evlog(self):
        evlogstat_binary = str(sdk2.ResourceData(self.Parameters.evlogstat_binary).path)

        self.Context.bgschema = self._get_bgschema()
        with tempfile.NamedTemporaryFile(mode='w') as bgschema_file:
            bgschema_file.write(json.dumps(self.Context.bgschema))
            bgschema_file.delete = False

        eventlog_dump = str(sdk2.ResourceData(self.Parameters.eventlog_dump).path)
        with open(eventlog_dump, 'r') as log_stream, sdk2.helpers.ProcessLog(self, logger='evlogstat') as pl:
            proc = subprocess.Popen(
                (
                    evlogstat_binary,
                    '--bgschema', bgschema_file.name,
                    '--start-frame', str(self.Context.batch_requests),
                ),
                stdin=log_stream,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
            )
            try:
                self.Context.statistics = json.loads(proc.stdout.read())
            except:
                self.Context.statistics = {}

    def _get_executable_path(self):
        if self.Parameters.service_type == ServiceType.BEGEMOT:
            return str(sdk2.ResourceData(self.Parameters.begemot_executable).path)
        elif self.Parameters.service_type == ServiceType.RTMODELS:
            return str(sdk2.ResourceData(self.Parameters.rtmodels_executable).path)
        elif self.Parameters.service_type == ServiceType.CFG_MODELS:
            return str(sdk2.ResourceData(self.Parameters.cfgmodels_executable).path)

    def _get_bgschema(self):
        executable_path = self._get_executable_path()
        with sdk2.helpers.ProcessLog(self, logger='bgschema') as pl:
            proc = subprocess.Popen((executable_path, '--print-bgschema'), stdout=subprocess.PIPE, stderr=pl.stderr)
            return json.load(proc.stdout)

    def _build_critical_path_metrics(self, target_metric):
        critical_info = self.Context.statistics['CriticalInfo']
        crit_time = [value['CriticalTime'] for value in critical_info.values()]
        crit_index = crit_time.index(max(crit_time))
        crit_key = critical_info.keys()[crit_index]
        crit_path = list()
        while crit_key is not None:
            crit_path.append(crit_key)
            crit_key = critical_info[crit_key].get('CriticalDependency', None)

        self.Context.critical_path_metrics = {
            '_longest_path': crit_path,
            'perfect_parallel_metric': crit_time[crit_index],
            'current_parallel_metric': self.Context.statistics['FrameStat'][target_metric],
            'metric': sum(metrics[target_metric] for metrics in self.Context.statistics['RuleStats'].itervalues())
        }

    @sdk2.report(title='Rules metrics', label='_rules_metrics_report')
    def _rules_metrics_report(self):
        rules_metrics = self.Context.statistics['RuleStats']
        metrics = ('mean', 'median', 'sqrt_variance', 'amount', 'quantile_95', 'quantile_90', 'quantile_75')

        header = ('Rules',) + metrics
        rows = [
            (
                rule,
                rule_metrics['Average'],
                rule_metrics['Quantility'][50],
                rule_metrics['SqrtVariance'],
                rule_metrics['Amount'],
                rule_metrics['Quantility'][95],
                rule_metrics['Quantility'][90],
                rule_metrics['Quantility'][75]
            )
            for rule, rule_metrics in rules_metrics.iteritems()
        ]
        rows.sort(key=lambda x: x[1], reverse=True)
        return self._render_table(header, rows)


    @sdk2.report(title='Critical path', label='_critical_path_report')
    def _critical_path_report(self):
        critical_path = self.Context.critical_path_metrics.get('_longest_path', ())

        header = ('Rule', 'Critical path mean')
        rows = [
            (
                rule,
                round(self.Context.statistics['CriticalInfo'][rule]['CriticalTime']),
            )
            for rule in critical_path
        ]
        return self._render_table(header, rows)


    @sdk2.header()
    def _header(self):
        perfect_parallel_metric = self.Context.critical_path_metrics.get('perfect_parallel_metric', 0.0)
        current_parallel_metric = self.Context.critical_path_metrics.get('current_parallel_metric', 0.0)
        metric = self.Context.critical_path_metrics.get('metric', 0.0)
        return self._get_template('header.html').format(
            perfect_parallelism=perfect_parallel_metric,
            perfect_parallelism_prc=round(perfect_parallel_metric * 100 / float(metric or 1), 2),
            current_parallelism=current_parallel_metric,
            current_parallelism_prc=round(current_parallel_metric * 100 / float(metric or 1), 2),
            rules_summary=metric,
        )


    def on_execute(self):
        self._collect_templates(["header.html", "table.html"])
        self._parse_evlog()
        if not self.Context.statistics:
            raise TaskFailure("Error building eventlog statistics.")
        else:
            self._build_critical_path_metrics('Average')
