import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.types.misc import NotExists
from sandbox.projects.websearch.begemot.tasks.GetBegemotResponsesPerf import GetBegemotResponsesPerf

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as st_helper
import sandbox.projects.release_machine.components.all as rmc

THRESHOLDS = {
    'WARN': 0.03,
    'CRIT': 0.2
}

class Alerts:
    OK = "**!!(green)OK!!**"
    WARN = "**!!(yellow)WARNING!!**"
    NO_DATA = "**!!(grey)NO DATA!!**"
    CRIT = "**!!(red)CRITICAL!!**"

def startrek_sandbox_task_ref(task_id):
    return "((https://sandbox.yandex-team.ru/task/{task_id}/view {task_id}))".format(task_id=task_id)

class BegemotPerfTestsAccumulation(sdk2.Task):
    class Parameters(sdk2.Parameters):
        new_tasks = sdk2.parameters.List(
            'List of GET_BEGEMOT_RESPONSES_PERF task numbers to compare with previous tasks',
            required=False
        )

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    def find_old_task(self, shard_tag, branch_tag):
        logging.debug("Try to find an old task for tags {}, {}".format(shard_tag, branch_tag))
        for task in sdk2.Task.find(task_type=GetBegemotResponsesPerf, tags=[shard_tag], hidden=True, status=ctt.Status.SUCCESS).limit(99):
            if branch_tag in self.server.task[task.id].read()["tags"]:
                logging.debug("found task {}".format(task.id))
                return task
        return None

    def report_to_release_ticket(self, ticket_info, release_num):
        logging.debug("Info about shards: {}".format(ticket_info))
        shards_info = []
        result = Alerts.OK
        for shard in ticket_info:
            shard_info = ["{}:".format(shard)]
            shard_info.append(startrek_sandbox_task_ref(ticket_info[shard]['new']))
            if 'old' in ticket_info[shard] and ticket_info[shard]['old'] is not None:
                shard_info.append("vs {}".format(startrek_sandbox_task_ref(ticket_info[shard]['old'])))
            if 'diff' in ticket_info[shard]:
                if ticket_info[shard]['diff'] > 0:
                    sign = '+'
                else:
                    sign = ''
                if ticket_info[shard]['diff'] < THRESHOLDS['WARN']:
                    alert = Alerts.OK
                elif ticket_info[shard]['diff'] < THRESHOLDS['CRIT']:
                    alert = Alerts.WARN
                    if result == Alerts.OK:
                        result = Alerts.WARN
                else:
                    alert = Alerts.CRIT
                    result = Alerts.CRIT
                shard_info.append("Time: {}{:.2f}% {}".format(sign, ticket_info[shard]['diff'] * 100, alert))
            else:
                shard_info.append(Alerts.NO_DATA)
                if result == Alerts.OK:
                    result = Alerts.WARN
            shards_info.append(" ".join(shard_info))

        message = "\n".join(
            ["Check begemot perf test: {}".format(startrek_sandbox_task_ref(self.id)),
            "<{{{}".format(result)] +
            shards_info +
            ["}>"]
        )
        logging.debug("Write to release ticket message: {}".format(message))
        st = st_helper.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        c_info = rmc.COMPONENTS["begemot"]()
        st.write_grouped_comment(
            rm_const.TicketGroups.PerfTest,
            "",
            message,
            release_num,
            c_info,
        )

    def on_execute(self):
        if self.Context.new_perf_tasks_ids is NotExists:
            self.Context.new_perf_tasks_ids = self.Parameters.new_tasks
        begemot_executable = sdk2.Resource["BEGEMOT_EXECUTABLE"].find(state='READY', attrs={'released': 'stable'}).first()
        logging.debug("Last released begemot executable: {}".format(begemot_executable.id))
        release_branch = None
        ticket_info = {}
        release_task = sdk2.Task.find(id=begemot_executable.task_id).first()
        try:
            release_branch = release_task.Context.checkout_arcadia_from_url.split('stable-')[1].split('-')[0]
        except Exception as e:
            logging.debug("Task failed: {}".format(e))
            raise TaskFailure("Failed to find release branch")

        for task_id in self.Context.new_perf_tasks_ids:
            new_task_tags = self.server.task[task_id].read()["tags"]
            shard_tag, branch_tag = None, None
            for tag in new_task_tags:
                if tag.split('-')[-1].isdigit():
                    branch_tag = tag
                if "GET_BEGEMOT_RESPONSES_PERF" in tag:
                    shard_tag = tag
            if shard_tag is None or branch_tag is None:
                self.set_info("Failed to parse tags of task {}. Cannot compare perf to previous one".format(task_id))
                continue
            shard = shard_tag.split('-')[-1].split('_')[-1]
            if "APPHOST" in shard_tag:
                shard += " (apphost queries)"

            new_task = sdk2.Task.find(id=task_id).first()
            old_task = self.find_old_task(shard_tag, '-'.join(branch_tag.split('-')[:-1] + [release_branch]))
            if old_task is None:
                continue
            ticket_info[shard] = {'old': old_task.id, 'new': new_task.id}
            if old_task is None:
                self.set_info("{}: failed to find perf computing task for previous release".format(shard))
                continue

            try:
                new_metrics_value = new_task.Context.critical_path_metrics['current_parallel_metric']
                old_metrics_value = old_task.Context.critical_path_metrics['current_parallel_metric']
            except:
                self.set_info("{}: metrics values not found".format(shard))
                continue
            diff = float(new_metrics_value) / old_metrics_value - 1
            if diff > 0:
                result = "Time increased by {}%".format(diff * 100)
            else:
                result = "Time decreased by {}%".format(abs(diff) * 100)
            self.set_info("\n".join([
                "{}:".format(shard),
                "Previous release parallelism value: {}".format(old_metrics_value),
                "Current release parallelism value: {}".format(new_metrics_value),
                result
            ]))
            ticket_info[shard]['diff'] = diff
        try:
            current_branch = self.Context.testenv_database.split('-')[-1]
            self.report_to_release_ticket(ticket_info, int(current_branch))
        except Exception as e:
            logging.debug("Failed to write info to release ticket. Exception: {}".format(e))
