# -*- coding: utf-8 -*-
import logging
import re

from sandbox import sdk2
from sandbox.projects.common.betas.yappy_api import YappyApi
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.websearch.begemot.resources import BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT, BEGEMOT_SPELLCHECKER_HUSQVARNA_API_LOG


YAPPY_API_URL = "https://yappy.z.yandex-team.ru/"
BEGEMOT_SPELLCHECKER_BETA_TYPE = "begemot-spellchecker"
SAVE_REPORT_PATH = "report.txt"
FRAMEWORK_API_EXECUTABLE_RESOURCE = 806750316


class BegemotSpellcheckerHusqvarnaQuality(sdk2.Task):
    '''
    Run test on huqvarna.yandex.ru.
    '''
    __logger = logging.getLogger('TASK_LOGGER')
    __logger.setLevel(logging.DEBUG)

    class Parameters(sdk2.Parameters):
        kill_timeout = 12 * 60 * 60

        framework_api_execatable = sdk2.parameters.Resource(
            'Executable with spellchecker test_framework api.',
            default=FRAMEWORK_API_EXECUTABLE_RESOURCE,
            name="framework_api_executable"
        )
        begemot_spellchecker_yappy_beta = sdk2.parameters.String(
            'Yappy beta name with begemot spellchecker.',
            name="begemot_spellchecker_yappy_beta"
        )
        husqvarna_test_group = sdk2.parameters.String(
            'Husqvarna test group to run.',
            name="husqvarna_test_group"
        )
        husqvarna_user = sdk2.parameters.String(
            'Husqvarna user.',
            name="husqvarna_user"
        )

    class Requirements(sdk2.Requirements):
        cores = 2
        disk_space = 1024

    def get_branch_n(self, beta_name):
        pattern = re.compile(r"begemot-spellchecker-(\d+)")
        logging.debug("Beta name: %s", beta_name)
        branch_n =  int(re.search(pattern, beta_name).group(1))
        logging.info("Parse branch number as %s, from beta name \"%s\"", branch_n, beta_name)
        return branch_n

    def on_execute(self):
        framework_api_executable_path = str(sdk2.ResourceData(self.Parameters.framework_api_execatable).path)
        beta_name = self.Parameters.begemot_spellchecker_yappy_beta
        husqvarna_test_group = self.Parameters.husqvarna_test_group
        husqvarna_user = self.Parameters.husqvarna_user

        api = YappyApi(YAPPY_API_URL, ssl_verify=False)
        if not api.beta_exists(beta_name):
            raise TaskFailure("Beta with name {beta_name} does not exist.".format(beta_name=beta_name))

        instances = api.get_instances(beta_name, BEGEMOT_SPELLCHECKER_BETA_TYPE)
        host, port = instances[0].split(":")

        adapter_addr = "{host}:{port}".format(host=host, port=int(port))
        test_run_label = "begemot-beta-{beta_name}-test".format(beta_name=beta_name)
        framework_api_commandline = [framework_api_executable_path,
                                    husqvarna_test_group,
                                    "--server", adapter_addr,
                                     "--porno",
                                     "--label", test_run_label,
                                     "--save-report", SAVE_REPORT_PATH,
                                     "-u", husqvarna_user]

        api_fout = open("api.out", "w")
        api_ferr = open("api.err", "w")
        api_subprocess = subprocess.Popen(framework_api_commandline, stdout=api_fout, stderr=api_ferr)
        api_subprocess.wait()

        BEGEMOT_SPELLCHECKER_HUSQVARNA_API_LOG(
            self,
            "Husqvarna api log for test of {beta_name}.".format(beta_name=beta_name),
            "api.err"
        )

        if api_subprocess.returncode:
            err = open("api.err", "r")
            raise TaskFailure("Framework api failed with code={code}.\nstderr: {stderr}".format(
                code=api_subprocess.returncode,
                stderr=err.read()))

        BEGEMOT_SPELLCHECKER_HUSQVARNA_TESTRUN_REPORT(
            self,
            "Husqvarna test report for {beta_name}.".format(beta_name=beta_name),
            SAVE_REPORT_PATH,
            ttl="inf",
            begemot_release=self.get_branch_n(beta_name)
        )
