import datetime as dt
import json
import logging
import os
import re
import requests
import subprocess
import time

from sandbox.projects.websearch.begemot.tasks.BegemotYT.BegemotStabilityCheck import get_begemot_release_tag
from sandbox.projects.websearch.begemot import resources as br

from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.notification import Transport
from sandbox.common.types.resource import State
import sandbox.common.types.task as ctt
from sandbox.projects.common.betas.yappy_api import YappyApi
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.sandboxsdk.svn import Arcadia, Svn
from sandbox import sdk2
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as st_helper
import sandbox.projects.release_machine.components.all as rmc
from sandbox.projects.tank.load_resources.resources import AMMO_FILE

NANNY_URL = 'http://nanny.yandex-team.ru'
YAPPY_API_URL = 'https://yappy.z.yandex-team.ru/'
DEFAULT_TANK = 'nanny:production_yandex_tank'
YASM_URL = 'https://yasm.yandex-team.ru/template/panel/{panel_name}/hosts={hosts}/?from={fr}&to={to}'
MATCH_DC = re.compile('(?P<dc>vla|man|sas)')


class BegemotTankLoadTest(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        tank_names = sdk2.parameters.List(
            "Tanks",
            required=True,
            default=DEFAULT_TANK,
        )
        beta = sdk2.parameters.String(
            "Yappy beta",
            required=False
        )
        type_or_quota = sdk2.parameters.String(
            "Component type or quota name",
            required=False
        )
        service = sdk2.parameters.String(
            "Nanny service to test",
            required=False,
            description='Required if the previous two parameters are not set'
        )
        load_plan = sdk2.parameters.String(
            "Tank load plan",
            required=True,
            description='e.g. line(1,10,30s) const(10,10m) step(10, 20, 5, 10s)'
        )
        queries = sdk2.parameters.Resource(
            "Queries for service in ammo format",
            required=False,
        )
        ah_queries = sdk2.parameters.Resource(
            "Apphost queries for service, see description",
            required=False,
            description='Queries will be converted into ammo format via converter. If you already have ammo file or use get requests, pass them through previous field',
        )
        queries_prefix = sdk2.parameters.String(
            "Prefix to write before each cgi query",
            required=False,
        )
        retries = sdk2.parameters.Integer(
            "Number of retries",
            required=True,
            default=1,
        )
        rps_to_survive = sdk2.parameters.Integer(
            "Minimal RPS to survive",
            default=0,
        )
        time_threshold = sdk2.parameters.Integer(
            "Max value for 0.95 response quantile, ms",
            default=50,
            description="If 0.95-quantile value is greater for X rps, task considers that service cannot survive X rps",
        )
        report_to_release_ticket = sdk2.parameters.Bool(
            "Report to release ticket",
            default=False
        )
        with report_to_release_ticket.value[True]:
            rm_component = sdk2.parameters.String(
                "Release machine component"
            )
            benchmark_task = sdk2.parameters.Integer(
                "Benchmark task id",
                default=0,
                description="If 0, task will find shooting for last released branch",
            )
            default_release_ticket = sdk2.parameters.String(
                "Default ticket for report",
                required=False,
            )
            shard = sdk2.parameters.String(
                "Begemot shard",
                required=False
            )
        report_to_mail = sdk2.parameters.Bool(
            "Send error mails",
            required=False,
            default=False,
        )
        with report_to_mail.value[True]:
            mail_recepients = sdk2.parameters.List(
                label="Recepients of error messages",
                value_type=sdk2.parameters.Staff,
                required=True
            )
        make_yasm_panel = sdk2.parameters.Bool(
            "Make yasm panel for host",
            required=False,
            default=False,
        )
        with make_yasm_panel.value[True]:
            with sdk2.parameters.RadioGroup("Panel type") as yasm_panel_type:
                yasm_panel_type.values["megamind"] = yasm_panel_type.Value("Megamind")
                yasm_panel_type.values["custom"] = yasm_panel_type.Value("Custom")

            with yasm_panel_type.value["custom"]:
                custom_panel_name = sdk2.parameters.String(
                    "Custom panel name with template args: hosts, from, to",
                    required=True
                )
        ticket = sdk2.parameters.String(
            "Startrek ticket connected to this shooting",
            required=True
        )
        nanny_token = sdk2.parameters.String(
            "Nanny token vault name for the owner",
            required=False,
            default='Begemot Nanny token'
        )
        hostname = sdk2.parameters.String(
            "Target hostname for tank",
            required=False
        )
        port = sdk2.parameters.String(
            "Hostname for tank",
            required=False
        )
        converter = sdk2.parameters.Resource(
            "Apphost converter executable",
            required=False,
            resource_type=br.BEGEMOT_AH_CONVERTER
        )
        force_same_dc = sdk2.parameters.Bool(
            "Force same dc for tank",
            required=True,
            default=False,
        )

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Context(sdk2.Context):
        restart_pool = 3
        did_full_retry = False

    def find_in_sandbox(self, label):
        try:
            res = sdk2.Resource.find(
                resource_type=AMMO_FILE,
                attrs={'ammo_label': label},
                status=State.READY,
            ).first()
            if res.state in [State.BROKEN, State.DELETED]:
                return None
            self.Context.ammo = res.id
            logging.debug("Found ammo file with id {}, no need to convert input resource".format(res.id))
            return res.id
        except:
            return None

    def convert_ah_queries(self):
        if not self.Parameters.ah_queries:
            raise TaskFailure("Task needs ammo queries or apphost queries")
        input_res = self.Parameters.ah_queries
        label = "from_res_{}".format(input_res.id)
        found = self.find_in_sandbox(label)
        if found is not None:
            return
        if not self.Parameters.converter:
            converter = sdk2.Resource["BEGEMOT_AH_CONVERTER"].find(state='READY').first()
        else:
            converter = self.Parameters.converter
        input_res = self.Parameters.ah_queries
        cv_path = str(sdk2.ResourceData(converter).path)
        input_path = str(sdk2.ResourceData(input_res).path)
        cv_stderr = self.log_path() / 'converter_stderr.txt'
        cv_stdout = '{}.ammo'.format(input_res.id)
        cmd = ' '.join([cv_path, input_path, "-o", "service_request", "-a", "-m"])
        with cv_stderr.open("w") as err:
            with open(cv_stdout, "w") as out:
                p = subprocess.Popen(cmd, stderr=err, stdout=out, shell=True)
                p.wait()
                if p.returncode:
                    raise Exception('Apphost converter exited with code {code}'.format(code=p.returncode))
        output = AMMO_FILE(
            self,
            "Apphost queries #{} converted by task {}".format(input_res.id, self.id),
            cv_stdout,
            ammo_label=label,
        )
        sdk2.ResourceData(output).ready()
        self.Context.ammo = output.id

    def convert_cgi_queries(self):
        input_res = self.Parameters.queries
        prefix = self.Parameters.queries_prefix
        label = "from_res_{}_prefix_{}".format(input_res.id, prefix)
        found = self.find_in_sandbox(label)
        if found is not None:
            return
        input_path = str(sdk2.ResourceData(input_res).path)
        output_path = '{}.ammo'.format(input_res.id)
        with open(input_path, "r") as fin:
            with open(output_path, "w") as fout:
                for line in fin.readlines():
                    fout.write(self.Parameters.queries_prefix + line)
        output = AMMO_FILE(
            self,
            "Cgi queries #{} with prefix {}".format(input_res.id, prefix),
            output_path,
            ammo_label=label,
        )
        sdk2.ResourceData(output).ready()
        self.Context.ammo = output.id

    def get_nanny_service(self):
        if self.Parameters.hostname and self.Parameters.port:
            return None
        if self.Parameters.service:
            return self.Parameters.service

        beta = self.Parameters.beta
        type_or_quota = self.Parameters.type_or_quota
        if not beta or not type_or_quota:
            raise TaskFailure("Set either nanny service or (beta, type_or_quota)")

        api = YappyApi(YAPPY_API_URL, ssl_verify=False)
        if not api.beta_exists(beta):
            raise TaskFailure("Beta with name {} does not exist.".format(beta))

        beta_info = api._make_req_post(
            '/api/yappy.services.Model/retrieveBeta',
            data=json.dumps({'name': beta, 'withHistory': False})
        )

        logging.debug("Trying to find  component with quota or type name: {}".format(self.Parameters.type_or_quota))
        for comp in beta_info['components']:
            quota = comp.get('slot', {}).get('quotaName')
            comp_type = comp.get('type', {}).get('name')
            logging.debug("Found component with type {} and quota {}".format(comp_type, quota))
            if quota == type_or_quota or comp_type == type_or_quota:
                service = comp.get('slot', {}).get('id')
                logging.debug("Task will shoot to service {}".format(service))
                return service

        raise TaskFailure("Task failed to find a component with required type or quota name")

    def build_shooting_config(self, nanny_service):
        arcadia_path = Arcadia.checkout('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia', 'arcadia', depth=Svn.Depth.IMMEDIATES)
        config_template_path = os.path.join(arcadia_path, 'sandbox/projects/websearch/begemot/tasks/BegemotTankLoadTest/shooting_config.txt')
        Arcadia.update(config_template_path, depth=Svn.Depth.IMMEDIATES, parents=True)
        with open(config_template_path, 'r') as f:
            hostname, port = None, None
            if self.Parameters.hostname and self.Parameters.port:
                hostname, port = self.Parameters.hostname, self.Parameters.port
            else:
                nanny = NannyClient(NANNY_URL, self.Parameters.nanny_token)
                instance_info = nanny.get_service_current_instances(nanny_service)['result'][0]
                hostname, port = instance_info['container_hostname'], instance_info['port']
            config = f.read()
            config = config.replace("<ADDRESS>", "{}:{}".format(hostname, port))
            config = config.replace("<AMMOFILE>", "https://proxy.sandbox.yandex-team.ru/{}".format(self.Context.ammo))
            config = config.replace("<LOAD_PLAN>", self.Parameters.load_plan)
            config = config.replace("<DESCRIPTION>", self.Parameters.description)
            config = config.replace("<AUTHOR>", self.author)
            config = config.replace("<TICKET>", self.Parameters.ticket)
            logging.debug("Tank config is:\n{}".format(config))
            return config, port, hostname

    def get_request(self, api):
        answer = requests.get(api, timeout=10)
        try:
            return json.loads(answer.text)
        except:
            return {}

    def report_to_ticket(self):
        branch_tag, job_tag = None, None
        tags = self.server.task[self.id].read()["tags"]
        for tag in tags:
            if 'TESTENV-JOB-' in tag:
                job_tag = tag
            try:
                branch = int(tag.split('-')[-1])
                branch_tag = tag
            except:
                pass

        if self.Parameters.shard:
            shard = self.Parameters.shard
        else:
            if job_tag is None:
                logging.debug("Task failed to find its testenv job tag")
                return
            shard = job_tag.split('_')[-2]

        logging.debug("Job tag: {}, shard name: {}".format(job_tag, shard))

        old_task = None
        branch_num = None

        if not self.Parameters.default_release_ticket:  # branch_num is necessary only for branch release ticket
            testenv_db = self.Context.testenv_database.split('-')[-1]
            try:
                branch_num = int(testenv_db)
            except:
                logging.debug("Context field testenv_database should have branch number in its end")
                return


        if self.Parameters.benchmark_task > 0:
            old_task = sdk2.Task.find(id=self.Parameters.benchmark_task).first()
        else:
            if branch_tag is None:
                logging.debug("Task failed to find its testenv branch tag")
                return
            if job_tag is None:
                logging.debug("Task failed to find its testenv job tag")
                return

            last_tag = get_begemot_release_tag()
            
            logging.debug("Task will try to find an old task for tags {}, {}".format(branch_tag, job_tag))
            for task in sdk2.Task.find(task_type=BegemotTankLoadTest, tags=[job_tag], hidden=True, status=ctt.Status.SUCCESS).limit(10):
                if branch_tag.replace(str(branch_num), str(last_tag)) in self.server.task[task.id].read()["tags"]:
                    logging.debug("found task {}".format(task.id))
                    old_task = task
                    break

        if old_task is None:
            logging.debug("Task failed to find shooting task for last released branch")
            message = "\n".join([
                "<{{{}: {} rps".format(shard, int(self.Context.test_result)),
                ", ".join(["(({}))".format(link) for link in self.Context.lunapark_links]),
                "Task did not found a shooting task for released branch",
                "}>"
            ])
        else:
            prev_beta, cur_beta = old_task.Parameters.beta, self.Parameters.beta
            if prev_beta == cur_beta:
                prev_beta = 'production copy'
            logging.debug("Will compare {} vs. {}".format(cur_beta, prev_beta))
            prev_result, cur_result = int(old_task.Context.test_result), int(self.Context.test_result)
            alert = "**!!(green)OK!!**"
            if cur_result < self.Parameters.rps_to_survive or prev_result * 0.9 > cur_result:
                alert = "**!!(yellow)WARNING!!**"
            if prev_result * 0.75 > cur_result:
                alert = "**!!(red)CRITICAL!!**"
            add_info = " (Survived RPS can be too small)" if cur_result < self.Parameters.rps_to_survive else ""

            if cur_result >= prev_result:
                diff = "+{:.2f}%".format((float(cur_result) / prev_result - 1) * 100)
            else:
                diff = "-{:.2f}%".format((1 - float(cur_result) / prev_result) * 100)
            message = "\n".join([
                "<{{{} {} ({} vs. {}): {}{}".format(alert, shard, cur_beta, prev_beta, diff, add_info),
                "Current branch task: ((https://sandbox.yandex-team.ru/task/{id}/view {id}))".format(id=self.id),
                "Released branch task: ((https://sandbox.yandex-team.ru/task/{id}/view {id}))".format(id=old_task.id),
                "Lunapark shooting jobs:",
                ", ".join(["(({}))".format(link) for link in self.Context.lunapark_links]),
                "Max rps before: {}".format(prev_result),
                "Max rps after: {}".format(cur_result),
                "}>",
            ])

        logging.debug("Message for startrek: {}".format(message))    
        st = st_helper.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        c_info = rmc.COMPONENTS[self.Parameters.rm_component]()
        group_name = "====Tank perf test"
        title = ""
        if self.Parameters.default_release_ticket:
            st.write_grouped_comment_in_issue(
                self.Parameters.default_release_ticket,
                group_name,
                title,
                message,
                None,
                False,
            )
        else:
            st.write_grouped_comment(
                group_name,
                title,
                message,
                branch_num,
                c_info,
            )
        
        if self.Parameters.report_to_mail and alert != "**!!(green)OK!!**":
            logging.debug("Send emails to {}".format(self.Parameters.mail_recepients))
            self.server.notification(
                subject="Begemot tank load test has failed",
                body=message,
                recipients=self.Parameters.mail_recepients,
                transport=Transport.EMAIL,
                urgent=False
            )

    def get_max_rps(self, job_id, start_ts, finish_ts):
        def dt_from_str(s):
            return dt.datetime.strptime(s, '%Y-%m-%d %H:%M:%S')

        def int_from_dt(d):
            return int(d.strftime("%Y%m%d%H%M%S"))

        start_ts = dt_from_str(start_ts)
        finish_ts = dt_from_str(finish_ts)
        rps_stat_api = "https://lunapark.yandex-team.ru/api/job/{}/data/req_cases.json".format(job_id)
        rps_stats = self.get_request(rps_stat_api)
        rps_by_time = {int_from_dt(dt_from_str(item['dt'])): item['cnt'] for item in rps_stats}
        quant_by_rps = {}
        ok_by_rps = {}

        cur_dt = start_ts
        while cur_dt <= finish_ts:
            ts = int_from_dt(cur_dt)
            perc_api = "https://lunapark.yandex-team.ru/api/job/{j}/dist/percentiles.json?percents=95&from={ts}&to={ts}".format(j=job_id, ts=ts)
            http_api = "https://lunapark.yandex-team.ru/api/job/{j}/dist/http.json?from={ts}&to={ts}".format(j=job_id, ts=ts)
            perc_json = self.get_request(perc_api)
            http_json = self.get_request(http_api)
            if not perc_json or not http_json:
                self.set_info("WARNING: task failed to find stats for timestamp {}".format(ts))
            else:
                quant_by_rps[rps_by_time[ts]] = perc_json[0]['ms']
                for item in http_json:
                    if item['http'] == 200:
                        ok_by_rps[rps_by_time[ts]] = item['percent']
                        break
                else:
                    ok_by_rps[rps_by_time[ts]] = 0.0
            cur_dt += dt.timedelta(seconds=1)

        last_good, last_ok = -1, -1
        for rps in quant_by_rps:
            if quant_by_rps[rps] <= self.Parameters.time_threshold and last_good < rps:
                last_good = rps
            if ok_by_rps[rps] == 100.0 and last_ok < rps:
                last_ok = rps
        return max(1, min(last_good, last_ok))

    def check_subtasks(self):
        if self.Context.shooting_task_ids is NotExists:
            return None
        for task_id, _ in self.Context.shooting_task_ids:
            task = sdk2.Task.find(id=task_id).first()
            if task.status == ctt.Status.EXCEPTION or task.status == ctt.Status.TIMEOUT:
                return task
        return None

    def do_full_retry(self):
        self.Context.did_full_retry = True
        self.Context.restart_pool = 3
        self.Context.shooting_task_ids = []
        self.Context.lunapark_links = []
        self.Context.shooting_results = []
        raise sdk2.WaitTime(60)

    def restart_broken_subtask(self, task):
        if self.Context.restart_pool <= 0:
            if not self.Context.did_full_retry:
                self.do_full_retry()
            else:
                raise TaskFailure("Child task {} failed. No restarts left".format(task.id))
        self.set_info("Shooting #{} failed. Trying to restart".format(task.id))
        self.Context.restart_pool -= 1
        task.enqueue()
        raise sdk2.WaitTask(task.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def on_execute(self):
        if self.Context.ammo is NotExists:
            if not self.Parameters.queries:
                self.convert_ah_queries()
            elif self.Parameters.queries_prefix:
                self.convert_cgi_queries()
            else:
                self.Context.ammo = self.Parameters.queries.id

        broken_task = self.check_subtasks()
        if broken_task is not None:
            self.restart_broken_subtask(broken_task)

        if self.Context.shooting_task_ids is NotExists or len(self.Context.shooting_task_ids) < self.Parameters.retries:
            nanny_service = self.get_nanny_service()
            config, port, hostname = self.build_shooting_config(nanny_service)

            if DEFAULT_TANK in self.Parameters.tank_names:
                shooting_task_id = ShootViaTankapi(
                    self,
                    nanny_service=nanny_service,
                    override_nanny_port=port,
                    use_public_tanks=True,
                    config_source='file',
                    config_content=config,
                    kill_timeout=10800,
                ).enqueue().id
            else:
                tanks = self.Parameters.tank_names
                if self.Parameters.force_same_dc:
                    dc = MATCH_DC.search(hostname)
                    if not dc or 'dc' not in dc.groupdict():
                        raise TaskFailure("Could not match dc for host {}".format(hostname))
                    dc = dc.groupdict()['dc']

                    tanks = list(filter(lambda x : MATCH_DC.search(x).groupdict()['dc'] == dc, tanks))
                    if not tanks:
                        raise TaskFailure("Tanks were not found in the {dc}}".format(dc))



                shooting_task_id = ShootViaTankapi(
                    self,
                    use_public_tanks=False,
                    tanks=tanks,
                    config_source='file',
                    config_content=config,
                    kill_timeout=10800,
                ).enqueue().id
            if self.Context.shooting_task_ids is NotExists:
                self.Context.shooting_task_ids = [(shooting_task_id, hostname)]
            else:
                self.Context.shooting_task_ids.append((shooting_task_id, hostname))
            raise sdk2.WaitTask(shooting_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)
        else:
            self.Context.lunapark_links = []
            self.Context.shooting_results = []
            for task_id, hostname in self.Context.shooting_task_ids:
                task = sdk2.Task.find(id=task_id, children=True).first()
                if task.status != ctt.Status.SUCCESS:
                    if not self.Context.did_full_retry:
                        self.do_full_retry()
                    else:
                        raise Exception("Shooting task {} failed".format(task_id))
                job_id = task.Parameters.lunapark_job_id
                lunapark_link = task.Parameters.lunapark_link
                self.Context.lunapark_links.append(lunapark_link)

                api = "https://lunapark.yandex-team.ru/api/job/{}/aggregates.json".format(job_id)
                stat = self.get_request(api)[0]
                time_from = stat['trail_start']
                time_to = stat['trail_stop']
                report = [
                    "Attempt {}".format(len(self.Context.lunapark_links)),
                    "Start time: {}".format(time_from),
                    "End time:   {}".format(time_to),
                    "Hostname:   {}".format(hostname),
                ]
                for val in [50, 90, 95, 99]:
                    report.append("Quantile {}: {} ms".format(val, stat['q{}'.format(val)]))
                report.append("HTTP 200: {}%".format(stat['http_200_percent']))

                max_rps = self.get_max_rps(job_id, time_from, time_to)
                self.Context.shooting_results.append(max_rps)

                report.append("Estimated max rps for service: {}".format(max_rps))
                report.append('Lunapark link: <a href="{link}">{link}</a>'.format(link=lunapark_link))

                if self.Parameters.make_yasm_panel and self.Parameters.yasm_panel_type:
                    if self.Parameters.yasm_panel_type == "megamind":
                        panel_name = "megabegemot-host-health"
                    if self.Parameters.yasm_panel_type == "custom":
                        panel_name = self.Parameters.custom_panel_name

                    get_ts = lambda tm : int(time.mktime(dt.datetime.strptime(tm, '%Y-%m-%d %H:%M:%S').timetuple())) * 1000
                    
                    panel = YASM_URL.format(panel_name=panel_name, hosts=hostname, fr=get_ts(time_from), to=get_ts(time_to))
                    report.append("Yasm host link: <a href=\"{link}\">link</a>".format(link=panel))
                self.set_info('<br>'.join(report), do_escape=False)

            test_result = sum(self.Context.shooting_results) / self.Parameters.retries
            self.Context.test_result = test_result
            try:
                if self.Parameters.report_to_release_ticket:
                    self.report_to_ticket()
            except Exception as e:
                self.set_info("WARNING: task failed to report to startrek ticket. Exception text: {}".format(e))
