#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import json
import sys
import subprocess
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CommonYtParameters, utc_from_now
from sandbox.common.errors import TaskFailure


class CheckBegemotResponsesYT(sdk2.Task):
    class Parameters(sdk2.Parameters):
        begemot_answers = sdk2.parameters.String('Begemot answers', required=True)
        output_path = CommonYtParameters.output_path()
        yt_proxy = CommonYtParameters.yt_proxy()
        yt_pool = CommonYtParameters.yt_pool()
        yt_token_vault_owner = CommonYtParameters.yt_token_vault_owner()
        yt_token_vault_name = CommonYtParameters.yt_token_vault_name()
        results_store_time = sdk2.parameters.Integer('Days to store results', default=3)

    class Requirements(sdk2.Requirements):
        disk_space = 20 * 1024
        ram = 1024
        environments = [PipEnvironment('yandex-yt', version='0.8.49')]

    def on_execute(self):
        import yt.wrapper as yt
        checker = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'checker.py')
        token = sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)
        env = os.environ.copy()
        env['YT_TOKEN'] = token
        args = [
            sys.executable, checker,
            '--answers', self.Parameters.begemot_answers,
            '--output_path', self.Parameters.output_path,
            '--yt_proxy', self.Parameters.yt_proxy,
            '--yt_pool', self.Parameters.yt_pool,
        ]

        yt_client = yt.YtClient(self.Parameters.yt_proxy, token)
        yt_client.create(
            'map_node', self.Parameters.output_path, recursive=True, force=True,
            attributes={'expiration_time': utc_from_now(self.Parameters.results_store_time)},
        )
        with sdk2.helpers.ProcessLog(self, logger='checker') as pl:
            result = subprocess.check_output(args, stderr=pl.stderr, env=env)
            logging.debug('Got result from checker script: {}'.format(result))

        result_json = json.loads(result)
        fail_task = False
        for label, info in result_json.items():
            if info['exceptions'] > 0:
                self.set_info('Check "{}" finished with some exceptions. Look for examples in checker log'.format(label))

            if info['passed'] < ((info['passed'] + info['failed']) * info['required']):
                fail_task = True
                self.set_info('Too much fails for check "{}". {} of {} answers failed check'.format(key, info['failed'], info['passed'] + info['failed']))

        if fail_task:
            raise TaskFailure('Some checks failed')
