#!/usr/bin/python
# -*- coding: utf-8 -*-

import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CommonYtParameters


class CoredumpBacktrace(sdk2.Task):
    class Parameters(sdk2.Parameters):
        binary = sdk2.parameters.Resource('Binary', required=True)
        coredump_table = sdk2.parameters.String('Coredump Table path', required=True)
        job_id = sdk2.parameters.String('Job id', required=True)
        coredump_size = sdk2.parameters.Integer('Coredump size in bytes', required=True)
        yt_proxy = CommonYtParameters.yt_proxy()
        yt_token_vault_name = CommonYtParameters.yt_token_vault_name()
        yt_token_vault_owner = CommonYtParameters.yt_token_vault_owner()
        gdb_output = sdk2.parameters.ParentResource('Gdb output', required=True)
        ya_token = sdk2.parameters.String('Ya token', required=True, default='Begemot Ya Token')

    class Requirements(sdk2.Requirements):
        disk_space = 1000
        ram = 8000

    def on_enqueue(self):
        self.Requirements.disk_space += (self.Parameters.coredump_size + self.Parameters.binary.size) / (1 << 20)

    def on_execute(self):
        token = sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)
        ya_token = sdk2.Vault.data(self.Parameters.ya_token)
        env = os.environ.copy()
        env['YT_TOKEN'] = token
        env['YA_TOKEN'] = ya_token

        # download_coredump args
        args = [
            sdk2.ResourceData(
                sdk2.Resource.find(type='YT_DOWNLOAD_COREDUMP').order(-sdk2.Resource.id).first()
            ).path.as_posix(),
            '-p', self.Parameters.yt_proxy,
            '-t', self.Parameters.coredump_table,
            '-j', self.Parameters.job_id,
            '-c', '0',
        ]

        with sdk2.helpers.ProcessLog(self, logger='download_coredump') as pl:
            pl.logger.propagate = 1
            subprocess.check_call(args, stdout=pl.stdout, stderr=subprocess.STDOUT, env=env)

        Arcadia.export(url='arcadia:/arc/trunk/arcadia/ya', path='./ya')

        args = [
            './ya', 'tool', 'gdb',
            str(sdk2.ResourceData(self.Parameters.binary).path),
            'core.0',
            '--ex', 'bt',
            '--ex', 'thread apply all bt',
            '--batch',
        ]

        gdb_output = self.Parameters.gdb_output
        with sdk2.helpers.ProcessLog(self, logger='gdb') as pl, gdb_output.path.open('w') as out:
            pl.logger.propagate = 1
            subprocess.check_call(args, stdout=out, stderr=subprocess.STDOUT, env=env)
        self.set_info('Backtrace:\n\n%s' % gdb_output.path.read_text())
