#!/usr/bin/python
# -*- coding: utf-8 -*-

import datetime
import logging
import os
import time


from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.errors import ResourceNotFound, TaskFailure
from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task as sandbox_task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects.websearch.begemot.resources import BEGEMOT_CYPRESS_SHARD


class CypressCache(SandboxStringParameter):
    name = 'CypressCache'
    description = 'Begemot data path in Cypress'
    default_value = '//home/begemot/begemot-tests/shards_cache'
    required = False


class YtProxy(SandboxStringParameter):
    name = 'YtProxy'
    description = 'YT_PROXY'
    default_value = 'hahn'


class CypressShardUpdater(sandbox_task.SandboxTask):
    input_parameters = [
        CypressCache,
        YtProxy,
    ]
    __logger = logging.getLogger('SHARD_UPDATER')
    __logger.setLevel(logging.DEBUG)

    def filter_rules(self, filename, rules_filter):
        result = []
        with open(filename, "r") as f:
            for line in f.readlines():
                line = line.strip("\n")
                if os.path.basename(line) in rules_filter:
                    logging.debug("OK for path {}".format(line))
                    result.append(line)
                else:
                    logging.debug("SKIP path {}".format(line))

        with open(filename, "w") as f:
            f.write('\n'.join(result) + '\n')

    # Updates shard in Cypress and creates resource with Cypress paths
    # Also saves the same file into parent tasks Cypress shard resource, if any
    def update_cypress_shard(self, local_shard_path, shard_name, is_fresh=False, existing_paths=(), rules_filter=None, retries=1, updater_id=None, test_updater=False, fake_update=False):
        updater_success = False
        result_file = '{}.txt'.format(shard_name)
        while not updater_success and retries > 0 and not fake_update:
            self.run_shard_updater(updater_id, result_file, local_shard_path, shard_name, is_fresh, ttl_days=30)
            updater_success = os.path.isfile(result_file)
            retries -= 1
            if not updater_success and retries > 0:
                time.sleep(15)

        if not updater_success and test_updater:
            raise TaskFailure("Shard updater failed")

        resource = self.create_resource(
            shard_name, result_file, BEGEMOT_CYPRESS_SHARD,
            attributes={'shard_name': shard_name, 'is_fresh': is_fresh},
        )
        channel.sandbox.set_resource_attribute(resource.id, 'is_broken', not updater_success)
        if not updater_success:
            self.__logger.error('Shard updater failed: %s not found, creating empty file', result_file)
            open(result_file, 'a').close()

        with open(result_file, 'a') as file:
            for path in existing_paths:
                file.write(path + '\n')

        if rules_filter is not None:
            logging.debug("Filtering rules for shard {}".format(shard_name))
            self.filter_rules(result_file, rules_filter)

        parent_resource = self.get_parent_resource(shard_name, is_fresh)
        if parent_resource:
            self.save_parent_task_resource(result_file, parent_resource)
            channel.sandbox.set_resource_attribute(parent_resource.id, 'is_broken', not updater_success)

        return result_file

    def get_parent_resource(self, shard_name, is_fresh):
        if self.parent_id is None:
            self.__logger.info('Parent task not found')
            return None
        resources = channel.sandbox.list_resources(
            BEGEMOT_CYPRESS_SHARD, task_id=self.parent_id, status=State.NOT_READY,
            all_attrs={'shard_name': shard_name, 'is_fresh': is_fresh},
        )
        if resources:
            self.__logger.info('Found parent resource, id: %d', resources[0].id)
            return resources[0]
        else:
            self.__logger.info('Parent resource not found')

    def get_cypress_cache_config(self):
        if not self.ctx.get(CypressCache.name):
            self.__logger.info('Updating shard in Cypress is disabled')
            return None

        try:
            token = self.get_vault_data('SEARCH-RELEASERS', 'yt_token_for_testenv')
        except Exception as e:
            raise RuntimeError('Failed to get yt token: {}'.format(e.message))

        return {
            'path': self.ctx.get(CypressCache.name),
            'proxy': self.ctx.get(YtProxy.name),
            'token': token,
        }

    def run_shard_updater(self, updater_id, result_file, local_shard_path, shard_name, is_fresh, ttl_days):
        config = self.get_cypress_cache_config()
        if config is None:
            return
        env = os.environ.copy()
        cypress_cache = config['path']
        env['YT_TOKEN'] = config['token']
        env['YT_PROXY'] = config['proxy']

        self.__logger.info(
            'Updating shard, cypress_cache: %s, shard_name: %s, is_fresh: %s, %ttl_days: %d',
            cypress_cache, shard_name, is_fresh, ttl_days,
        )
        if updater_id is None:
            shard_updater_resource = apihelpers.get_last_released_resource('BEGEMOT_SHARD_UPDATER')
            if shard_updater_resource is None:
                raise ResourceNotFound('BEGEMOT_SHARD_UPDATER resource not found')
            updater_id = shard_updater_resource.id

        self.__logger.info("Using shard updater #{}".format(updater_id))

        shard_updater = self.sync_resource(updater_id)
        expiration_date = datetime.datetime.utcnow() + datetime.timedelta(days=ttl_days)
        args = [
            shard_updater,
            '--data', local_shard_path,
            '--result-paths-file', result_file,
            '--cache-path', cypress_cache,
            '--ttl', expiration_date.isoformat(),
            '--jobs', '20',
            '--debug',
        ]
        try:
            process.run_process(args, wait=True, log_prefix='shard_updater', environment=env)
        except BaseException as e:
            self.__logger.error('Shard updater process failed: ' + e.message)

    def on_execute(self):
        raise NotImplementedError('This task is abstract')


class CommonYtParameters(sdk2.Parameters):
    input_table = sdk2.parameters.String('Input table')
    eventlog_table = sdk2.parameters.String('Eventlog table (will be used if input table is empty)')
    output_path = sdk2.parameters.String('Output directory', required=True)
    ignore_existing = sdk2.parameters.Bool('Ignore existing', default=True, description='Rewrite output path if it '
                                                                                        'exists')
    wait_time = sdk2.parameters.Integer('Seconds between operation state checks', default=300)
    yt_proxy = sdk2.parameters.String('YT_PROXY', default='hahn')
    yt_token_vault_owner = sdk2.parameters.String('Vault owner for yt token', default='SEARCH-RELEASERS')
    yt_token_vault_name = sdk2.parameters.String('Vault name for yt token', default='yt_token_for_testenv')
    yt_pool = sdk2.parameters.String('YT_POOL', default='begemot')
    kill_timeout = datetime.timedelta(hours=15).seconds
    fail_on_any_error = True


def utc_from_now(days):
    return (datetime.datetime.utcnow() + datetime.timedelta(days)).strftime('%Y-%m-%d %H:%M:%SZ')